<?php

namespace App\Http\Controllers\LifeAtGainup\Merchandiser;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;

class GUTProductionController extends Controller
{
    public function production_gut_dashboard($portal)
    {
        return view('LifeAtGainup.GUTProduction.production_gut_dashboard', ['portal_type' => $portal]);
    }

    public function production_gar_dashboard_beta()
    {
        return view('LifeAtGainup.Production.prod_gar_static');
    }

    public function gut_history_dashboard(Request $request)
    {
        $portal_type = ucfirst($request->portal);
        $fromdate    = $request->query('fromdate');
        $enddate     = $request->query('enddate');
        $type        = $request->query('type');

        return view('LifeAtGainup.GUTProduction.history_dashboard', compact('portal_type', 'fromdate', 'enddate', 'type'));
    }

    public function gut_cutting_output($portal)
    {
        return view('LifeAtGainup.GUTProduction.Cutting_output_dashboard', ['portal_type' => $portal]);
    }

    public function gut_wip($portal)
    {
        return view('LifeAtGainup.GUTProduction.wip', ['portal_type' => $portal]);
    }

    public function gut_smstock($portal)
    {
        return view('LifeAtGainup.GUTProduction.smstock', ['portal_type' => $portal]);
    }

    public function gut_direct_labour($portal)
    {
        return view('LifeAtGainup.GUTProduction.directlabour', ['portal_type' => $portal]);
    }

    public function gut_linestop_analysis($portal)
    {
        return view('LifeAtGainup.GUTProduction.Linestop_analysis_dashboard', ['portal_type' => $portal]);
    }

    public function gut_issued($portal)
    {
        return view('LifeAtGainup.GUTProduction.issued', ['portal_type' => $portal]);
    }

    public function gutsmstockissued(Request $request)
    {
        try {
            $portaltype = ucfirst($request->portaltype ?? 'Woven');
            $ocnNo      = $request->ocnno   ?? 'GIL\\OCN04460';
            $itemID     = $request->itemid  ?? '913';
            $colorId    = $request->colorid ?? '1297';
            $sizeId     = $request->sizeid  ?? '618';

            $cacheKey = "sm_stock_issued_{$portaltype}_{$ocnNo}_{$itemID}_{$colorId}_{$sizeId}";
            $cacheTTL = env('SM_STOCK_ISSUED_CACHE_TTL', 600);

            $data = Cache::remember($cacheKey, $cacheTTL, function () use ($portaltype, $ocnNo, $itemID, $colorId, $sizeId) {
                switch ($portaltype) {
                    case 'Gloves':
                        return DB::select(
                            'EXEC CHENNAI_ERP.dbo.Line_Issued_Proc_Gloves @Order_No = ?, @Itemid = ?, @Colorid = ?, @Sizeid = ?',
                            [$ocnNo, $itemID, $colorId, $sizeId]
                        );
                    case 'Headwear':
                        return DB::select(
                            'EXEC CHENNAI_ERP.dbo.Line_Issued_Proc_Headwear @Order_No = ?, @Itemid = ?, @Colorid = ?, @Sizeid = ?',
                            [$ocnNo, $itemID, $colorId, $sizeId]
                        );
                    case 'Woven':
                    default:
                        return DB::select(
                            'EXEC CHENNAI_ERP.dbo.Line_Issued_Proc_Woven @Order_No = ?, @Itemid = ?, @Colorid = ?, @Sizeid = ?',
                            [$ocnNo, $itemID, $colorId, $sizeId]
                        );
                }
            });

            $dataArray = collect($data)->map(fn($d) => (array) $d);
            $grouped = $dataArray
                ->groupBy(function ($r) { return (string)($r['Line'] ?? $r['line'] ?? ''); })
                ->map(function ($items, $line) {
                    $sorted = $items->sortByDesc(function ($it) {
                        return Carbon::parse($it['Edate'] ?? $it['Date'] ?? '1970-01-01');
                    })->values();
                    return [
                        'Line'          => (string)$line,
                        'Total_Iss_Qty' => $sorted->sum(fn($x) => (int)($x['Iss_Qty'] ?? 0)),
                        'Entries'       => $sorted->map(function ($x) {
                            return [
                                'Entry_No' => $x['Entry_No'] ?? $x['Entry'] ?? null,
                                'Line'     => (string)($x['Line'] ?? ''),
                                'Iss_Qty'  => (int)($x['Iss_Qty'] ?? 0),
                                'Unit'     => $x['Unit'] ?? null,
                                'Edate'    => $x['Edate'] ?? $x['Date'] ?? null,
                            ];
                        })->all(),
                    ];
                })
                ->values();

            return response()->json([
                'success' => true,
                'portal'  => $portaltype,
                'data'    => $grouped,
            ]);
        } catch (\Throwable $e) {
            Log::error('Error fetching gutsmstockissued data', ['message' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Failed to load stock issued data.'], 500);
        }
    }

    public function gut_dailysnapshot(Request $request)
    {
        $type = $request->type;
        $portaltype = ucfirst($request->portaltype);

        $fdate = $request->fromdate
            ? Carbon::parse($request->fromdate)->format('d-M-Y')
            : Carbon::today()->format('d-M-Y');
        $enddate = $request->enddate
            ? Carbon::parse($request->enddate)->format('d-M-Y')
            : $fdate;

        $cacheKey = "{$portaltype}_daily_snapshot_{$fdate}_{$enddate}_{$type}";

        try {
            $final = Cache::remember($cacheKey, 600, function () use ($fdate, $enddate, $portaltype) {
                switch ($portaltype) {
                    case 'Gloves':
                        $auditProc = 'Chennai_Erp.DBO.Audit_Details_DB_Details_Proc_Gloves';
                        $prodProc  = 'Chennai_Erp.DBO.Production_Main_DB_Proc_Gloves';
                        break;
                    case 'Woven':
                        $auditProc = 'Chennai_Erp.DBO.Audit_Details_DB_Details_Proc_Woven';
                        $prodProc  = 'Chennai_Erp.DBO.Production_Main_DB_Proc_Woven';
                        break;
                    default:
                        throw new \Exception("Invalid portal type: {$portaltype}");
                }

                $resultData = DB::select("exec $auditProc @Fdate = ?, @Tdate = ?", [$fdate, $enddate]);
                $dashboardData = DB::select("exec $prodProc @Fdate = ?, @Tdate = ?", [$fdate, $enddate]);

                $collection = collect(json_decode(json_encode($resultData), true));

                $units = $collection
                    ->groupBy('Unit')
                    ->map(function ($unitGroup, $unitName) {
                        $zones = collect($unitGroup)
                            ->groupBy('Zone')
                            ->map(function ($zoneGroup, $zoneName) {
                                $totalErnMins = $zoneGroup->sum(fn($it) => floatval($it['Ern_Mins'] ?? 0));
                                $totalAvlMins = $zoneGroup->sum(fn($it) => floatval($it['Avl_Mins'] ?? 0));
                                $totalOutput  = $zoneGroup->sum(fn($it) => floatval($it['Output_Qty'] ?? 0));
                                $samPerPiece  = $totalOutput > 0 ? ($totalErnMins / $totalOutput) : 0;
                                $avgZoneEff = $totalAvlMins > 0 ? round(($totalErnMins / $totalAvlMins) * 100, 2) : 0;
                                return [
                                    'Zone' => $zoneName,
                                    'Total_Output_Qty' => $totalOutput,
                                    'Total_Line_WIP'   => $zoneGroup->sum(fn($it) => floatval($it['Line_WIP'] ?? 0)),
                                    'Total_Ern_Mins'   => round($totalErnMins, 2),
                                    'Total_Avl_Mins'   => round($totalAvlMins, 2),
                                    'Total_Ern_Sam'    => round($samPerPiece, 2),
                                    'Total_Tailors'    => $zoneGroup->sum(fn($it) => floatval($it['Tailors'] ?? 0)),
                                    'Total_Dir_Lab'    => $zoneGroup->sum(fn($it) => floatval($it['Dir_Cnt'] ?? 0)),
                                    'Total_Ern_EfficiencyPerc' => $avgZoneEff,
                                    'data' => $zoneGroup->values()->all(),
                                ];
                            })
                            ->values()
                            ->all();

                        $unitTotalErnMins = collect($zones)->sum('Total_Ern_Mins');
                        $unitTotalAvlMins = collect($zones)->sum('Total_Avl_Mins');
                        $unitTotalOutput  = collect($zones)->sum('Total_Output_Qty');
                        $unitEfficiency = $unitTotalAvlMins > 0 ? round(($unitTotalErnMins / $unitTotalAvlMins) * 100, 2) : 0;

                        return [
                            'Unit' => $unitName,
                            'Total_Output_Qty' => $unitTotalOutput,
                            'Total_Line_WIP'   => collect($zones)->sum('Total_Line_WIP'),
                            'Total_Ern_Mins'   => round($unitTotalErnMins, 2),
                            'Total_Avl_Mins'   => round($unitTotalAvlMins, 2),
                            'Total_Ern_Sam'    => collect($zones)->avg('Total_Ern_Sam'),
                            'Total_Tailors'    => collect($zones)->sum('Total_Tailors'),
                            'Total_Dir_Lab'    => collect($zones)->sum('Total_Dir_Lab'),
                            'Total_Ern_EfficiencyPerc' => $unitEfficiency,
                            'zones' => $zones,
                        ];
                    })
                    ->values()
                    ->all();

                $overallErnMins = collect($units)->sum('Total_Ern_Mins');
                $overallAvlMins = collect($units)->sum('Total_Avl_Mins');
                $overallEfficiency = $overallAvlMins > 0 ? round(($overallErnMins / $overallAvlMins) * 100, 2) : 0;
                $totalTailors = collect($units)->sum('Total_Tailors');
                $totalDirLab  = collect($units)->sum('Total_Dir_Lab');
                $overallDirectLabour = $totalTailors + $totalDirLab;

                $overallTotals = json_decode(json_encode($dashboardData), true);
                $overallTotals[] = ['Type' => 'Direct Labour', 'Cutting' => round($overallDirectLabour, 2), 'Slno' => 7];
                $overallTotals[] = ['Type' => 'Efficiency', 'Cutting' => $overallEfficiency, 'Slno' => 6];

                return [
                    'overall_totals' => $overallTotals,
                    'units' => $units,
                ];
            });

            return response()->json(['success' => true, 'overall_totals' => $final['overall_totals'], 'units' => $final['units']]);
        } catch (\Throwable $e) {
            Log::error('ProductionController gut_dailysnapshot error', ['message' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching data'], 500);
        }
    }

    public function gut_linestopminis(Request $request)
    {
        try {
            $type = $request->type;
            $portaltype = ucfirst($request->portaltype);
            $fdate = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate ? Carbon::parse($request->enddate)->format('d-M-Y') : $fdate;

            $cacheKey = "line_stop_minis_{$fdate}_{$enddate}_{$type}_{$portaltype}";
            $cacheTTL = env('LINE_STOP_CACHE_TTL', 1800);

            $resultdata = Cache::remember($cacheKey, $cacheTTL, function () use ($fdate, $enddate, $portaltype) {
                if($portaltype == 'Gloves'){
                    return DB::select('exec  Chennai_Erp.DBO.Line_Stop_Minis_gloves  @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                } else if($portaltype == 'Woven'){
                    return DB::select('exec Chennai_Erp.DBO.Line_Stop_Minis_Woven @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                } else if($portaltype == 'Headwear'){
                    return DB::select('exec Chennai_Erp.DBO.Line_Stop_Minis_Headwear @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                }
                return [];
            });

            $collection = collect($resultdata);
            $grouped = $collection->groupBy(['Unit', 'Zone', 'New_Line_No']);

            $units = $grouped
                ->sortKeys()
                ->map(function ($unitGroup, $unitName) {
                    $zones = $unitGroup
                        ->sortKeys()
                        ->map(function ($zoneGroup, $zoneName) {
                            $lines = $zoneGroup
                                ->sortKeys()
                                ->map(function ($lineRows, $lineNo) {
                                    $linesData = collect($lineRows)->map(fn($line) => [
                                        'Order_No'   => $line->Order_No,
                                        'DTM'        => (float) $line->DTM,
                                        'Tot_DTM'    => (float) $line->Tot_DTM,
                                        'Reason'     => $line->Reason,
                                        'Reason_Lkup'=> $line->Reason_Lkup,
                                        'Loss_Amnt'  => (float) $line->Loss_Amnt,
                                        'Remarks'    => $line->Remarks,
                                        'Date'       => $line->DTM_Date ? Carbon::parse($line->DTM_Date)->format('d-m-y') : null,
                                        'Style'      => $line->Style  ?? '-',
                                        'Color'      => $line->Color  ?? '-',
                                        'Buyer'      => $line->Buyer  ?? '-',
                                        'Start_Time' => $line->Start_Time ? Carbon::parse($line->Start_Time)->format('h:i A') : '-',
                                        'End_Time'   => $line->End_Time ? Carbon::parse($line->End_Time)->format('h:i A') : '-',
                                    ]);

                                    $totalLineDTM = $linesData->sum('DTM');
                                    $totalLineTotDTM = $linesData->sum('Tot_DTM');

                                    return [
                                        'Line_No'       => $lineNo,
                                        'total_DTM'     => $totalLineDTM,
                                        'total_Tot_DTM' => $totalLineTotDTM,
                                        'data'          => $linesData->values(),
                                    ];
                                });

                            $totalZoneDTM = $lines->sum('total_DTM');

                            return [
                                'Zone'      => $zoneName,
                                'total_DTM' => $totalZoneDTM,
                                'lines'     => $lines->values(),
                            ];
                        });

                    $totalUnitDTM = $zones->sum('total_DTM');

                    return [
                        'Unit'      => $unitName,
                        'total_DTM' => $totalUnitDTM,
                        'zones'     => $zones->values(),
                    ];
                })
                ->values();

            return response()->json(['success' => true, 'units' => $units]);
        } catch (\Throwable $e) {
            Log::error('Error in gut_linestopminis: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching line stop minis.'], 500);
        }
    }

    public function gut_cuttingreport(Request $request)
    {
        try {
            $type    = $request->type;
            $portaltype = ucfirst($request->portaltype);
            $fdate   = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate ? Carbon::parse($request->enddate)->format('d-M-Y') : $fdate;
            $unitCode = $request->unit_code;

            $cacheKey = "cutting_details_{$fdate}_{$enddate}_{$type}_{$portaltype}";

            $resultData = Cache::remember($cacheKey, 300, function () use ($fdate, $enddate, $portaltype) {
                if($portaltype == 'Gloves'){
                    return DB::select('exec chennai_erp.dbo.GetCuttingDetails_Gloves @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                } else if($portaltype == 'Woven'){
                    return DB::select('exec Chennai_Erp.DBO.GetCuttingDetails_Woven @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                } else if($portaltype == 'Headwear'){
                    return DB::select('exec Chennai_Erp.DBO.GetCuttingDetails_Headwear @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                }
                return [];
            });

            $collection = collect($resultData);
            if (!empty($unitCode)) {
                $collection = $collection->filter(fn($row) => isset($row->Unit) && $row->Unit == $unitCode);
            }

            $units = $collection->groupBy('Unit')->map(function ($unitItems, $unitName) {
                $unitCutQtySum = $unitItems->sum(fn($item) => (float) ($item->Cut_Qty ?? 0));
                $machines = $unitItems->groupBy('Machine_type')->map(function ($machineItems, $machineName) {
                    $machineCutQtySum = $machineItems->sum(fn($item) => (float) ($item->Cut_Qty ?? 0));
                    // Convert rows to arrays for consistent JSON encoding
                    $machineData = $machineItems->map(fn($it) => (array) $it)->values();
                    return [
                        'Cut_Qty_Sum' => $machineCutQtySum,
                        'data' => $machineData,
                    ];
                });

                return [
                    'Cut_Qty_Sum' => $unitCutQtySum,
                    'machines' => $machines,
                ];
            });

            return response()->json(['success' => true, 'units' => $units]);
        } catch (\Throwable $e) {
            Log::error('Error fetching cutting report: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function gut_wipdetail(Request $request)
    {
        try {
            $type       = $request->type;
            $portaltype = ucfirst($request->portaltype);
            $fdate = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate ? Carbon::parse($request->enddate)->format('d-M-Y') : $fdate;

            $cacheKey = "wip_details_{$fdate}_{$enddate}_{$type}_{$portaltype}";
            $cacheTTL = env('WIP_CACHE_TTL', 300);

            $resultData = Cache::remember($cacheKey, $cacheTTL, function () use ($fdate, $enddate, $portaltype) {
                if ($portaltype == 'Gloves') {
                    return DB::select('exec Chennai_Erp.Dbo.WIP_Proc_Gloves');
                } elseif ($portaltype == 'Woven') {
                    return DB::select('exec Chennai_Erp.DBO.WIP_Proc_woven');
                } elseif ($portaltype == 'Headwear') {
                    return DB::select('exec Chennai_Erp.DBO.WIP_Proc_headwear');
                } else {
                    return DB::select('exec Chennai_Erp.DBO.WIP_Proc @Fdate = ?, @Tdate = ?', [$fdate, $enddate]);
                }
            });

            $collection = collect($resultData);
            $units = $collection->groupBy('Unit')->map(function ($unitGroup, $unitName) {
                $types = $unitGroup->groupBy('Type')->map(function ($typeGroup, $typeName) {
                    $buyers = $typeGroup->groupBy('Buyer')->map(function ($buyerGroup, $buyerName) {
                        return [
                            'Buyer' => $buyerName,
                            'data'  => $buyerGroup->values()
                        ];
                    })->values();

                    return [
                        'Type'   => $typeName,
                        'buyers' => $buyers
                    ];
                })->values();

                return [
                    'Unit'  => $unitName,
                    'types' => $types
                ];
            })->values();

            return response()->json(['success' => true, 'units' => $units], 200);
        } catch (\Throwable $e) {
            Log::error('Error in gut_wipdetail: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching WIP details.'], 500);
        }
    }

    public function gut_smstockdetail(Request $request)
    {
        try {
            $type       = $request->type;
            $portaltype = ucfirst($request->portaltype);
            $fdate = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate ? Carbon::parse($request->enddate)->format('d-M-Y') : $fdate;

            $cacheKey = "smstock_details_{$fdate}_{$enddate}_{$type}_{$portaltype}";
            $cacheTTL = env('SM_STOCK_CACHE_TTL', 300);

            $resultData = Cache::remember($cacheKey, $cacheTTL, function () use ($fdate, $enddate, $portaltype) {
                if ($portaltype == 'Gloves') {
                    return DB::select('exec chennai_erp.DBO.SM_Proc_Gloves');
                } elseif ($portaltype == 'Woven') {
                    return DB::select('exec Chennai_Erp.DBO.SM_Proc_Woven');
                } elseif ($portaltype == 'Headwear') {
                    return DB::select('exec Chennai_Erp.DBO.SM_Proc_Woven');
                } else {
                    return DB::select('exec Chennai_Erp.DBO.SM_Proc');
                }
            });

            $collection = collect($resultData)->map(fn($r) => (array)$r);
            $units = $collection->groupBy('Unit')->map(function ($unitGroup, $unitName) {
                $buyers = $unitGroup->groupBy('Buyer')->map(function ($buyerGroup, $buyerName) {
                    $normalizedBuyer = trim((string)$buyerName) === '' ? 'Waiting' : (string)$buyerName;
                    $buyerData = $buyerGroup->map(function ($row) {
                        if (isset($row['Buyer']) && trim((string)$row['Buyer']) === '') {
                            $row['Buyer'] = 'Waiting';
                        }
                        return $row;
                    })->values()->all();
                    return [
                        'Buyer' => $normalizedBuyer,
                        'data'  => $buyerData,
                    ];
                })->values();

                return [
                    'Unit'   => $unitName,
                    'buyers' => $buyers,
                ];
            })->values();

            return response()->json(['success' => true, 'units' => $units], 200);
        } catch (\Throwable $e) {
            Log::error('gutsmstockDetails error: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching SM Stock details.'], 500);
        }
    }

    public function gut_sizeBreakupCutting(Request $request)
    {
        $layID = $request->layid;
        $cacheKey = "gut_size_breakup_cutting_{$layID}";
        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID) {
                return DB::select('exec Chennai_Erp.DBO.SIze_Brkup_Cutting_Proc_Woven @Masid = ?', [$layID]);
            });
            return response()->json(['success' => true, 'data' => $data]);
        } catch (\Throwable $e) {
            Log::error('gut_sizeBreakupCutting error', ['message' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching size breakup data'], 500);
        }
    }

    public function gut_getAckBreakdown(Request $request)
    {
        $layID = $request->layid;
        $arg = $request->arg;
        $cacheKey = "gut_size_breakup_ack_{$layID}_{$arg}";
        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID, $arg) {
                return DB::select('exec Chennai_Erp.Dbo.SIze_Brkup_AckPending_Proc_Woven @Masid = ? , @Arg = ?', [$layID, $arg]);
            });
            return response()->json(['success' => true, 'data' => $data]);
        } catch (\Throwable $e) {
            Log::error('gut_getAckBreakdown error', ['message' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching size breakup data'], 500);
        }
    }

    public function gut_sizerollbreakup(Request $request)
    {
        $layID = $request->layid;
        $cacheKey = "size_roll_cutting_{$layID}";
        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID) {
                return DB::select('exec CHENNAI_ERP.dbo.Rollwise_Breakup_Cutting_Proc_Woven @Masid = ?', [$layID]);
            });
            return response()->json(['success' => true, 'data' => $data]);
        } catch (\Throwable $e) {
            Log::error('gut_sizerollbreakup error', ['message' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching size breakup data'], 500);
        }
    }

    public function gut_directlabour_details(Request $request)
    {
        try {
            $type = $request->type ?? 1;
            $portaltype = ucfirst($request->portaltype ?? 'Woven');
            $fdate = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate ? Carbon::parse($request->enddate)->format('d-M-Y') : $fdate;

            $cacheKey = "{$portaltype}_directlabour1_{$fdate}_{$enddate}_{$type}";
            $rawData = Cache::remember($cacheKey, now()->addMinutes(30), function () use ($portaltype, $fdate, $enddate, $type) {
                switch ($portaltype) {
                    case 'GLoves':
                        return DB::select('EXEC vaahini_erp_gainup.Dbo.Gloves_Emp_Allocation_Proc @Fdate = ?', [$fdate]);
                    case 'Woven':
                        return DB::select('EXEC VAAHINI_ERP_GAINUP.DBO.Woven_Emp_Allocation_Proc @Fdate = ?', [$fdate]);
                    case 'Headwear':
                        return DB::select('EXEC VAAHINI_ERP_GAINUP.DBO.Headwear_Direct_Labour_Proc @Fdate = ?', [$fdate]);
                    default:
                        return [];
                }
            });

            $collection = collect($rawData);
            $grouped = $collection->groupBy(['Unit', 'Zone']);
            $units = [];
            foreach ($grouped as $unit => $zones) {
                $unitTotalEmp = 0;
                $unitZones = [];
                foreach ($zones as $zone => $rows) {
                    $zoneCollection = collect($rows);
                    $zoneEmpTotal = (int) $zoneCollection->sum('Emp_Cnt');
                    $unitZones[] = [
                        'Zone' => $zone,
                        'Total_Emp_Cnt' => $zoneEmpTotal,
                        'data' => $zoneCollection->values()->toArray(),
                    ];
                    $unitTotalEmp += $zoneEmpTotal;
                }
                $units[] = [
                    'Unit' => $unit,
                    'Unit_Code' => $unit,
                    'Total_Emp_Cnt' => $unitTotalEmp,
                    'zones' => $unitZones,
                ];
            }

            return response()->json(['success' => true, 'units' => $units]);
        } catch (\Throwable $e) {
            Log::error('Direct Labour Details Error: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Failed to load direct labour details.'], 500);
        }
    }

    public function gutproductionhistoricaldata(Request $request)
    {
        ini_set('memory_limit', '512M');
        try {
            $type        = $request->type ?? 1;
            $fdate       = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate     = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;
            $portaltype  = ucfirst($request->portaltype ?? 'Woven');

            $cacheKey = "audit_historical_{$type}_".($fdate ?? 'no_from')."_".($enddate ?? 'no_end')."_{$portaltype}";
            Cache::forget($cacheKey);

            $data = Cache::remember($cacheKey, 3600, function () use ($type, $fdate, $enddate, $portaltype) {
                switch (strtolower($portaltype)) {
                    case 'woven':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.Audit_Details_History_DB_Proc_Woven';
                        break;
                    case 'gloves':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.Audit_Details_History_DB_Proc_Gloves';
                        break;
                    case 'headwear':
                        $procedure = 'VAAHINI_ERP_GAINUP_SOCKS.DBO.Audit_Details_History_DB_Proc';
                        break;
                    default:
                        $procedure = 'VAAHINI_ERP_GAINUP.DBO.Audit_Details_History_DB_Proc';
                        break;
                }
                if ($fdate && $enddate) {
                    return DB::select("EXEC {$procedure} @Arg = ?, @Fdate = ?, @Tdate = ?", [$type, $fdate, $enddate]);
                } else {
                    return DB::select("EXEC {$procedure} @Arg = ?", [$type]);
                }
            });

            $dataArray = json_decode(json_encode($data), true);
            if (empty($dataArray)) { return response()->json(['success' => true, 'data' => []]); }

            $valueColumn = $type == 2 ? 'Qty' : 'Qty';
            $grouped = [];
            foreach ($dataArray as $row) {
                $month = $row['Mon_Name'] ?? 'Unknown';
                $unit  = $row['Unit'] ?? 'Unknown';
                $buyer = $row['Buyer'] ?? 'Unknown';
                $line  = $row['Line_No'] ?? 'Unknown';
                $qty   = (float) ($row[$valueColumn] ?? 0);
                if (!isset($grouped[$month])) {
                    $grouped[$month] = ['Mon_Name' => $month, 'month_total_Audit_Qty' => 0, 'units' => []];
                }
                $grouped[$month]['month_total_Audit_Qty'] += $qty;
                if (!isset($grouped[$month]['units'][$unit])) {
                    $grouped[$month]['units'][$unit] = ['Unit' => $unit, 'buyers' => []];
                }
                if (!isset($grouped[$month]['units'][$unit]['buyers'][$buyer])) {
                    $grouped[$month]['units'][$unit]['buyers'][$buyer] = ['Buyer' => $buyer, 'buyer_total_Audit_Qty' => 0, 'lines' => []];
                }
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['buyer_total_Audit_Qty'] += $qty;
                if (!isset($grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line])) {
                    $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line] = ['Line_No' => $line, 'line_total_Audit_Qty' => 0, 'data' => []];
                }
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line]['line_total_Audit_Qty'] += $qty;
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line]['data'][] = $row;
            }

            $grouped = array_values(array_map(function ($month) {
                ksort($month['units']);
                $month['units'] = array_values(array_map(function ($unit) {
                    $unit['buyers'] = array_values(array_map(function ($buyer) {
                        $buyer['lines'] = array_values($buyer['lines']);
                        return $buyer;
                    }, $unit['buyers']));
                    return $unit;
                }, $month['units']));
                return $month;
            }, $grouped));

            return response()->json(['success' => true, 'portal' => $portaltype, 'data' => $grouped, 'cached' => Cache::has($cacheKey)]);
        } catch (\Throwable $e) {
            Log::error('Unexpected error in gutproductionhistoricaldata: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching data.'], 500);
        }
    }

    public function gutproduction_minis_historicaldata(Request $request)
    {
        ini_set('memory_limit', '512M');
        try {
            $type        = $request->type ?? 1;
            $fdate       = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate     = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;
            $portaltype  = ucfirst($request->portaltype);

            $cacheKey = "gut_historical_details_minis_{$type}_".($fdate ?? 'no_from')."_".($enddate ?? 'no_end')."_{$portaltype}";
            Cache::forget($cacheKey);

            $data = Cache::remember($cacheKey, 600, function () use ($type, $fdate, $enddate, $portaltype) {
                switch ($portaltype) {
                    case 'Woven':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.DTM_Details_History_DB_Proc_Woven';
                        break;
                    case 'Gloves':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.DTM_Details_History_DB_Proc_Gloves';
                        break;
                    case 'Headwear':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.DTM_Details_History_DB_Proc_headwear';
                        break;
                    default:
                        $procedure = null;
                        break;
                }
                if (!$procedure) return [];
                if ($fdate && $enddate) {
                    return DB::select("EXEC {$procedure} @Arg = ?, @Fdate = ?, @Tdate = ?", [$type, $fdate, $enddate]);
                } else {
                    return DB::select("EXEC {$procedure} @Arg = ?", [$type]);
                }
            });

            $dataArray = json_decode(json_encode($data), true);
            $grouped = collect($dataArray)
                ->groupBy('Mon_Name')
                ->map(function ($monthGroup) {
                    $monthTotal = $monthGroup->sum('Qty');
                    $units = $monthGroup
                        ->groupBy('Unit')
                        ->sortBy(function ($unitGroup, $unitName) { return $unitName; })
                        ->map(function ($unitGroup) {
                            return [
                                'Unit' => $unitGroup->first()['Unit'],
                                'buyers' => $unitGroup->groupBy('Buyer')->map(function ($buyerGroup) {
                                    $buyerTotal = $buyerGroup->sum('Qty');
                                    return [
                                        'Buyer' => $buyerGroup->first()['Buyer'],
                                        'buyer_total_Audit_Qty' => $buyerTotal,
                                        'lines' => $buyerGroup->groupBy('Line_No')->map(function ($lineGroup) {
                                            return [
                                                'Line_No' => $lineGroup->first()['Line_No'],
                                                'line_total_Audit_Qty' => $lineGroup->sum('Qty'),
                                                'data' => array_values($lineGroup->toArray()),
                                            ];
                                        })->values()->toArray(),
                                    ];
                                })->values()->toArray(),
                            ];
                        })->values()->toArray();
                    return [
                        'Mon_Name' => $monthGroup->first()['Mon_Name'],
                        'month_total_Audit_Qty' => $monthTotal,
                        'units' => $units
                    ];
                })->values()->toArray();

            return response()->json(['success' => true, 'portal' => $portaltype, 'data' => $grouped, 'cached' => Cache::has($cacheKey)]);
        } catch (\Throwable $e) {
            Log::error('Unexpected error in gutproduction_minis_historicaldata: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching data.'], 500);
        }
    }

    public function gutproduction_cutting_historicaldata(Request $request)
    {
        ini_set('memory_limit', '512M');
        try {
            $portaltype = ucfirst($request->portaltype);
            $type       = $request->type ?? 1;
            $fdate      = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate    = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;

            $cacheKey = "gut_historical_details_cutting_{$portaltype}_{$type}_".($fdate ?? 'no_from')."_".($enddate ?? 'no_end');
            Cache::forget($cacheKey);

            $data = Cache::remember($cacheKey, 600, function () use ($portaltype, $fdate, $enddate) {
                switch ($portaltype) {
                    case 'Woven':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.Cutting_Details_History_DB_Proc_Woven';
                        break;
                    case 'Gloves':
                        $procedure = 'Vaahini_Erp_Gainup.Dbo.Cutting_Details_History_DB_Proc_Gloves';
                        break;
                    case 'Headwear':
                        $procedure = 'VAAHINI_ERP_GAINUP_HEADWEAR.DBO.Cutting_Details_History_DB_Proc';
                        break;
                    default:
                        throw new \Exception("Invalid portal type: {$portaltype}");
                }
                if ($fdate && $enddate) {
                    return DB::select("EXEC {$procedure} @Fdate = ?, @Tdate = ?", [$fdate, $enddate]);
                } else {
                    return DB::select("EXEC {$procedure}");
                }
            });

            $dataArray = json_decode(json_encode($data), true);
            $grouped = collect($dataArray)
                ->groupBy('Mon_Name')
                ->map(function ($monthGroup) {
                    $monthTotal = $monthGroup->sum('Qty');
                    $units = $monthGroup
                        ->groupBy('Unit')
                        ->sortKeys()
                        ->map(function ($unitGroup) {
                            return [
                                'Unit' => $unitGroup->first()['Unit'],
                                'buyers' => $unitGroup->groupBy('Buyer')->map(function ($buyerGroup) {
                                    return [
                                        'Buyer' => $buyerGroup->first()['Buyer'],
                                        'buyer_total_Audit_Qty' => $buyerGroup->sum('Qty'),
                                        'data'  => array_values($buyerGroup->toArray()),
                                    ];
                                })->values()->toArray()
                            ];
                        })->values()->toArray();
                    return [
                        'Mon_Name' => $monthGroup->first()['Mon_Name'],
                        'month_total_Audit_Qty' => $monthTotal,
                        'units' => $units
                    ];
                })->values()->toArray();

            return response()->json(['success' => true, 'portaltype' => $portaltype, 'data' => $grouped, 'cached' => Cache::has($cacheKey)]);
        } catch (\Throwable $e) {
            Log::error('Unexpected error in gutproduction_cutting_historicaldata: '.$e->getMessage());
            return response()->json(['success' => false, 'message' => 'Something went wrong while fetching data.'], 500);
        }
    }
}