<?php

namespace App\Http\Controllers\LifeAtGainup\Merchandiser;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class SocksProductionController extends Controller
{
    public function production_socks_dashboard()
    {
        return view('LifeAtGainup.socksproduction.production_socks_dashboard');
    }

    public function production_gar_dashboard_beta()
    {
        return view('LifeAtGainup.Production.prod_gar_static');
    }

    public function history_dashboard()
    {
        return view('LifeAtGainup.Production.history_dashboard');
    }

    public function cutting_output()
    {
        return view('LifeAtGainup.Production.Cutting_output_dashboard');
    }

	public function wip()
    {
        return view('LifeAtGainup.Production.wip');
    }

	public function smstock()
    {
        return view('LifeAtGainup.Production.smstock');
    }

	public function directlabour()
    {
        return view('LifeAtGainup.Production.directlabour');
    }

    public function socks_linestop_analysis()
    {
        return view('LifeAtGainup.socksproduction.Linestop_analysis_dashboard');
    }

    public function issued()
    {
        return view('LifeAtGainup.Production.issued');
    }

    public function socksdailysnapshot(Request $request){
        $type = $request->type;

        $fdate = $request->fromdate
            ? Carbon::parse($request->fromdate)->format('d-M-Y')
            : Carbon::today()->format('d-M-Y');

        $enddate = $request->enddate
            ? Carbon::parse($request->enddate)->format('d-M-Y')
            : $fdate;

        $cacheKey = "socks_daily_snapshot_{$fdate}_{$enddate}_{$type}";

        try {
            Cache::forget($cacheKey);

            $final = Cache::remember($cacheKey, 600, function () use ($fdate, $enddate) {

                $resultData = DB::select(
                    'EXEC FITSOCKS.DBO.Line_Output_DB_Socks @Fdate = ?, @Tdate = ?',
                    [$fdate, $enddate]
                );

                $dashboardData = DB::select(
                    'EXEC VAAHINI_ERP_GAINUP.DBO.Production_Main_DB_Proc @Fdate = ?, @Tdate = ?',
                    [$fdate, $enddate]
                );

                $rows = json_decode(json_encode($resultData), true);
                $collection = collect($rows);

                $units = $collection
                    ->groupBy('Unit')
                    ->map(function ($unitGroup, $unitName) {

                        $dates = collect($unitGroup)
                            ->groupBy(fn($it) => Carbon::parse($it['EntryDate'])->format('Y-m-d'))
                            ->map(function ($dateGroup, $dateKey) {

                                $shifts = collect($dateGroup)
                                    ->groupBy(fn($it) => trim($it['Shift']))
                                    ->map(function ($shiftGroup, $shiftName) {

                                        $totalOutput   = $shiftGroup->sum(fn($it) => floatval($it['Output_Qty'] ?? 0));
                                        $overallTarget = $shiftGroup->sum(fn($it) => floatval($it['Target'] ?? 0));
                                        $totalWip      = $shiftGroup->sum(fn($it) => floatval($it['Line_WIP'] ?? 0));
                                        $totalErnMins  = $shiftGroup->sum(fn($it) => floatval($it['Ern_Mins'] ?? ($it['Ern Mins'] ?? 0)));
                                        $totalDirLab   = $shiftGroup->sum(fn($it) => floatval($it['Dir_Lab'] ?? 0));
                                        $noOfMachine   = floatval($shiftGroup->first()['No_Of_Machine'] ?? 0);
                                        $totalTailors  = floatval($shiftGroup->first()['Tailor'] ?? 0);

                                        $samPerPiece = $totalOutput > 0 ? ($totalErnMins / $totalOutput) : 0;
                                        $effPerc     = ($overallTarget > 0) ? (($totalOutput / $overallTarget) * 100) : 0;

                                        return [
                                            'Shift' => $shiftName,
                                            'No_Of_Machine' => $noOfMachine,
                                            'Total_Output_Qty' => $totalOutput,
                                            'Total_Line_WIP' => $totalWip,
                                            'Total_Ern_Mins' => round($totalErnMins, 2),
                                            'Total_Ern_Sam' => round($samPerPiece, 2),
                                            'Total_Tailors' => $totalTailors,
                                            'Total_Dir_Lab' => $totalDirLab,
                                            'Total_Ern_Rundays' => $noOfMachine,
                                            'Total_Ern_EfficiencyPerc' => round($effPerc, 2),
                                            'Target' => $overallTarget,
                                            'data' => $shiftGroup->map(function ($row) {
                                                $row['Ern_Mins'] = $row['Ern_Mins'] ?? ($row['Ern Mins'] ?? 0);
                                                unset($row['Ern Mins']);
                                                return $row;
                                            })->values()->all(),
                                        ];
                                    })
                                    ->values()
                                    ->all();

                                $dateOutput   = $dateGroup->sum(fn($it) => floatval($it['Output_Qty'] ?? 0));
                                $dateTarget   = $dateGroup->sum(fn($it) => floatval($it['Target'] ?? 0));
                                $dateWip      = $dateGroup->sum(fn($it) => floatval($it['Line_WIP'] ?? 0));
                                $dateErnMins  = $dateGroup->sum(fn($it) => floatval($it['Ern_Mins'] ?? ($it['Ern Mins'] ?? 0)));
                                $dateDirLab   = $dateGroup->sum(fn($it) => floatval($it['Dir_Lab'] ?? 0));

                                $dayWiseMachineAvg = collect($shifts)->avg('No_Of_Machine');
                                $dateTailors       = collect($shifts)->sum('Total_Tailors');

                                $samPerPiece = $dateOutput > 0 ? ($dateErnMins / $dateOutput) : 0;
                                $effPerc     = ($dateTarget > 0) ? (($dateOutput / $dateTarget) * 100) : 0;
                                $formattedDate = Carbon::parse($dateKey)->format('d-m-y');

                                return [
                                    'EntryDate' => $formattedDate,
                                    'Total_Output_Qty' => $dateOutput,
                                    'Total_Line_WIP' => $dateWip,
                                    'Total_Ern_Mins' => round($dateErnMins, 2),
                                    'Total_Ern_Sam' => round($samPerPiece, 2),
                                    'Total_Tailors' => $dateTailors,
                                    'Total_Dir_Lab' => $dateDirLab,
                                    'Total_Ern_Rundays' => round($dayWiseMachineAvg, 2),
                                    'Total_Ern_EfficiencyPerc' => round($effPerc, 2),
                                    'No_Of_Machine' => round($dayWiseMachineAvg, 2),
                                    'Target' => round($dateTarget, 2),
                                    'shifts' => $shifts,
                                ];
                            })
                            ->sortKeys()
                            ->values()
                            ->all();

                        $unitOutput  = $unitGroup->sum(fn($it) => floatval($it['Output_Qty'] ?? 0));
                        $unitTarget  = $unitGroup->sum(fn($it) => floatval($it['Target'] ?? 0));
                        $unitWip     = floatval($unitGroup->first()['Line_WIP'] ?? 0);
                        $unitErnMins = $unitGroup->sum(fn($it) => floatval($it['Ern_Mins'] ?? ($it['Ern Mins'] ?? 0)));
                        $unitDirLab  = $unitGroup->sum(fn($it) => floatval($it['Dir_Lab'] ?? 0));

                        $unitMachineAvg = collect($dates)->avg('No_Of_Machine');
                        $unitTailors    = collect($dates)->avg('Total_Tailors');

                        $samPerPiece = $unitOutput > 0 ? ($unitErnMins / $unitOutput) : 0;
                        $effPerc     = ($unitTarget > 0) ? (($unitOutput / $unitTarget) * 100) : 0;

                        return [
                            'Unit' => $unitName,
                            'Total_Output_Qty' => $unitOutput,
                            'Total_Line_WIP' => $unitWip,
                            'Total_Ern_Mins' => round($unitErnMins, 2),
                            'Total_Ern_Sam' => round($samPerPiece, 2),
                            'Total_Tailors' => $unitTailors,
                            'Total_Dir_Lab' => $unitDirLab,
                            'Total_Ern_Rundays' => round($unitMachineAvg, 2),
                            'Total_Ern_EfficiencyPerc' => round($effPerc, 2),
                            'No_Of_Machine' => round($unitMachineAvg, 2),
                            'Target' => round($unitTarget, 2),
                            'dates' => $dates,
                        ];
                    })
                    ->sortKeys()
                    ->values()
                    ->all();

                $totalOutput = $collection->sum(fn($it) => floatval($it['Output_Qty'] ?? 0));
                $totalTarget = $collection->sum(fn($it) => floatval($it['Target'] ?? 0));
                $totalTailors = $collection->sum(fn($it) => floatval($it['Tailors'] ?? 0));
                $totalDirLab  = $collection->sum(fn($it) => floatval($it['Dir_Lab'] ?? 0));

                $overallDirectLabour = $totalTailors + $totalDirLab;
                $overallEfficiency = ($totalTarget > 0)
                    ? (($totalOutput / $totalTarget) * 100)
                    : 0;

                $overallTotals = json_decode(json_encode($dashboardData), true);
                $overallTotals[] = [
                    'Type' => 'Direct Labour',
                    'Cutting' => round($overallDirectLabour, 2),
                    'Slno' => 7
                ];
                $overallTotals[] = [
                    'Type' => 'Efficiency',
                    'Cutting' => round($overallEfficiency, 2),
                    'Slno' => 6
                ];

                return [
                    'overall_totals' => $overallTotals,
                    'units' => $units,
                ];
            });

            return response()->json([
                'success' => true,
                'overall_totals' => $final['overall_totals'],
                'units' => $final['units'],
            ]);

        } catch (\Throwable $e) {
            Log::error('ProductionController socksDailySnapshot error', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Something went wrong while fetching Socks Daily Snapshot'
            ], 500);
        }
    }

    public function productionhistoricaldata(Request $request){
        try {
            $type    = $request->type ?? 1;
            $fdate   = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;

            $cacheKey = "audit_historical_{$type}_" . ($fdate ?? 'no_from') . "_" . ($enddate ?? 'no_end');
            Cache::forget($cacheKey);

            try {
                $data = Cache::remember($cacheKey, 3600, function () use ($type, $fdate, $enddate) {
                    if ($fdate && $enddate) {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.Audit_Details_History_DB_Proc @Arg = ?, @Fdate = ?, @Tdate = ?',
                            [$type, $fdate, $enddate]
                        );
                    } else {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.Audit_Details_History_DB_Proc @Arg = ?',
                            [$type]
                        );
                    }
                });
            } catch (\Exception $e) {
                Log::warning("DB error in productionHistoricalData: {$e->getMessage()}");
                $data = Cache::get($cacheKey, []);
            }

            $dataArray = json_decode(json_encode($data), true);
            if (empty($dataArray)) {
                return response()->json(['success' => true, 'data' => []]);
            }

            $valueColumn = $type == 2 ? 'Qty' : 'Qty';
            $grouped = [];
            foreach ($dataArray as $row) {
                $month = $row['Mon_Name'] ?? 'Unknown';
                $unit  = $row['Unit'] ?? 'Unknown';
                $buyer = $row['Buyer'] ?? 'Unknown';
                $line  = $row['Line_No'] ?? 'Unknown';
                $qty   = (float) ($row[$valueColumn] ?? 0);

                if (!isset($grouped[$month])) {
                    $grouped[$month] = [
                        'Mon_Name' => $month,
                        'month_total_Audit_Qty' => 0,
                        'units' => []
                    ];
                }
                $grouped[$month]['month_total_Audit_Qty'] += $qty;

                if (!isset($grouped[$month]['units'][$unit])) {
                    $grouped[$month]['units'][$unit] = [
                        'Unit' => $unit,
                        'buyers' => []
                    ];
                }

                if (!isset($grouped[$month]['units'][$unit]['buyers'][$buyer])) {
                    $grouped[$month]['units'][$unit]['buyers'][$buyer] = [
                        'Buyer' => $buyer,
                        'buyer_total_Audit_Qty' => 0,
                        'lines' => []
                    ];
                }
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['buyer_total_Audit_Qty'] += $qty;

                if (!isset($grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line])) {
                    $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line] = [
                        'Line_No' => $line,
                        'line_total_Audit_Qty' => 0,
                        'data' => []
                    ];
                }
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line]['line_total_Audit_Qty'] += $qty;
                $grouped[$month]['units'][$unit]['buyers'][$buyer]['lines'][$line]['data'][] = $row;
            }

            $grouped = array_values(array_map(function ($month) {
                $month['units'] = array_values(array_map(function ($unit) {
                    $unit['buyers'] = array_values(array_map(function ($buyer) {
                        $buyer['lines'] = array_values($buyer['lines']);
                        return $buyer;
                    }, $unit['buyers']));
                    return $unit;
                }, $month['units']));
                return $month;
            }, $grouped));

            return response()->json([
                'success' => true,
                'data'    => $grouped,
                'cached'  => Cache::has($cacheKey),
            ]);

        } catch (\Exception $e) {
            Log::error('Unexpected error in productionHistoricalData: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching data.',
            ], 500);
        }
    }

    public function production_minis_historicaldata(Request $request){
        try {
            $type    = $request->type ?? 1;
            $fdate   = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;

            $cacheKey = "historical_details_minis_{$type}_" . ($fdate ?? 'no_from') . "_" . ($enddate ?? 'no_end');

            try {
                $data = Cache::remember($cacheKey, 600, function () use ($type, $fdate, $enddate) {
                    if ($fdate && $enddate) {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.DTM_Details_History_DB_Proc @Arg = ?, @Fdate = ?, @Tdate = ?',
                            [$type, $fdate, $enddate]
                        );
                    } else {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.DTM_Details_History_DB_Proc @Arg = ?',
                            [$type]
                        );
                    }
                });
            } catch (\Exception $e) {
                Log::warning("DB error in productionHistoricalData: {$e->getMessage()}");
                $data = Cache::get($cacheKey, []);
            }

            $dataArray = json_decode(json_encode($data), true);

            $grouped = collect($dataArray)
                ->groupBy('Mon_Name')
                ->map(function ($monthGroup) {
                    $monthTotal = $monthGroup->sum('Qty');

                    return [
                        'Mon_Name' => $monthGroup->first()['Mon_Name'],
                        'month_total_Audit_Qty' => $monthTotal,
                        'units' => $monthGroup->groupBy('Unit')->map(function ($unitGroup) {
                            return [
                                'Unit' => $unitGroup->first()['Unit'],
                                'buyers' => $unitGroup->groupBy('Buyer')->map(function ($buyerGroup) {
                                    $buyerTotal = $buyerGroup->sum('Qty');

                                    return [
                                        'Buyer' => $buyerGroup->first()['Buyer'],
                                        'buyer_total_Audit_Qty' => $buyerTotal,
                                        'lines' => $buyerGroup->groupBy('Line_No')->map(function ($lineGroup) {
                                            return [
                                                'Line_No' => $lineGroup->first()['Line_No'],
                                                'line_total_Audit_Qty' => $lineGroup->sum('Qty'),
                                                'data' => array_values($lineGroup->toArray()),
                                            ];
                                        })->values()->toArray(),
                                    ];
                                })->values()->toArray()
                            ];
                        })->values()->toArray()
                    ];
                })->values()->toArray();

            return response()->json([
                'success' => true,
                'data'    => $grouped,
                'cached'  => Cache::has($cacheKey),
            ]);

        } catch (\Exception $e) {
            Log::error('Unexpected error in productionHistoricalData: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching data.',
            ], 500);
        }
    }

	public function production_cutting_historicaldata(Request $request){
        try {
            $type    = $request->type ?? 1;
            $fdate   = $request->fromdate ? Carbon::parse($request->fromdate)->format('d-M-Y') : null;
            $enddate = $request->enddate  ? Carbon::parse($request->enddate)->format('d-M-Y')  : null;

            $cacheKey = "historical_details_cutting_{$type}_" . ($fdate ?? 'no_from') . "_" . ($enddate ?? 'no_end');

            try {
                $data = Cache::remember($cacheKey, 600, function () use ($fdate, $enddate) {
                    if ($fdate && $enddate) {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.Cutting_Details_History_DB_Proc @Fdate = ?, @Tdate = ?',
                            [$fdate, $enddate]
                        );
                    } else {
                        return DB::select(
                            'EXEC VAAHINI_ERP_GAINUP.DBO.Cutting_Details_History_DB_Proc'
                        );
                    }
                });
            } catch (\Exception $e) {
                Log::warning("DB error in cuttingHistoricalData: {$e->getMessage()}");
                $data = Cache::get($cacheKey, []);
            }

            $dataArray = json_decode(json_encode($data), true);

            $grouped = collect($dataArray)
                ->groupBy('Mon_Name')
                ->map(function ($monthGroup) {
                    $monthTotal = $monthGroup->sum('Qty');

                    return [
                        'Mon_Name' => $monthGroup->first()['Mon_Name'],
                        'month_total_Audit_Qty' => $monthTotal,
                        'units' => $monthGroup->groupBy('Unit')->map(function ($unitGroup) {
                            return [
                                'Unit' => $unitGroup->first()['Unit'],
                                'buyers' => $unitGroup->groupBy('Buyer')->map(function ($buyerGroup) {
                                    return [
                                        'Buyer' => $buyerGroup->first()['Buyer'],
                                        'buyer_total_Audit_Qty' => $buyerGroup->sum('Qty'),
                                        'data'  => array_values($buyerGroup->toArray()),
                                    ];
                                })->values()->toArray()
                            ];
                        })->values()->toArray()
                    ];
                })->values()->toArray();

            return response()->json([
                'success' => true,
                'data'    => $grouped,
                'cached'  => Cache::has($cacheKey),
            ]);

        } catch (\Exception $e) {
            Log::error('Unexpected error in cuttingHistoricalData: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching data.',
            ], 500);
        }
    }

    public function sockslinestopminis(Request $request){
        try {
            $type = $request->type;
            $fdate = $request->fromdate
                ? Carbon::parse($request->fromdate)->format('d-M-Y')
                : Carbon::today()->format('d-M-Y');

            $enddate = $request->enddate
                ? Carbon::parse($request->enddate)->format('d-M-Y')
                : $fdate;

            $cacheKey = "socks_line_stop_minis_{$fdate}_{$enddate}_{$type}";
            $cacheTTL = env('LINE_STOP_CACHE_TTL', 1800);

            $resultdata = Cache::remember($cacheKey, $cacheTTL, function () use ($fdate, $enddate) {
                return DB::select(
                    'exec Chennai_Erp.DBO.Line_Stop_Minis_Socks @Fdate = ?, @Tdate = ?',
                    [$fdate, $enddate]
                );
            });

            $collection = collect($resultdata);
            $grouped = $collection->groupBy(['Unit', 'Zone', 'New_Line_No']);

            $units = $grouped
                ->sortKeys()
                ->map(function ($unitGroup, $unitName) {
                    $zones = $unitGroup
                        ->sortKeys()
                        ->map(function ($zoneGroup, $zoneName) {
                            $lines = $zoneGroup
                                ->sortKeys()
                                ->map(function ($lineRows, $lineNo) {
                                    $linesData = $lineRows->map(fn($line) => [
                                        'Order_No'   => $line->Order_No,
                                        'DTM'        => (float) $line->DTM_SHIFT,
                                        'Tot_DTM'    => (float) $line->Tot_DTM,
                                        'Reason'     => $line->Reason,
                                        'Reason_Lkup'=> $line->Reason_Lkup,
                                        'Loss_Amnt'  => (float) $line->Loss_Amnt,
                                        'Remarks'    => $line->Remarks,
                                        'Date'       => $line->DTM_Date
                                            ? Carbon::parse($line->DTM_Date)->format('d-m-y')
                                            : null,
                                        'Style'      => $line->Style  ?? '-',
                                        'Color'      => $line->Color  ?? '-',
                                        'Buyer'      => $line->Buyer  ?? '-',
                                        'Start_Time' => $line->Start_Time
                                            ? Carbon::parse($line->Start_Time)->format('h:i A')
                                            : '-',
                                        'End_Time'   => $line->End_Time
                                            ? Carbon::parse($line->End_Time)->format('h:i A')
                                            : '-',
                                    ]);

                                    $totalLineDTM = $linesData->sum('DTM');
                                    $totalLineTotDTM = $linesData->sum('Tot_DTM');

                                    return [
                                        'Line_No'       => $lineNo,
                                        'total_DTM'     => $totalLineDTM,
                                        'total_Tot_DTM' => $totalLineTotDTM,
                                        'data'          => $linesData->values(),
                                    ];
                                });

                            $totalZoneDTM = $lines->sum('total_DTM');

                            return [
                                'Zone'      => $zoneName,
                                'total_DTM' => $totalZoneDTM,
                                'lines'     => $lines->values(),
                            ];
                        });

                    $totalUnitDTM = $zones->sum('total_DTM');

                    return [
                        'Unit'      => $unitName,
                        'total_DTM' => $totalUnitDTM,
                        'zones'     => $zones->values(),
                    ];
                })
                ->values();

            return response()->json([
                'success' => true,
                'units'   => $units,
            ]);

        } catch (\Exception $e) {
            Log::error('Error in lineStopMinis: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching line stop minis.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function cuttingreport(Request $request){
        try {
            $type    = $request->type;
            $fdate   = $request->fromdate
                ? Carbon::parse($request->fromdate)->format('d-M-Y')
                : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate
                ? Carbon::parse($request->enddate)->format('d-M-Y')
                : $fdate;

            $cacheKey = "cutting_details_{$fdate}_{$enddate}_{$type}";

            $resultData = Cache::remember($cacheKey, 60 * 5, function () use ($fdate, $enddate) {
                return DB::select(
                    'exec Chennai_Erp.DBO.GetCuttingDetails @Fdate = ?, @Tdate = ?',
                    [$fdate, $enddate]
                );
            });

            $collection = collect($resultData);
            $unitWise = $collection->groupBy('Unit')->map(function ($unitItems) {
                $unitCutQtySum = $unitItems->sum(fn($item) => (float) $item->Cut_Qty);
                $machines = $unitItems->groupBy('Machine_type')->map(function ($machineItems) {
                    $machineCutQtySum = $machineItems->sum(fn($item) => (float) $item->Cut_Qty);
                    return [
                        'Cut_Qty_Sum' => $machineCutQtySum,
                        'data' => $machineItems->values(),
                    ];
                });

                return [
                    'Cut_Qty_Sum' => $unitCutQtySum,
                    'machines' => $machines,
                ];
            });

            return response()->json([
                'success' => true,
                'units' => $unitWise,
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching cutting report: '.$e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function wipdetail(Request $request){
        try {
            $type    = $request->type;
            $fdate   = $request->fromdate
                ? Carbon::parse($request->fromdate)->format('d-M-Y')
                : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate
                ? Carbon::parse($request->enddate)->format('d-M-Y')
                : $fdate;

            $cacheKey = "wip_details_{$fdate}_{$enddate}_{$type}";

            $resultData = Cache::remember($cacheKey, 60 * 5, function () {
                return DB::select('exec VAAHINI_ERP_GAINUP.DBO.WIP_Proc');
            });

            $collection = collect($resultData);

            $units = $collection->groupBy('Unit')->map(function ($unitGroup, $unitName) {
                $types = $unitGroup->groupBy('Type')->map(function ($typeGroup, $typeName) {
                    $buyers = $typeGroup->groupBy('Buyer')->map(function ($buyerGroup, $buyerName) {
                        return [
                            'Buyer' => $buyerName,
                            'data'  => $buyerGroup->values()
                        ];
                    })->values();

                    return [
                        'Type'   => $typeName,
                        'buyers' => $buyers
                    ];
                })->values();

                return [
                    'Unit'  => $unitName,
                    'types' => $types
                ];
            })->values();

            return response()->json([
                'success' => true,
                'units'   => $units
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching WIP details.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function smstockdetail(Request $request){
        try {
            $type    = $request->type;
            $fdate   = $request->fromdate
                ? Carbon::parse($request->fromdate)->format('d-M-Y')
                : Carbon::today()->format('d-M-Y');
            $enddate = $request->enddate
                ? Carbon::parse($request->enddate)->format('d-M-Y')
                : $fdate;

            $cacheKey = "smstock_details_{$fdate}_{$enddate}_{$type}";

            $resultData = Cache::remember($cacheKey, 60 * 5, function () {
                return DB::select('exec VAAHINI_ERP_GAINUP.DBO.SM_Proc');
            });

            $collection = collect($resultData);

            $units = $collection->groupBy('Unit')->map(function ($unitGroup, $unitName) {
                $buyers = $unitGroup->groupBy('Buyer')->map(function ($buyerGroup, $buyerName) {
                    return [
                        'Buyer' => substr($buyerName, 0, 5),
                        'data'  => $buyerGroup->values()
                    ];
                })->values();

                return [
                    'Unit'   => $unitName,
                    'buyers' => $buyers
                ];
            })->values();

            return response()->json([
                'success' => true,
                'units'   => $units
            ], 200);

        } catch (\Throwable $e) {
            Log::error('smstockDetails error', [
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while fetching SM Stock details.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }



    public function sizeBreakupCutting(Request $request){
        $layID = $request->layid;
        $cacheKey = "size_breakup_cutting_{$layID}";

        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID) {
                return DB::select(
                    'exec VAAHINI_ERP_GAINUP.DBO.SIze_Brkup_Cutting_Proc @Masid = ?',
                    [$layID]
                );
            });

            return response()->json([
                'success' => true,
                'data'    => $data,
            ]);

        } catch (\Throwable $e) {
            Log::error('ProductionController sizrBreakupCutting error', [
                'Lay_ID'  => $layID,
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
                'message' => 'Something went wrong while fetching size breakup data',
            ], 500);
        }

    }

    public function getAckBreakdown(Request $request){
        $layID = $request->layid;
        $agrumaent = $request->arg;
        $cacheKey = "size_breakup_ack_{$layID}_{$agrumaent}";

        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID, $agrumaent) {
                return DB::select(
                    'exec VAAHINI_ERP_GAINUP.DBO.SIze_Brkup_AckPending_Proc @Masid = ? , @Arg = ?',
                    [$layID ,$agrumaent]
                );
            });

            return response()->json([
                'success' => true,
                'data'    => $data,
            ]);

        } catch (\Throwable $e) {
            Log::error('ProductionController sizrBreakupAckpending error', [
                'Lay_ID'  => $layID,
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
                'message' => 'Something went wrong while fetching size breakup data',
            ], 500);
        }

    }

    public function smstockissued(Request $request){
        try {
            $ocnNo   = $request->ocnno   ?? 'GIL\\OCN04460';
            $itemID  = $request->itemid  ?? '913';
            $colorId = $request->colorid ?? '1297';
            $sizeId  = $request->sizeid  ?? '618';

            $cacheKey = "sm_stock_issued_{$ocnNo}_{$itemID}_{$colorId}_{$sizeId}";

            $data = Cache::remember($cacheKey, 600, function () use ($ocnNo, $itemID, $colorId, $sizeId) {
                return DB::select(
                    'EXEC VAAHINI_ERP_GAINUP.DBO.Line_Issued_Proc @Order_No = ?, @Itemid = ?, @Colorid = ?, @Sizeid = ?',
                    [$ocnNo, $itemID, $colorId, $sizeId]
                );
            });

            $dataArray = collect($data)->map(fn($d) => (array) $d);

            $grouped = $dataArray
                ->groupBy('Line')
                ->map(function ($items, $line) {
                    $sortedEntries = $items->sortByDesc(function ($item) {
                        return Carbon::parse($item['Edate']);
                    })->values();

                    return [
                        'Line' => $line,
                        'Total_Iss_Qty' => $sortedEntries->sum('Iss_Qty'),
                        'Entries' => $sortedEntries,
                    ];
                })
                ->values();

            return response()->json([
                'success' => true,
                'data' => $grouped,
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching smStockIssued data: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load stock issued data.',
            ], 500);
        }
    }

    public function sizerollbreakup(Request $request){
        $layID = $request->layid;
        $cacheKey = "size_roll_cutting_{$layID}";

        try {
            $data = Cache::remember($cacheKey, 600, function () use ($layID) {
                return DB::select(
                    'exec VAAHINI_ERP_GAINUP.DBO.Rollwise_Breakup_Cutting_Proc @Masid = ?',
                    [$layID]
                );
            });

            return response()->json([
                'success' => true,
                'data'    => $data,
            ]);

        } catch (\Throwable $e) {
            Log::error('ProductionController sizrBreakupCutting error', [
                'Lay_ID'  => $layID,
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
                'message' => 'Something went wrong while fetching size breakup data',
            ], 500);
        }
    }
}