<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;


class LineplannerController extends Controller
{
    public function index(Request $request)
    {
         // Default values for the view
         $unitFilter = $request->input('unit', '3');
         $startDate = $request->input('start_date', now()->toDateString());
         //$endDate = $request->input('end_date', '2025-09-23');
         $endDate = $request->input('end_date', \Carbon\Carbon::parse($startDate)->addMonth()->toDateString());
         
        // Generate default date range for display
        $dates = [];
        try {
            $start = Carbon::parse($startDate);
            $end = Carbon::parse($endDate);
            // Wrap daysUntil in collect() to convert Generator to Collection
            $dates = collect($start->daysUntil($end))->map(function ($date) {
                return $date->format('d-M');
            })->toArray();
        } catch (\Exception $e) {
           // \Log::error('Error parsing dates: ' . $e->getMessage());
        }

        return view('lineplanner', compact('dates', 'unitFilter', 'startDate', 'endDate'));
    }

    public function getLinePlanData(Request $request)
    {
        $fromDate = $request->query('from_date', '2025-09-01');
        $toDate   = $request->query('to_date', '2025-09-23');
        $unitCode = $request->query('unit_code', '13');

        // Convert "null" to actual null
        if ($unitCode === 'null') {
            $unitCode = null;
        }

        // Build DB query exactly like lineplanReport()
        if (!is_null($unitCode)) {
            $sql    = 'exec chennai_erp.dbo.Line_Planning_Report_Proc @Fdate = ?, @Tdate = ?, @Unit_Code = ?';
            $params = [$fromDate, $toDate, (int)$unitCode];
        } else {
            $sql    = 'exec chennai_erp.dbo.Line_Planning_Report_Proc @Fdate = ?, @Tdate = ?';
            $params = [$fromDate, $toDate];
        }

        // ✅ Direct DB execution (no try catch)
        $plandetails = DB::select($sql, $params);

        if (empty($plandetails)) {
            return response()->json([
                'success' => false,
                'message' => 'No data found',
                'dates'   => []
            ], 404);
        }

        // Convert to array to allow key access
        $dataArr = json_decode(json_encode($plandetails), true);

        // ✅ Process each record (same as your API response processing)
        $processedData = array_map(function ($record) {

            $filteredRecord = [
                'Unit'          => $record['Unit'] ?? 'AAG-III',
                'Buyer'         => $record['Buyer'] ?? '',
                'Line_No'       => $record['Line_No'] ?? '',
                'Ocn_No'        => $record['Ocn_No'] ?? '',
                'Item'          => $record['Item'] ?? '',
                'Color'         => $record['Color'] ?? '',
                'Sam'           => $record['Sam'] ?? '',
                'No_Of_Opr'     => $record['No_Of_Opr'] ?? '',
                'Targ_Eff'      => $record['Targ_Eff'] ?? '',
                'Targ_Qty'      => $record['Targ_Qty'] ?? '',
                'Total_Planned' => $record['Total_Planned'] ?? '',
                'Line_Name'     => $record['Line_Name'] ?? null,
            ];

            // ✅ Add dynamic date columns YYYY-MM-DD
            foreach ($record as $key => $val) {
                if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $key)) {
                    $filteredRecord[$key] = $val;
                }
            }

            return $filteredRecord;
        }, $dataArr);

        // ✅ Extract all date columns
        $dateColumns = array_filter(
            array_keys($dataArr[0] ?? []),
            fn ($key) => preg_match('/^\d{4}-\d{2}-\d{2}$/', $key)
        );

        // Format dates as d-M
        $dateColumns = array_map(function ($date) {
            return Carbon::parse($date)->format('d-M');
        }, $dateColumns);

        return response()->json([
            'success'          => true,
            'recordsTotal'     => count($processedData),
            'recordsFiltered'  => count($processedData),
            'draw'             => intval($request->query('draw', 1)),
            'data'             => $processedData,
            'dates'            => array_values($dateColumns),
            'message'          => 'details'
        ]);
    }

    
    public function getBuyerList()
    {
        try {
            $url = env('API_URL') . 'lineplanpolist';
            $response = Http::get($url);

            if ($response->successful()) {
                $data = $response->json()['data'] ?? [];
                return response()->json(['success' => true, 'data' => $data]);
            }

            return response()->json(['success' => false, 'message' => 'Failed to fetch buyer list'], 500);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
}