import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from './ui/table';
import { Button } from './ui/button';
import { CreditCard, Download, Eye, CheckCircle, AlertTriangle, Clock } from 'lucide-react';

interface AdvanceDetailsBreakupProps {
  data: any;
}

export function AdvanceDetailsBreakup({ data }: AdvanceDetailsBreakupProps) {
  if (!data || !data.supplierAdvancePaid) return null;

  // Generate mock advance details based on PO data
  const generateAdvanceDetails = () => {
    const advanceDetails = [];
    const advanceCount = Math.floor(Math.random() * 3) + 1; // 1-3 advance payments
    
    for (let i = 0; i < advanceCount; i++) {
      const advDate = new Date(data.poDate);
      advDate.setDate(advDate.getDate() + (i * 7)); // Stagger advances by 7 days
      
      const closeDate = new Date(advDate);
      closeDate.setDate(closeDate.getDate() + Math.floor(Math.random() * 30) + 15); // Close 15-45 days later
      
      const advanceAmount = Math.floor((data.poVal * 0.3) / advanceCount); // Split advance into parts
      const advancePercentage = ((advanceAmount / data.poVal) * 100);
      
      const isClosed = Math.random() > 0.3; // 70% chance of being closed
      
      advanceDetails.push({
        advVouchNo: `ADV${2025}${(parseInt(data.id.replace('PO', '')) + i).toString().padStart(4, '0')}`,
        advDate: advDate.toLocaleDateString('en-GB'),
        closeDate: isClosed ? closeDate.toLocaleDateString('en-GB') : null,
        poNo: data.id,
        advAmount: advanceAmount,
        advPercentage: advancePercentage.toFixed(1),
        status: isClosed ? 'Closed' : 'Open',
        supplier: data.supplier,
        paymentMode: ['Bank Transfer', 'Cheque', 'Online'][Math.floor(Math.random() * 3)],
        approvedBy: ['Mr.Britto', 'Mr.Udhaya'][Math.floor(Math.random() * 2)],
        bankRef: `BNK${Math.floor(Math.random() * 9999999).toString().padStart(7, '0')}`
      });
    }
    
    return advanceDetails;
  };

  const advanceDetails = generateAdvanceDetails();
  const totalAdvanceAmount = advanceDetails.reduce((sum, adv) => sum + adv.advAmount, 0);
  const openAdvances = advanceDetails.filter(adv => adv.status === 'Open').length;

  return (
    <div className="space-y-6">
      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <CreditCard className="h-4 w-4 text-blue-500" />
              <div>
                <p className="text-sm text-muted-foreground">Total Advances</p>
                <p className="text-lg font-bold">₹{totalAdvanceAmount.toLocaleString()}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <CheckCircle className="h-4 w-4 text-green-500" />
              <div>
                <p className="text-sm text-muted-foreground">Closed Advances</p>
                <p className="text-lg font-bold">{advanceDetails.length - openAdvances}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-red-500" />
              <div>
                <p className="text-sm text-muted-foreground">Open Advances</p>
                <p className="text-lg font-bold">{openAdvances}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Clock className="h-4 w-4 text-orange-500" />
              <div>
                <p className="text-sm text-muted-foreground">Avg Days to Close</p>
                <p className="text-lg font-bold">
                  {Math.floor(Math.random() * 20) + 15} days
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Detailed Advance Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <CreditCard className="h-5 w-5" />
              Advance Payment Details
            </div>
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Adv Vouch No</TableHead>
                  <TableHead>Adv Date</TableHead>
                  <TableHead>Close Date</TableHead>
                  <TableHead>PO No</TableHead>
                  <TableHead>Supplier</TableHead>
                  <TableHead className="text-right">Adv Amount</TableHead>
                  <TableHead className="text-right">Adv %</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Payment Mode</TableHead>
                  <TableHead>Approved By</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {advanceDetails.map((advance, index) => (
                  <TableRow key={index}>
                    <TableCell className="font-medium">{advance.advVouchNo}</TableCell>
                    <TableCell>{advance.advDate}</TableCell>
                    <TableCell>
                      {advance.closeDate || (
                        <Badge variant="outline" className="text-orange-600">
                          Pending
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell>{advance.poNo}</TableCell>
                    <TableCell className="max-w-[150px] truncate" title={advance.supplier}>
                      {advance.supplier}
                    </TableCell>
                    <TableCell className="text-right font-medium">
                      ₹{advance.advAmount.toLocaleString()}
                    </TableCell>
                    <TableCell className="text-right">{advance.advPercentage}%</TableCell>
                    <TableCell>
                      <Badge 
                        variant={advance.status === 'Closed' ? 'default' : 'destructive'}
                        className="text-xs"
                      >
                        {advance.status}
                      </Badge>
                    </TableCell>
                    <TableCell>{advance.paymentMode}</TableCell>
                    <TableCell>{advance.approvedBy}</TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="sm">
                          <Eye className="h-3 w-3" />
                        </Button>
                        <Button variant="ghost" size="sm">
                          <Download className="h-3 w-3" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Summary Footer */}
          <div className="mt-4 p-4 bg-muted/50 rounded-lg">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Total Advance Paid: </span>
                <span className="font-medium">₹{totalAdvanceAmount.toLocaleString()}</span>
              </div>
              <div>
                <span className="text-muted-foreground">Percentage of PO Value: </span>
                <span className="font-medium">{((totalAdvanceAmount / data.poVal) * 100).toFixed(1)}%</span>
              </div>
              <div>
                <span className="text-muted-foreground">Pending Closure: </span>
                <span className="font-medium">{openAdvances} advance(s)</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
      
      {/* Action Items */}
      {openAdvances > 0 && (
        <Card className="border-orange-200 bg-orange-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-orange-600" />
              <div>
                <p className="font-medium text-orange-800">Action Required</p>
                <p className="text-sm text-orange-700">
                  {openAdvances} advance payment(s) pending closure. Please coordinate with accounts team for bill adjustment.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}