import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Progress } from './ui/progress';
import { Separator } from './ui/separator';
import { 
  DollarSign, 
  FileText, 
  AlertTriangle, 
  CheckCircle, 
  Clock, 
  CreditCard,
  Package,
  Receipt,
  Calendar,
  TrendingUp,
  TrendingDown,
  AlertCircle
} from 'lucide-react';

interface AdvancedPODetailsProps {
  data: any;
}

export function AdvancedPODetails({ data }: AdvancedPODetailsProps) {
  if (!data) return null;

  const grnToBillLeadDays = data.billHandoverDays || 0;
  const isDelayed = grnToBillLeadDays > 7;
  const paymentOutstandingPercentage = data.paymentOutstanding ? 
    (data.paymentOutstanding / data.poVal) * 100 : 0;

  return (
    <div className="space-y-6">
      {/* PO Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Package className="h-5 w-5" />
            Purchase Order Details
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div>
              <p className="text-sm text-muted-foreground">PO Number</p>
              <p className="font-medium">{data.id}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Supplier</p>
              <p className="font-medium">{data.supplier}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">PO Value</p>
              <p className="font-medium">₹{data.poVal?.toLocaleString()}</p>
            </div>
            <div>
              <p className="text-sm text-muted-foreground">Customer/OCN</p>
              <p className="font-medium">{data.customer} / {data.ocn}</p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Advanced Details Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* GRN Details */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Receipt className="h-5 w-5" />
              GRN Details
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-muted-foreground">Total GRNs</span>
              <Badge variant="outline">{data.grnCount || 1} GRN(s)</Badge>
            </div>
            
            <div className="space-y-2">
              <p className="text-sm font-medium">GRN Numbers:</p>
              <div className="flex flex-wrap gap-2">
                {data.grnNumbers?.map((grn: string) => (
                  <Badge key={grn} variant="secondary" className="text-xs">
                    {grn}
                  </Badge>
                )) || <Badge variant="secondary">GRN5001</Badge>}
              </div>
            </div>

            <Separator />

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Material Received</span>
                <span className="text-sm font-medium">{data.materialReceivedDate || data.grnDate}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Bill Handover</span>
                <span className="text-sm font-medium">{data.billHandoverDate || 'Pending'}</span>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Billing & Payment Tracking */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Billing & Payment
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-muted-foreground">Invoice Number</span>
              <span className="text-sm font-medium">{data.invoiceNumber || 'Not Generated'}</span>
            </div>

            <div className="flex justify-between items-center">
              <span className="text-sm text-muted-foreground">Payment Status</span>
              <Badge variant={
                data.paymentStatus === 'Paid' ? 'default' :
                data.paymentStatus === 'Overdue' ? 'destructive' :
                data.paymentStatus === 'Partial' ? 'secondary' : 'outline'
              }>
                {data.paymentStatus || 'Pending'}
              </Badge>
            </div>

            {data.paymentOutstanding > 0 && (
              <div className="space-y-2">
                <div className="flex justify-between items-center">
                  <span className="text-sm text-muted-foreground">Outstanding Amount</span>
                  <span className="text-sm font-medium text-red-600">
                    ₹{data.paymentOutstanding?.toLocaleString()}
                  </span>
                </div>
                <Progress value={paymentOutstandingPercentage} className="h-2" />
                <p className="text-xs text-muted-foreground">
                  {paymentOutstandingPercentage.toFixed(1)}% of PO value outstanding
                </p>
              </div>
            )}
          </CardContent>
        </Card>

        {/* Supplier Advance Details */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <CreditCard className="h-5 w-5" />
              Supplier Advance Details
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-muted-foreground">Advance Payment</span>
              <Badge variant={data.supplierAdvancePaid ? 'default' : 'secondary'}>
                {data.supplierAdvancePaid ? 'Paid' : 'Not Paid'}
              </Badge>
            </div>

            {data.supplierAdvancePaid && (
              <>
                <div className="flex justify-between items-center">
                  <span className="text-sm text-muted-foreground">Advance Amount</span>
                  <span className="text-sm font-medium">₹{data.advanceAmount?.toLocaleString()}</span>
                </div>

                <div className="flex justify-between items-center">
                  <span className="text-sm text-muted-foreground">Bill Closing Status</span>
                  <Badge variant={data.advancePendingBillClose ? 'destructive' : 'default'}>
                    {data.advancePendingBillClose ? 'Pending' : 'Closed'}
                  </Badge>
                </div>

                {data.advancePendingBillClose && (
                  <div className="flex items-center gap-2 p-2 bg-red-50 rounded-lg border border-red-200">
                    <AlertTriangle className="h-4 w-4 text-red-600" />
                    <span className="text-sm text-red-700">Advance pending bill closure</span>
                  </div>
                )}
              </>
            )}
          </CardContent>
        </Card>

        {/* Customer Payment Terms */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <DollarSign className="h-5 w-5" />
              Customer Payment Terms
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-muted-foreground">Payment Type</span>
              <Badge variant={data.customerPaymentType === 'Credit' ? 'secondary' : 'default'}>
                {data.customerPaymentType || 'Credit'}
              </Badge>
            </div>

            {data.customerPaymentType === 'Credit' && (
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Credit Days</span>
                <span className="text-sm font-medium">{data.customerCreditDays || 30} days</span>
              </div>
            )}

            <Separator />

            <div className="space-y-2">
              <p className="text-sm font-medium">Payment Schedule</p>
              <div className="text-xs text-muted-foreground">
                Based on {data.customerPaymentType === 'Credit' ? 'credit terms' : 'advance payment'}
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Lead Time Analysis */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Clock className="h-5 w-5" />
            Lead Time Analysis
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="text-center">
              <div className="flex items-center justify-center gap-2 mb-2">
                <Calendar className="h-4 w-4 text-blue-500" />
                <span className="text-sm font-medium">GRN to Bill Handover</span>
              </div>
              <div className={`text-2xl font-bold ${isDelayed ? 'text-red-600' : 'text-green-600'}`}>
                {grnToBillLeadDays} days
              </div>
              <div className="flex items-center justify-center gap-1 mt-1">
                {isDelayed ? (
                  <>
                    <TrendingUp className="h-3 w-3 text-red-500" />
                    <span className="text-xs text-red-600">Delayed (&gt;7 days)</span>
                  </>
                ) : (
                  <>
                    <CheckCircle className="h-3 w-3 text-green-500" />
                    <span className="text-xs text-green-600">Within target</span>
                  </>
                )}
              </div>
            </div>

            <div className="text-center">
              <div className="flex items-center justify-center gap-2 mb-2">
                <Package className="h-4 w-4 text-purple-500" />
                <span className="text-sm font-medium">Material Lead Time</span>
              </div>
              <div className="text-2xl font-bold text-blue-600">
                {data.leadDays || 'N/A'} days
              </div>
              <div className="flex items-center justify-center gap-1 mt-1">
                <span className="text-xs text-muted-foreground">
                  Target: {data.targetLeadDays} days
                </span>
              </div>
            </div>

            <div className="text-center">
              <div className="flex items-center justify-center gap-2 mb-2">
                <AlertCircle className="h-4 w-4 text-orange-500" />
                <span className="text-sm font-medium">Overall Status</span>
              </div>
              <div className="text-2xl font-bold">
                <Badge variant={
                  isDelayed || data.paymentStatus === 'Overdue' ? 'destructive' :
                  data.paymentStatus === 'Paid' ? 'default' : 'secondary'
                } className="text-sm">
                  {isDelayed || data.paymentStatus === 'Overdue' ? 'At Risk' :
                   data.paymentStatus === 'Paid' ? 'Completed' : 'In Progress'}
                </Badge>
              </div>
            </div>
          </div>

          {isDelayed && (
            <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded-lg">
              <div className="flex items-center gap-2">
                <AlertTriangle className="h-4 w-4 text-red-600" />
                <span className="text-sm font-medium text-red-800">Action Required</span>
              </div>
              <p className="text-sm text-red-700 mt-1">
                Bill handover is delayed beyond the 7-day target. Please expedite processing to accounts department.
              </p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Process Timeline */}
      <Card>
        <CardHeader>
          <CardTitle>Process Timeline</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div className="flex items-center gap-3">
              <div className="w-3 h-3 bg-green-500 rounded-full"></div>
              <div className="flex-1">
                <p className="text-sm font-medium">PO Created</p>
                <p className="text-xs text-muted-foreground">{data.poDate}</p>
              </div>
            </div>

            <div className="flex items-center gap-3">
              <div className={`w-3 h-3 rounded-full ${data.grnDate ? 'bg-green-500' : 'bg-gray-300'}`}></div>
              <div className="flex-1">
                <p className="text-sm font-medium">Material Received & GRN Created</p>
                <p className="text-xs text-muted-foreground">
                  {data.materialReceivedDate || data.grnDate || 'Pending'}
                </p>
              </div>
            </div>

            <div className="flex items-center gap-3">
              <div className={`w-3 h-3 rounded-full ${data.invoiceNumber ? 'bg-green-500' : 'bg-gray-300'}`}></div>
              <div className="flex-1">
                <p className="text-sm font-medium">Invoice Generated</p>
                <p className="text-xs text-muted-foreground">
                  {data.invoiceNumber ? 'Completed' : 'Pending'}
                </p>
              </div>
            </div>

            <div className="flex items-center gap-3">
              <div className={`w-3 h-3 rounded-full ${
                data.billHandoverDate && !isDelayed ? 'bg-green-500' : 
                isDelayed ? 'bg-red-500' : 'bg-gray-300'
              }`}></div>
              <div className="flex-1">
                <p className="text-sm font-medium">Bill Handover to Accounts</p>
                <p className="text-xs text-muted-foreground">
                  {data.billHandoverDate || 'Pending'} 
                  {isDelayed && ' (Delayed)'}
                </p>
              </div>
            </div>

            <div className="flex items-center gap-3">
              <div className={`w-3 h-3 rounded-full ${
                data.paymentStatus === 'Paid' ? 'bg-green-500' : 
                data.paymentStatus === 'Overdue' ? 'bg-red-500' : 'bg-gray-300'
              }`}></div>
              <div className="flex-1">
                <p className="text-sm font-medium">Payment Completed</p>
                <p className="text-xs text-muted-foreground">
                  Status: {data.paymentStatus || 'Pending'}
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}