import React from 'react';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  LineChart,
  Line,
  PieChart,
  Pie,
  Cell,
  ComposedChart,
  Area,
  AreaChart,
  TreeMap
} from 'recharts';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { TrendingUp, TrendingDown, Users, ShoppingBag, Star, Clock } from 'lucide-react';

const CUSTOMER_COLORS = {
  'Takko': '#2563eb',
  'Primark': '#dc2626',
  'Reliance': '#16a34a',
  'H&M': '#ca8a04',
  'Zara': '#9333ea',
  'Walmart': '#0891b2',
  'Target': '#c2410c',
  'Marks & Spencer': '#be185d'
};

const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length) {
    return (
      <div className="bg-white p-4 border rounded-lg shadow-lg">
        <p className="font-medium mb-2">{`${label}`}</p>
        {payload.map((entry: any, index: number) => (
          <div key={index} className="flex items-center gap-2">
            <div 
              className="w-3 h-3 rounded-full" 
              style={{ backgroundColor: entry.color }}
            ></div>
            <span className="text-sm">
              {entry.name}: {typeof entry.value === 'number' ? 
                (entry.name.includes('%') ? entry.value.toFixed(1) + '%' : 
                 entry.name.includes('₹') ? '₹' + entry.value.toLocaleString() : 
                 entry.value.toLocaleString()) : entry.value}
            </span>
          </div>
        ))}
      </div>
    );
  }
  return null;
};

interface CustomerChartProps {
  data: any[];
  title: string;
  onClick?: (data: any) => void;
  height?: number;
}

export function CustomerValueChart({ data, title, onClick, height = 400 }: CustomerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <ShoppingBag className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <ComposedChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 60 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="customer" 
              angle={-45}
              textAnchor="end"
              height={80}
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              yAxisId="left"
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <YAxis 
              yAxisId="right" 
              orientation="right"
              fontSize={12}
              stroke="#64748b"
            />
            <Tooltip content={<CustomTooltip />} />
            <Bar 
              yAxisId="left"
              dataKey="totalValue" 
              name="Total Purchase Value"
              fill="#2563eb"
              fillOpacity={0.8}
              radius={[4, 4, 0, 0]}
            />
            <Line 
              yAxisId="right"
              type="monotone" 
              dataKey="orders" 
              name="Number of Orders"
              stroke="#dc2626"
              strokeWidth={3}
              dot={{ fill: "#dc2626", strokeWidth: 2, r: 4 }}
            />
          </ComposedChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function CustomerPerformanceMatrix({ data, title, onClick, height = 400 }: CustomerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Star className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <BarChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 60 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="customer" 
              angle={-45}
              textAnchor="end"
              height={80}
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              domain={[0, 100]}
            />
            <Tooltip content={<CustomTooltip />} />
            <Bar 
              dataKey="qualityScore" 
              name="Quality Score (%)"
              fill="#10b981"
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              dataKey="deliveryScore" 
              name="On-Time Delivery (%)"
              fill="#f59e0b"
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              dataKey="costEfficiency" 
              name="Cost Efficiency (%)"
              fill="#8b5cf6"
              radius={[4, 4, 0, 0]}
            />
          </BarChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function CustomerTrendAnalysis({ data, title, onClick, height = 350 }: CustomerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <TrendingUp className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <LineChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="month" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <Tooltip content={<CustomTooltip />} />
            {Object.keys(CUSTOMER_COLORS).slice(0, 4).map((customer) => (
              <Line 
                key={customer}
                type="monotone" 
                dataKey={customer.toLowerCase()} 
                name={customer}
                stroke={CUSTOMER_COLORS[customer]}
                strokeWidth={2}
                dot={{ fill: CUSTOMER_COLORS[customer], strokeWidth: 2, r: 4 }}
              />
            ))}
          </LineChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function CustomerDistribution({ data, title, onClick, height = 350 }: CustomerChartProps) {
  const renderCustomLabel = ({ cx, cy, midAngle, innerRadius, outerRadius, percent, name }: any) => {
    const RADIAN = Math.PI / 180;
    const radius = innerRadius + (outerRadius - innerRadius) * 0.5;
    const x = cx + radius * Math.cos(-midAngle * RADIAN);
    const y = cy + radius * Math.sin(-midAngle * RADIAN);

    return (
      <text 
        x={x} 
        y={y} 
        fill="white" 
        textAnchor={x > cx ? 'start' : 'end'} 
        dominantBaseline="central"
        fontSize={11}
        fontWeight={600}
      >
        {`${(percent * 100).toFixed(0)}%`}
      </text>
    );
  };

  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Users className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <PieChart>
            <Pie
              data={data}
              cx="50%"
              cy="50%"
              labelLine={false}
              label={renderCustomLabel}
              outerRadius={120}
              fill="#8884d8"
              dataKey="value"
              onClick={onClick}
            >
              {data.map((entry: any, index: number) => (
                <Cell 
                  key={`cell-${index}`} 
                  fill={CUSTOMER_COLORS[entry.name] || `#${Math.floor(Math.random()*16777215).toString(16)}`} 
                />
              ))}
            </Pie>
            <Tooltip content={<CustomTooltip />} />
          </PieChart>
        </ResponsiveContainer>
        <div className="mt-4 grid grid-cols-2 gap-2">
          {data.slice(0, 6).map((customer: any, index: number) => (
            <div key={customer.name} className="flex items-center gap-2">
              <div 
                className="w-3 h-3 rounded-full" 
                style={{ backgroundColor: CUSTOMER_COLORS[customer.name] || '#8884d8' }}
              ></div>
              <span className="text-sm truncate">{customer.name}</span>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}

export function CustomerLeadTimeAnalysis({ data, title, onClick, height = 350 }: CustomerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Clock className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <BarChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 60 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="customer" 
              angle={-45}
              textAnchor="end"
              height={80}
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              label={{ value: 'Days', angle: -90, position: 'insideLeft' }}
            />
            <Tooltip content={<CustomTooltip />} />
            <Bar 
              dataKey="targetLeadTime" 
              name="Target Lead Time (Days)"
              fill="#94a3b8"
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              dataKey="actualLeadTime" 
              name="Actual Lead Time (Days)"
              fill="#2563eb"
              radius={[4, 4, 0, 0]}
            />
          </BarChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function CustomerKPICards({ customerData }: { customerData: any[] }) {
  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
      {customerData.slice(0, 4).map((customer) => (
        <Card key={customer.name} className="relative overflow-hidden hover:shadow-md transition-shadow">
          <div 
            className="absolute top-0 left-0 w-full h-1"
            style={{ backgroundColor: CUSTOMER_COLORS[customer.name] }}
          ></div>
          <CardHeader className="pb-3">
            <CardTitle className="flex items-center justify-between">
              <div>
                <h3 className="font-semibold text-sm">{customer.name}</h3>
                <p className="text-xs text-muted-foreground">Customer</p>
              </div>
              <Badge 
                variant={customer.overallScore >= 90 ? 'default' : 
                        customer.overallScore >= 75 ? 'secondary' : 'destructive'}
                className="text-xs"
              >
                {customer.overallScore}%
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            <div className="grid grid-cols-2 gap-2">
              <div className="text-center p-2 bg-blue-50 rounded">
                <div className="text-lg font-bold text-blue-600">{customer.orders}</div>
                <p className="text-xs text-blue-700">Orders</p>
              </div>
              <div className="text-center p-2 bg-green-50 rounded">
                <div className="text-lg font-bold text-green-600">₹{(customer.totalValue / 1000).toFixed(0)}K</div>
                <p className="text-xs text-green-700">Value</p>
              </div>
            </div>

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <span className="text-xs text-muted-foreground">Quality</span>
                <Badge variant="outline" className="text-xs">
                  {customer.qualityScore}%
                </Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-xs text-muted-foreground">On-Time</span>
                <Badge variant="outline" className="text-xs">
                  {customer.onTimeDelivery}%
                </Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-xs text-muted-foreground">Lead Time</span>
                <Badge variant="outline" className="text-xs">
                  {customer.avgLeadTime}d
                </Badge>
              </div>
            </div>

            <div className="pt-2 border-t">
              <div className="flex justify-between items-center">
                <span className="text-xs font-medium">Growth</span>
                <div className="flex items-center gap-1">
                  {customer.growth >= 0 ? (
                    <TrendingUp className="h-3 w-3 text-green-500" />
                  ) : (
                    <TrendingDown className="h-3 w-3 text-red-500" />
                  )}
                  <span className={`text-xs ${customer.growth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {customer.growth >= 0 ? '+' : ''}{customer.growth}%
                  </span>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}