import React, { useState } from 'react';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from './ui/table';
import { Input } from './ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Search, Filter, Download, Eye, MoreHorizontal } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from './ui/dropdown-menu';

interface PurchaseOrder {
  id: string;
  poDate: string;
  poType: string;
  supplier: string;
  itemGroup: string;
  dcn: string;
  uom: string;
  poQty: number;
  poRate: number;
  poVal: number;
  cancelQty: number;
  budgetRate: number;
  leadDays: string;
  grnDate: string;
  division: string;
}

interface DetailedDataGridProps {
  data: PurchaseOrder[];
  title: string;
  onRowClick?: (row: PurchaseOrder) => void;
}

export function DetailedDataGrid({ data, title, onRowClick }: DetailedDataGridProps) {
  const [searchTerm, setSearchTerm] = useState('');
  const [sortField, setSortField] = useState<keyof PurchaseOrder>('poDate');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('desc');
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage, setItemsPerPage] = useState(25);
  const [filterDivision, setFilterDivision] = useState('All');
  const [filterPoType, setFilterPoType] = useState('All');

  // Filter data
  const filteredData = data.filter(item => {
    const matchesSearch = Object.values(item).some(value =>
      value.toString().toLowerCase().includes(searchTerm.toLowerCase())
    );
    const matchesDivision = filterDivision === 'All' || item.division === filterDivision;
    const matchesPoType = filterPoType === 'All' || item.poType === filterPoType;
    
    return matchesSearch && matchesDivision && matchesPoType;
  });

  // Sort data
  const sortedData = [...filteredData].sort((a, b) => {
    const aVal = a[sortField];
    const bVal = b[sortField];
    
    if (typeof aVal === 'number' && typeof bVal === 'number') {
      return sortDirection === 'asc' ? aVal - bVal : bVal - aVal;
    }
    
    const comparison = aVal.toString().localeCompare(bVal.toString());
    return sortDirection === 'asc' ? comparison : -comparison;
  });

  // Paginate data
  const startIndex = (currentPage - 1) * itemsPerPage;
  const paginatedData = sortedData.slice(startIndex, startIndex + itemsPerPage);
  const totalPages = Math.ceil(sortedData.length / itemsPerPage);

  const handleSort = (field: keyof PurchaseOrder) => {
    if (sortField === field) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
    } else {
      setSortField(field);
      setSortDirection('asc');
    }
  };

  const getStatusBadge = (item: PurchaseOrder) => {
    if (item.cancelQty > 0) {
      return <Badge variant="destructive">Partially Cancelled</Badge>;
    }
    if (item.grnDate) {
      return <Badge variant="default" className="bg-green-100 text-green-800">Delivered</Badge>;
    }
    return <Badge variant="secondary">Pending</Badge>;
  };

  const getVarianceBadge = (poRate: number, budgetRate: number) => {
    if (!budgetRate) return null;
    const variance = ((poRate - budgetRate) / budgetRate * 100);
    if (Math.abs(variance) < 1) return null;
    
    return (
      <Badge variant={variance > 0 ? "destructive" : "default"} className={variance < 0 ? "bg-green-100 text-green-800" : ""}>
        {variance > 0 ? '+' : ''}{variance.toFixed(1)}%
      </Badge>
    );
  };

  const divisions = ['All', ...Array.from(new Set(data.map(item => item.division)))];
  const poTypes = ['All', ...Array.from(new Set(data.map(item => item.poType)))];

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>{title}</CardTitle>
          <div className="flex gap-2">
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
          </div>
        </div>
        
        {/* Filters and Search */}
        <div className="flex flex-wrap gap-4 mt-4">
          <div className="relative flex-1 min-w-[200px]">
            <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search all fields..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10"
            />
          </div>
          <Select value={filterDivision} onValueChange={setFilterDivision}>
            <SelectTrigger className="w-40">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {divisions.map(div => (
                <SelectItem key={div} value={div}>{div}</SelectItem>
              ))}
            </SelectContent>
          </Select>
          <Select value={filterPoType} onValueChange={setFilterPoType}>
            <SelectTrigger className="w-32">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {poTypes.map(type => (
                <SelectItem key={type} value={type}>{type}</SelectItem>
              ))}
            </SelectContent>
          </Select>
          <Select value={itemsPerPage.toString()} onValueChange={(value) => setItemsPerPage(Number(value))}>
            <SelectTrigger className="w-20">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="25">25</SelectItem>
              <SelectItem value="50">50</SelectItem>
              <SelectItem value="100">100</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </CardHeader>
      
      <CardContent>
        <div className="rounded-md border">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead 
                  className="cursor-pointer hover:bg-muted/50"
                  onClick={() => handleSort('id')}
                >
                  PO Number {sortField === 'id' && (sortDirection === 'asc' ? '↑' : '↓')}
                </TableHead>
                <TableHead 
                  className="cursor-pointer hover:bg-muted/50"
                  onClick={() => handleSort('poDate')}
                >
                  Date {sortField === 'poDate' && (sortDirection === 'asc' ? '↑' : '↓')}
                </TableHead>
                <TableHead>Type</TableHead>
                <TableHead 
                  className="cursor-pointer hover:bg-muted/50"
                  onClick={() => handleSort('supplier')}
                >
                  Supplier {sortField === 'supplier' && (sortDirection === 'asc' ? '↑' : '↓')}
                </TableHead>
                <TableHead>Item Group</TableHead>
                <TableHead>Division</TableHead>
                <TableHead className="text-right">Quantity</TableHead>
                <TableHead className="text-right">Rate</TableHead>
                <TableHead className="text-right">Value</TableHead>
                <TableHead>Variance</TableHead>
                <TableHead>Lead Days</TableHead>
                <TableHead>Status</TableHead>
                <TableHead className="w-[50px]"></TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {paginatedData.map((item) => (
                <TableRow 
                  key={item.id}
                  className={onRowClick ? "cursor-pointer hover:bg-muted/50" : ""}
                  onClick={() => onRowClick?.(item)}
                >
                  <TableCell className="font-medium">{item.id}</TableCell>
                  <TableCell>{item.poDate}</TableCell>
                  <TableCell>
                    <Badge variant={item.poType === 'Import' ? 'default' : 'secondary'}>
                      {item.poType}
                    </Badge>
                  </TableCell>
                  <TableCell>{item.supplier}</TableCell>
                  <TableCell>{item.itemGroup}</TableCell>
                  <TableCell>
                    <Badge variant="outline">{item.division}</Badge>
                  </TableCell>
                  <TableCell className="text-right">
                    {item.poQty.toLocaleString()} {item.uom}
                  </TableCell>
                  <TableCell className="text-right">₹{item.poRate.toLocaleString()}</TableCell>
                  <TableCell className="text-right font-medium">₹{item.poVal.toLocaleString()}</TableCell>
                  <TableCell>{getVarianceBadge(item.poRate, item.budgetRate)}</TableCell>
                  <TableCell>{item.leadDays || 'N/A'}</TableCell>
                  <TableCell>{getStatusBadge(item)}</TableCell>
                  <TableCell>
                    <DropdownMenu>
                      <DropdownMenuTrigger asChild>
                        <Button variant="ghost" className="h-8 w-8 p-0">
                          <MoreHorizontal className="h-4 w-4" />
                        </Button>
                      </DropdownMenuTrigger>
                      <DropdownMenuContent align="end">
                        <DropdownMenuItem onClick={() => onRowClick?.(item)}>
                          <Eye className="mr-2 h-4 w-4" />
                          View Details
                        </DropdownMenuItem>
                      </DropdownMenuContent>
                    </DropdownMenu>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </div>

        {/* Pagination */}
        <div className="flex items-center justify-between space-x-2 py-4">
          <div className="text-sm text-muted-foreground">
            Showing {startIndex + 1} to {Math.min(startIndex + itemsPerPage, sortedData.length)} of {sortedData.length} entries
            {filteredData.length !== data.length && ` (filtered from ${data.length} total entries)`}
          </div>
          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
              disabled={currentPage === 1}
            >
              Previous
            </Button>
            <div className="flex items-center gap-1">
              {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                const pageNum = Math.max(1, Math.min(totalPages - 4, currentPage - 2)) + i;
                return (
                  <Button
                    key={pageNum}
                    variant={currentPage === pageNum ? "default" : "outline"}
                    size="sm"
                    onClick={() => setCurrentPage(pageNum)}
                    className="w-10"
                  >
                    {pageNum}
                  </Button>
                );
              })}
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
              disabled={currentPage === totalPages}
            >
              Next
            </Button>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}