import { useEffect, useState } from 'react';
import { TrendingUp, TrendingDown, Package, Users, Star, Clock, DollarSign, ShoppingCart, AlertTriangle, CheckCircle, BarChart3, Table, PieChart as LucidePieChart, LineChart as LucideLineChart, Target, UserCheck, ShoppingBag, FileText } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Button } from './ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from './ui/tabs';
import { DetailedDataGrid } from './DetailedDataGrid';
import { DrilldownModal } from './DrilldownModal';
import { AdvancedPODetails } from './AdvancedPODetails';
import { AdvanceDetailsBreakup } from './AdvanceDetailsBreakup';
import { GRNToBillBreakup } from './GRNToBillBreakup';
import { DelayedBillsBreakup } from './DelayedBillsBreakup';
import { CostSavingsBreakup } from './CostSavingsBreakup';
import { ProfessionalBarChart, ProfessionalLineChart, ProfessionalPieChart, ProfessionalComposedChart, ProfessionalAreaChart, SupplierPerformanceChart } from './ProfessionalCharts';
import { ManagerKPICards, ManagerPerformanceComparison, ManagerRadarChart, TargetVsActualChart, WeeklyPerformanceChart } from './ManagerPerformanceCharts';
import { CustomerKPICards, CustomerValueChart, CustomerDistribution, CustomerPerformanceMatrix, CustomerLeadTimeAnalysis, CustomerTrendAnalysis } from './CustomerAnalysisCharts';
import { DivisionTargetVsActualChart } from './DivisionTargetVsActualChart';

// Mock data based on the provided Excel structure
const mockPurchaseData = [
  {
    id: 'PO113214',
    poDate: '01/08/2025',
    poType: 'Local',
    supplier: 'Laxmi Labels',
    itemGroup: 'Label',
    dcn: 'DCN4123',
    uom: 'Nos',
    poQty: 100,
    poRate: 0.35,
    poVal: 35,
    cancelQty: 0,
    budgetRate: 0.4,
    leadDays: '',
    grnDate: '',
    division: 'Garments',
    purchaseManager: 'Mr.Britto',
    customer: 'Takko',
    ocn: 'OCN2025001',
    targetLeadDays: 5,
    targetQuality: 95,
    actualQuality: 100
  },
  {
    id: 'PO113215',
    poDate: '02/08/2025',
    poType: 'Local',
    supplier: 'Madras Coats',
    itemGroup: 'Thread',
    dcn: 'DCN4127',
    uom: 'Cone',
    poQty: 30,
    poRate: 65,
    poVal: 1950,
    cancelQty: 5,
    budgetRate: 65,
    leadDays: '7',
    grnDate: '09/08/2025',
    division: 'Garments',
    purchaseManager: 'Mr.Udhaya',
    customer: 'Primark',
    ocn: 'OCN2025002',
    targetLeadDays: 10,
    targetQuality: 90,
    actualQuality: 83
  },
  {
    id: 'PO113216',
    poDate: '03/08/2025',
    poType: 'Import',
    supplier: 'Xiangao',
    itemGroup: 'Hanger',
    dcn: 'DCN4124',
    uom: 'Nos',
    poQty: 100,
    poRate: 0.35,
    poVal: 35,
    cancelQty: 0,
    budgetRate: 0.35,
    leadDays: '15',
    grnDate: '18/08/2025',
    division: 'Garments',
    purchaseManager: 'Mr.Britto',
    customer: 'Reliance',
    ocn: 'OCN2025003',
    targetLeadDays: 20,
    targetQuality: 95,
    actualQuality: 100
  },
  {
    id: 'PO113217',
    poDate: '04/08/2025',
    poType: 'Local',
    supplier: 'Raj Stationaries',
    itemGroup: 'General',
    dcn: 'DCN4130',
    uom: 'Roll',
    poQty: 15,
    poRate: 25,
    poVal: 375,
    cancelQty: 4,
    budgetRate: 25,
    leadDays: '3',
    grnDate: '07/08/2025',
    division: 'Spinning',
    purchaseManager: 'Mr.Udhaya',
    customer: 'Takko',
    ocn: 'OCN2025004',
    targetLeadDays: 5,
    targetQuality: 90,
    actualQuality: 73
  },
  {
    id: 'PO112121',
    poDate: '05/08/2025',
    poType: 'Local',
    supplier: 'Sri Laxmi Spe mills',
    itemGroup: 'Yarn',
    dcn: 'DCN4135',
    uom: 'Kgs',
    poQty: 500.18,
    poRate: 285,
    poVal: 142551.3,
    cancelQty: 0,
    budgetRate: 290,
    leadDays: '10',
    grnDate: '15/08/2025',
    division: 'Spinning',
    purchaseManager: 'Mr.Britto',
    customer: 'Primark',
    ocn: 'OCN2025005',
    targetLeadDays: 12,
    targetQuality: 92,
    actualQuality: 100
  }
];

// Generate additional mock data for comprehensive dashboard
const generateMockData = () => {
  const divisionConfig = {
    'Garments': {
      suppliers: [
        'Laxmi Labels', 'Madras Coats', 'Kumar Textiles', 'Vishnu Fabrics', 'Metro Labels',
        'Chennai Threads', 'Bangalore Buttons', 'Mumbai Zippers', 'Delhi Trims', 'Kolkata Cotton',
        'Hyderabad Hangers', 'Pune Packaging', 'Coimbatore Cottons', 'Tirupur Textiles', 'Erode Exports',
        'Salem Silks', 'Karur Knits', 'Dindigul Dyes', 'Madurai Mills', 'Thoothukudi Threads',
        'Vellore Velvets', 'Hosur Haberdashery', 'Krishnagiri Knits', 'Dharmapuri Dyes'
      ],
      itemGroups: [
        'Label', 'Thread', 'Button', 'Zipper', 'Hanger', 'Packaging',
        'Elastic', 'Velcro', 'Interlining', 'Shoulder Pad', 'Bias Tape', 'Piping',
        'Snap Button', 'Hook & Eye', 'D-Ring', 'Buckle', 'Grommet', 'Eyelet',
        'Care Label', 'Size Label', 'Brand Label', 'Swing Tag', 'Poly Bag', 'Carton Box'
      ],
      uoms: ['Nos', 'Cone', 'Pieces', 'Meters', 'Yards', 'Dozen', 'Gross', 'Set'],
      customers: ['Takko', 'Reliance', 'Primark', 'H&M', 'Zara', 'Next', 'Marks & Spencer', 'Target', 'Walmart']
    },
    'Spinning': {
      suppliers: [
        'Sri Laxmi Cotton Mills', 'Karnataka Cotton', 'Tamil Nadu Cotton Corp', 'Rajasthan Cotton', 'Gujarat Cotton Ltd',
        'Andhra Pradesh Cotton', 'Maharashtra Cotton Mills', 'Punjab Cotton Corp', 'Haryana Cotton', 'Uttar Pradesh Cotton',
        'Madhya Pradesh Mills', 'Odisha Cotton', 'West Bengal Cotton', 'Telangana Cotton', 'Kerala Cotton Mills',
        'Assam Cotton Corp', 'Jharkhand Mills', 'Chhattisgarh Cotton', 'Uttarakhand Mills', 'Himachal Cotton',
        'Bihar Cotton Mills', 'Goa Cotton Corp', 'Nagaland Cotton', 'Manipur Mills', 'Tripura Cotton'
      ],
      itemGroups: [
        'Cotton', 'Cotton Bales', 'Raw Cotton', 'Combed Cotton', 'Carded Cotton',
        'Cotton Yarn', 'Polyester Yarn', 'Viscose Yarn', 'Blended Yarn', 'Organic Cotton',
        'BCI Cotton', 'Supima Cotton', 'Pima Cotton', 'Egyptian Cotton', 'Cotton Waste',
        'Cotton Linters', 'Cotton Seeds', 'Cotton Fiber', 'Recycled Cotton', 'Open End Yarn'
      ],
      uoms: ['Kgs', 'Tonnes', 'Bales', 'Pounds', 'Quintals', 'MT', 'Cones', 'Hanks'],
      customers: [
        'Arvind Mills', 'Vardhman Textiles', 'Welspun India', 'Raymond Ltd', 'Trident Group',
        'Indo Count', 'Bombay Dyeing', 'Himatsingka Seide', 'Alok Industries', 'JCT Mills',
        'Nahar Industrial', 'Loyal Textiles', 'Banswara Syntex', 'Sutlej Textiles', 'RSWM Ltd'
      ]
    },
    'Socks': {
      suppliers: [
        'Ganesh Yarn Mills', 'Balaji Textiles', 'Shree Yarn Corp', 'Modern Knits', 'Elite Yarn',
        'Premium Hosiery', 'Comfort Knits', 'Fashion Feet', 'Ankle Arts', 'Toe Touch',
        'Sock Supreme', 'Knit King', 'Hosiery Hub', 'Comfort Zone', 'Feet First',
        'Sock Star', 'Ankle Avenue', 'Toe Trends', 'Knit Nest', 'Sock Sense'
      ],
      itemGroups: [
        'Yarn', 'Elastic', 'Labels', 'Trims', 'Toe Seaming Thread',
        'Heel Yarn', 'Sole Yarn', 'Cuff Elastic', 'Arch Support', 'Cushioning',
        'Moisture Wicking', 'Anti-Bacterial', 'Bamboo Fiber', 'Merino Wool', 'Cotton Blend'
      ],
      uoms: ['Kgs', 'Meters', 'Nos', 'Pieces', 'Pairs', 'Dozen', 'Cones', 'Bobbins'],
      customers: ['Decathlon', 'Nike', 'Adidas', 'Puma', 'Reebok', 'Under Armour', 'New Balance']
    },
    'Gloves': {
      suppliers: [
        'Perfect Fabrics', 'Supreme Textiles', 'Royal Fabric Mills', 'Elite Trims', 'Quality Fabrics',
        'Grip Masters', 'Hand Protectors', 'Safety First', 'Work Wear', 'Industrial Gloves',
        'Leather Craft', 'Synthetic Solutions', 'Protective Gear', 'Hand Guard', 'Safety Zone'
      ],
      itemGroups: [
        'Fabric', 'Lining Fabric', 'Trims', 'Elastic', 'Thread',
        'Leather', 'Synthetic Leather', 'Rubber Coating', 'Nitrile', 'Latex',
        'PVC Coating', 'Palm Coating', 'Grip Dots', 'Velcro Strap', 'Cuff Elastic'
      ],
      uoms: ['Yard', 'Mtr', 'Pieces', 'Nos', 'Pairs', 'Dozen', 'Sq Ft', 'Kg'],
      customers: ['Decathlon', 'Stanley', 'DeWalt', 'Makita', 'Bosch', 'Milwaukee', 'Ryobi']
    },
    'Woven': {
      suppliers: [
        'Weaving Masters', 'Loom Textiles', 'Fabric World', 'Textile Dynamics', 'Woven Excellence',
        'Shuttle Craft', 'Warp & Weft', 'Fabric Focus', 'Textile Trends', 'Weave Wonder',
        'Loom Logic', 'Fabric Fantasy', 'Textile Touch', 'Weave World', 'Fabric Flow'
      ],
      itemGroups: [
        'Fabric', 'Lining', 'Interfacing', 'Trims', 'Buttons',
        'Woven Labels', 'Jacquard Fabric', 'Dobby Fabric', 'Plain Weave', 'Twill Weave',
        'Satin Weave', 'Canvas', 'Denim', 'Corduroy', 'Flannel'
      ],
      uoms: ['Yard', 'Mtr', 'Pieces', 'Nos', 'Rolls', 'Sq Mtr', 'Linear Mtr'],
      customers: ['Decathlon', 'IKEA', 'H&M Home', 'Zara Home', 'Next Home', 'Target Home']
    },
    'Headwear': {
      suppliers: [
        'Cap Fabrics Ltd', 'Headwear Materials', 'Fashion Fabrics', 'Specialty Trims', 'Cap Components',
        'Hat Haven', 'Crown Crafts', 'Peak Performance', 'Brim Builders', 'Cap Creators',
        'Head Hunters', 'Crown Components', 'Peak Perfection', 'Brim & Beyond', 'Hat Hardware'
      ],
      itemGroups: [
        'Fabric', 'Peak Board', 'Buckle', 'Trims', 'Embroidery Thread',
        'Mesh Fabric', 'Sweatband', 'Eyelets', 'Closure Strap', 'Visor',
        'Crown Lining', 'Underbrim', 'Top Button', 'Size Adjustment', 'Snap Closure'
      ],
      uoms: ['Yard', 'Mtr', 'Pieces', 'Nos', 'Dozen', 'Set', 'Pairs'],
      customers: ['Decathlon', 'Adidas', 'Nike', 'Puma', 'New Era', 'Under Armour', 'Reebok', 'Columbia']
    }
  };

  const divisions = Object.keys(divisionConfig);
  const purchaseManagers = ['Mr.Britto', 'Mr.Udhaya'];
  
  // Helper function to generate valid dates
  const generateValidDate = (year: number, month: number, day: number): Date => {
    // Ensure valid day for the month
    const daysInMonth = new Date(year, month + 1, 0).getDate();
    const validDay = Math.min(day, daysInMonth);
    return new Date(year, month, validDay);
  };

  // Helper function to format date as DD/MM/YYYY
  const formatDate = (date: Date): string => {
    const day = date.getDate().toString().padStart(2, '0');
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const year = date.getFullYear();
    return `${day}/${month}/${year}`;
  };

  // Helper function to add days to a date
  const addDays = (date: Date, days: number): Date => {
    const result = new Date(date);
    result.setDate(result.getDate() + days);
    return result;
  };
  
  const additionalData = [];
  // Increased from 300 to 2000 for much more comprehensive data
  for (let i = 0; i < 2000; i++) {
    const division = divisions[Math.floor(Math.random() * divisions.length)];
    const config = divisionConfig[division];
    
    const supplier = config.suppliers[Math.floor(Math.random() * config.suppliers.length)];
    const itemGroup = config.itemGroups[Math.floor(Math.random() * config.itemGroups.length)];
    const uom = config.uoms[Math.floor(Math.random() * config.uoms.length)];
    const customer = config.customers[Math.floor(Math.random() * config.customers.length)];
    const manager = purchaseManagers[Math.floor(Math.random() * purchaseManagers.length)];
    
    // Generate realistic quantities and rates based on division and item type
    let poQty, poRate;
    if (division === 'Spinning') {
      if (itemGroup.includes('Cotton') || itemGroup.includes('Yarn')) {
        poQty = Math.floor(Math.random() * 10000) + 1000; // Large quantities for cotton/yarn
        poRate = Math.floor(Math.random() * 100) + 80; // Higher rates per kg
      } else {
        poQty = Math.floor(Math.random() * 5000) + 500;
        poRate = Math.floor(Math.random() * 150) + 50;
      }
    } else if (division === 'Garments') {
      if (itemGroup.includes('Thread') || itemGroup.includes('Yarn')) {
        poQty = Math.floor(Math.random() * 500) + 50;
        poRate = Math.floor(Math.random() * 200) + 30;
      } else if (itemGroup.includes('Button') || itemGroup.includes('Label')) {
        poQty = Math.floor(Math.random() * 10000) + 1000;
        poRate = Math.random() * 5 + 0.1; // Very low rates for small items
      } else {
        poQty = Math.floor(Math.random() * 2000) + 100;
        poRate = Math.floor(Math.random() * 50) + 5;
      }
    } else if (['Gloves', 'Woven', 'Headwear'].includes(division)) {
      if (itemGroup.includes('Fabric')) {
        poQty = Math.floor(Math.random() * 3000) + 200;
        poRate = Math.floor(Math.random() * 300) + 50;
      } else {
        poQty = Math.floor(Math.random() * 1000) + 50;
        poRate = Math.floor(Math.random() * 100) + 10;
      }
    } else if (division === 'Socks') {
      if (itemGroup.includes('Yarn')) {
        poQty = Math.floor(Math.random() * 2000) + 200;
        poRate = Math.floor(Math.random() * 250) + 100;
      } else {
        poQty = Math.floor(Math.random() * 5000) + 500;
        poRate = Math.floor(Math.random() * 50) + 5;
      }
    } else {
      poQty = Math.floor(Math.random() * 1000) + 10;
      poRate = Math.floor(Math.random() * 100) + 5;
    }
    
    const targetLeadDays = Math.floor(Math.random() * 25) + 5;
    const actualLeadDays = Math.floor(Math.random() * 35) + 1;
    const targetQuality = 85 + Math.floor(Math.random() * 15);
    const actualQuality = 70 + Math.floor(Math.random() * 30);
    
    // Advanced PO details with more variety
    const grnCount = Math.floor(Math.random() * 4) + 1; // 1-4 GRNs per PO
    const supplierAdvancePaid = Math.random() > 0.65; // 35% chance of advance payment
    const customerCreditBased = Math.random() > 0.3; // 70% chance of credit payment
    
    // Generate valid dates for 2025
    const currentYear = 2025;
    const poMonth = Math.floor(Math.random() * 12); // 0-11
    const poDay = Math.floor(Math.random() * 28) + 1; // 1-28 to avoid invalid dates
    const poDate = generateValidDate(currentYear, poMonth, poDay);
    
    const materialReceivedDate = addDays(poDate, actualLeadDays);
    const billHandoverDays = Math.floor(Math.random() * 20) + 1; // 1-20 days for more variation
    const billHandoverDate = addDays(materialReceivedDate, billHandoverDays);
    
    // More realistic payment status distribution
    const paymentStatuses = ['Paid', 'Pending', 'Overdue', 'Partial'];
    const paymentWeights = [0.6, 0.25, 0.1, 0.05]; // 60% paid, 25% pending, 10% overdue, 5% partial
    let paymentStatus = 'Paid';
    const rand = Math.random();
    let cumulative = 0;
    for (let j = 0; j < paymentStatuses.length; j++) {
      cumulative += paymentWeights[j];
      if (rand <= cumulative) {
        paymentStatus = paymentStatuses[j];
        break;
      }
    }
    
    // More realistic budget variance
    const budgetVariance = (Math.random() - 0.5) * 20; // -10% to +10% variance
    const budgetRate = poRate * (1 + budgetVariance / 100);
    
    additionalData.push({
      id: `PO${113000 + i}`,
      poDate: formatDate(poDate),
      poType: Math.random() > 0.75 ? 'Import' : 'Local', // 25% import, 75% local
      supplier,
      itemGroup,
      dcn: `DCN${4000 + Math.floor(Math.random() * 500)}`,
      uom,
      poQty: parseFloat(poQty.toFixed(2)),
      poRate: parseFloat(poRate.toFixed(2)),
      poVal: parseFloat((poQty * poRate).toFixed(2)),
      cancelQty: Math.random() > 0.85 ? Math.floor(Math.random() * (poQty * 0.1)) : 0, // 15% chance of cancellation
      budgetRate: parseFloat(budgetRate.toFixed(2)),
      leadDays: actualLeadDays.toString(),
      grnDate: Math.random() > 0.2 ? formatDate(materialReceivedDate) : '', // 80% have GRN
      division,
      purchaseManager: manager,
      customer,
      ocn: `OCN2025${(i + 6).toString().padStart(4, '0')}`,
      targetLeadDays,
      targetQuality,
      actualQuality,
      
      // Advanced PO details
      grnNumbers: Array.from({ length: grnCount }, (_, idx) => `GRN${5000 + i + idx}`),
      grnCount,
      supplierAdvancePaid,
      advanceAmount: supplierAdvancePaid ? parseFloat((poQty * poRate * (0.2 + Math.random() * 0.3)).toFixed(2)) : 0, // 20-50% advance
      advancePendingBillClose: supplierAdvancePaid ? Math.random() > 0.6 : false, // 40% pending closure
      customerPaymentType: customerCreditBased ? 'Credit' : 'Advance',
      customerCreditDays: customerCreditBased ? Math.floor(Math.random() * 120) + 30 : 0, // 30-150 days credit
      materialReceivedDate: formatDate(materialReceivedDate),
      billHandoverDate: formatDate(billHandoverDate),
      billHandoverDays,
      billHandoverDelayed: billHandoverDays > 7,
      invoiceNumber: `INV${currentYear}${(i + 1000).toString().padStart(5, '0')}`,
      paymentOutstanding: paymentStatus !== 'Paid' ? parseFloat((Math.random() * poQty * poRate * 0.8).toFixed(2)) : 0,
      paymentStatus,
      
      // Additional realistic fields
      orderPriority: ['High', 'Medium', 'Low'][Math.floor(Math.random() * 3)],
      deliveryLocation: ['Chennai', 'Bangalore', 'Mumbai', 'Delhi', 'Kolkata', 'Hyderabad', 'Pune', 'Coimbatore'][Math.floor(Math.random() * 8)],
      paymentTerms: customerCreditBased ? `${Math.floor(Math.random() * 120) + 30} Days Credit` : 'Advance Payment',
      currencyType: Math.random() > 0.9 ? 'USD' : 'INR', // 10% USD, 90% INR
      exchangeRate: Math.random() > 0.9 ? 82.5 + Math.random() * 2 : 1, // USD exchange rate if applicable,
    });
  }
  
  return [...mockPurchaseData, ...additionalData];
};

const allPurchaseData = generateMockData();

export function PurchaseDashboard() {
  const [selectedDivision, setSelectedDivision] = useState('All');
  const [selectedTimeframe, setSelectedTimeframe] = useState('Monthly');
  const [drilldownData, setDrilldownData] = useState(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [modalTitle, setModalTitle] = useState('');
  const [selectedBuyer, setSelectedBuyer] = useState('All');

  const [buyersList, setBuyersList] = useState<string[]>([]);


      useEffect(() => {
        fetch("/api/lineplanpolist")
          .then(res => res.json())
          .then(data => {
            if (data.success) setBuyersList(data.data);
          })
          .catch(err => console.error("Error fetching buyers:", err));
      }, []);

const filteredData = selectedDivision === 'All' && selectedBuyer === 'All'
  ? allPurchaseData
  : allPurchaseData.filter(item =>
      (selectedDivision === 'All' || item.division === selectedDivision) &&
      (selectedBuyer === 'All' || item.customer === selectedBuyer)
    );

  // Calculate KPIs
  const totalOrders = filteredData.length;
  const totalValue = filteredData.reduce((sum, item) => sum + item.poVal, 0);
  const averageLeadTime = filteredData
    .filter(item => item.leadDays)
    .reduce((sum, item) => sum + parseInt(item.leadDays), 0) / 
    filteredData.filter(item => item.leadDays).length;
  const onTimeDelivery = (filteredData.filter(item => item.grnDate).length / totalOrders) * 100;

  // Advanced KPIs
  const totalPaymentOutstanding = filteredData.reduce((sum, item) => sum + (item.paymentOutstanding || 0), 0);
  const advancePaymentOrders = filteredData.filter(item => item.supplierAdvancePaid).length;
  const delayedBillHandovers = filteredData.filter(item => item.billHandoverDelayed).length;
  const avgBillHandoverDays = filteredData
    .filter(item => item.billHandoverDays)
    .reduce((sum, item) => sum + item.billHandoverDays, 0) / 
    filteredData.filter(item => item.billHandoverDays).length;

  // Cost Savings KPI
  const costSavingsData = filteredData
    .map(item => {
      if (!item.budgetRate || !item.poRate || item.poRate >= item.budgetRate) {
        return { ...item, totalSavings: 0 };
      }
      const savingsPerUnit = item.budgetRate - item.poRate;
      const totalSavings = savingsPerUnit * item.poQty;
      return { ...item, totalSavings };
    })
    .filter(item => item.totalSavings > 0);
  
  const totalCostSavings = costSavingsData.reduce((sum, item) => sum + item.totalSavings, 0);
  const avgCostSavingsPercentage = costSavingsData.length > 0 
    ? costSavingsData.reduce((sum, item) => sum + ((item.budgetRate - item.poRate) / item.budgetRate * 100), 0) / costSavingsData.length 
    : 0;

  // Supplier performance data
  const supplierPerformance = filteredData.reduce((acc, item) => {
    if (!acc[item.supplier]) {
      acc[item.supplier] = {
        name: item.supplier,
        orders: 0,
        totalValue: 0,
        avgLeadTime: 0,
        onTimeDeliveries: 0,
        totalDeliveries: 0,
        qualityScore: Math.floor(Math.random() * 20) + 80, // Mock quality score
        rating: Math.floor(Math.random() * 2) + 4 // Mock rating 4-5 stars
      };
    }
    
    acc[item.supplier].orders += 1;
    acc[item.supplier].totalValue += item.poVal;
    if (item.leadDays) {
      acc[item.supplier].avgLeadTime = 
        (acc[item.supplier].avgLeadTime + parseInt(item.leadDays)) / 2;
    }
    if (item.grnDate) {
      acc[item.supplier].onTimeDeliveries += 1;
    }
    acc[item.supplier].totalDeliveries += 1;
    
    return acc;
  }, {});

  const supplierData = Object.values(supplierPerformance).map(supplier => ({
    ...supplier,
    onTimePercentage: (supplier.onTimeDeliveries / supplier.totalDeliveries) * 100
  }));

  // Time-based analysis
  const monthlyData = filteredData.reduce((acc, item) => {
    const month = item.poDate.split('/')[1] + '/' + item.poDate.split('/')[2];
    if (!acc[month]) {
      acc[month] = { month, orders: 0, value: 0 };
    }
    acc[month].orders += 1;
    acc[month].value += item.poVal;
    return acc;
  }, {});

  const timeSeriesData = Object.values(monthlyData);

  // Division breakdown
  const divisionData = allPurchaseData.reduce((acc, item) => {
    if (!acc[item.division]) {
      acc[item.division] = { name: item.division, value: 0, orders: 0 };
    }
    acc[item.division].value += item.poVal;
    acc[item.division].orders += 1;
    return acc;
  }, {});

  const divisionChartData = Object.values(divisionData);

  const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8', '#82CA9D'];

  // Manager performance analysis
  const managerPerformance = filteredData.reduce((acc, item) => {
    if (!acc[item.purchaseManager]) {
      acc[item.purchaseManager] = {
        name: item.purchaseManager,
        totalOrders: 0,
        totalValue: 0,
        leadTimeSum: 0,
        leadTimeCount: 0,
        targetLeadTimeSum: 0,
        qualitySum: 0,
        qualityCount: 0,
        targetQualitySum: 0,
        budgetVarianceSum: 0,
        budgetVarianceCount: 0,
        ordersGrowth: Math.floor(Math.random() * 30) - 10,
        valueGrowth: Math.floor(Math.random() * 25) - 5
      };
    }
    
    const manager = acc[item.purchaseManager];
    manager.totalOrders += 1;
    manager.totalValue += item.poVal;
    
    if (item.leadDays) {
      manager.leadTimeSum += parseInt(item.leadDays);
      manager.leadTimeCount += 1;
    }
    if (item.targetLeadDays) {
      manager.targetLeadTimeSum += item.targetLeadDays;
    }
    if (item.actualQuality) {
      manager.qualitySum += item.actualQuality;
      manager.qualityCount += 1;
    }
    if (item.targetQuality) {
      manager.targetQualitySum += item.targetQuality;
    }
    if (item.budgetRate && item.poRate) {
      const variance = ((item.poRate - item.budgetRate) / item.budgetRate) * 100;
      manager.budgetVarianceSum += variance;
      manager.budgetVarianceCount += 1;
    }
    
    return acc;
  }, {});

  const managerData = Object.values(managerPerformance).map(manager => {
    const avgActualLeadTime = manager.leadTimeCount > 0 ? manager.leadTimeSum / manager.leadTimeCount : 0;
    const avgTargetLeadTime = manager.leadTimeCount > 0 ? manager.targetLeadTimeSum / manager.leadTimeCount : 0;
    const avgActualQuality = manager.qualityCount > 0 ? manager.qualitySum / manager.qualityCount : 0;
    const avgTargetQuality = manager.qualityCount > 0 ? manager.targetQualitySum / manager.qualityCount : 0;
    const avgBudgetVariance = manager.budgetVarianceCount > 0 ? manager.budgetVarianceSum / manager.budgetVarianceCount : 0;
    
    const leadTimeScore = avgTargetLeadTime > 0 ? Math.max(0, 100 - ((avgActualLeadTime - avgTargetLeadTime) / avgTargetLeadTime * 100)) : 85;
    const qualityScore = avgTargetQuality > 0 ? (avgActualQuality / avgTargetQuality * 100) : 85;
    const costScore = 100 - Math.abs(avgBudgetVariance);
    const overallScore = (leadTimeScore + qualityScore + costScore) / 3;
    
    return {
      ...manager,
      avgActualLeadTime: avgActualLeadTime.toFixed(1),
      avgTargetLeadTime: avgTargetLeadTime.toFixed(1),
      avgActualQuality: avgActualQuality.toFixed(1),
      avgTargetQuality: avgTargetQuality.toFixed(1),
      avgBudgetVariance: avgBudgetVariance.toFixed(1),
      leadTimeScore: Math.round(leadTimeScore),
      qualityScore: Math.round(qualityScore),
      costScore: Math.round(Math.max(0, costScore)),
      overallScore: Math.round(overallScore),
      targetAchievement: Math.round(85 + Math.random() * 20)
    };
  });

  // Customer performance analysis
  const customerPerformance = filteredData.reduce((acc, item) => {
    if (!acc[item.customer]) {
      acc[item.customer] = {
        name: item.customer,
        orders: 0,
        totalValue: 0,
        leadTimeSum: 0,
        leadTimeCount: 0,
        targetLeadTimeSum: 0,
        qualitySum: 0,
        qualityCount: 0,
        onTimeDeliveries: 0,
        totalDeliveries: 0
      };
    }
    
    const customer = acc[item.customer];
    customer.orders += 1;
    customer.totalValue += item.poVal;
    customer.totalDeliveries += 1;
    
    if (item.leadDays) {
      customer.leadTimeSum += parseInt(item.leadDays);
      customer.leadTimeCount += 1;
    }
    if (item.targetLeadDays) {
      customer.targetLeadTimeSum += item.targetLeadDays;
    }
    if (item.actualQuality) {
      customer.qualitySum += item.actualQuality;
      customer.qualityCount += 1;
    }
    if (item.grnDate) {
      customer.onTimeDeliveries += 1;
    }
    
    return acc;
  }, {});

  const customerData = Object.values(customerPerformance).map(customer => {
    const avgLeadTime = customer.leadTimeCount > 0 ? customer.leadTimeSum / customer.leadTimeCount : 0;
    const avgTargetLeadTime = customer.leadTimeCount > 0 ? customer.targetLeadTimeSum / customer.leadTimeCount : 0;
    const qualityScore = customer.qualityCount > 0 ? customer.qualitySum / customer.qualityCount : 0;
    const onTimeDelivery = (customer.onTimeDeliveries / customer.totalDeliveries) * 100;
    const overallScore = (qualityScore + onTimeDelivery + (100 - Math.abs(avgLeadTime - avgTargetLeadTime))) / 3;
    
    return {
      ...customer,
      avgLeadTime: avgLeadTime.toFixed(1),
      avgTargetLeadTime: avgTargetLeadTime.toFixed(1),
      qualityScore: Math.round(qualityScore),
      onTimeDelivery: Math.round(onTimeDelivery),
      overallScore: Math.round(overallScore),
      growth: Math.floor(Math.random() * 30) - 10,
      targetLeadTime: avgTargetLeadTime.toFixed(1),
      actualLeadTime: avgLeadTime.toFixed(1),
      deliveryScore: Math.round(onTimeDelivery),
      costEfficiency: Math.round(85 + Math.random() * 15)
    };
  });

  // Generate mock data for charts based on filtered data
  const generateFilteredManagerData = () => {
    const filteredManagerData = managerData.filter(manager => 
      filteredData.some(item => item.purchaseManager === manager.name)
    );
    
    return {
      weekly: [
        { week: 'Week 1', brittoScore: 85 + Math.random() * 10, udhayaScore: 88 + Math.random() * 8 },
        { week: 'Week 2', brittoScore: 89 + Math.random() * 8, udhayaScore: 82 + Math.random() * 12 },
        { week: 'Week 3', brittoScore: 92 + Math.random() * 6, udhayaScore: 87 + Math.random() * 10 },
        { week: 'Week 4', brittoScore: 88 + Math.random() * 9, udhayaScore: 91 + Math.random() * 7 }
      ],
      monthly: filteredManagerData.length >= 2 ? [
        { 
          period: 'Jan', 
          brittoOrders: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalOrders * 0.3 || 15,
          udhayaOrders: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalOrders * 0.3 || 12,
          brittoValue: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalValue * 0.3 || 50000,
          udhayaValue: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalValue * 0.3 || 40000
        },
        { 
          period: 'Feb', 
          brittoOrders: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalOrders * 0.35 || 18,
          udhayaOrders: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalOrders * 0.35 || 15,
          brittoValue: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalValue * 0.35 || 60000,
          udhayaValue: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalValue * 0.35 || 48000
        },
        { 
          period: 'Mar', 
          brittoOrders: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalOrders * 0.35 || 16,
          udhayaOrders: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalOrders * 0.35 || 14,
          brittoValue: filteredManagerData.find(m => m.name === 'Mr.Britto')?.totalValue * 0.35 || 55000,
          udhayaValue: filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.totalValue * 0.35 || 45000
        }
      ] : [
        { period: 'Jan', brittoOrders: 15, udhayaOrders: 12, brittoValue: 50000, udhayaValue: 40000 },
        { period: 'Feb', brittoOrders: 18, udhayaOrders: 15, brittoValue: 60000, udhayaValue: 48000 },
        { period: 'Mar', brittoOrders: 16, udhayaOrders: 14, brittoValue: 55000, udhayaValue: 45000 }
      ],
      targetVsActual: filteredManagerData.length >= 2 ? [
        { 
          manager: 'Mr.Britto', 
          leadTimeVariance: parseFloat(filteredManagerData.find(m => m.name === 'Mr.Britto')?.avgBudgetVariance || '0') / 10,
          qualityVariance: (filteredManagerData.find(m => m.name === 'Mr.Britto')?.qualityScore || 85) - 90,
          costVariance: parseFloat(filteredManagerData.find(m => m.name === 'Mr.Britto')?.avgBudgetVariance || '0')
        },
        { 
          manager: 'Mr.Udhaya', 
          leadTimeVariance: parseFloat(filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.avgBudgetVariance || '0') / 10,
          qualityVariance: (filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.qualityScore || 85) - 90,
          costVariance: parseFloat(filteredManagerData.find(m => m.name === 'Mr.Udhaya')?.avgBudgetVariance || '0')
        }
      ] : [
        { manager: 'Mr.Britto', leadTimeVariance: -2.5, qualityVariance: 5.2, costVariance: -1.8 },
        { manager: 'Mr.Udhaya', leadTimeVariance: 1.2, qualityVariance: -3.1, costVariance: 2.4 }
      ]
    };
  };

  const generateFilteredCustomerTrends = () => {
    const availableCustomers = [...new Set(filteredData.map(item => item.customer))];
    const months = ['Jan', 'Feb', 'Mar'];
    
    return months.map(month => {
      const monthData = { month };
      availableCustomers.slice(0, 4).forEach(customer => {
        const customerData = filteredData.filter(item => item.customer === customer);
        const avgValue = customerData.length > 0 
          ? customerData.reduce((sum, item) => sum + item.poVal, 0) / customerData.length
          : Math.floor(Math.random() * 50000) + 30000;
        monthData[customer.toLowerCase().replace('&', '')] = Math.floor(avgValue * (0.8 + Math.random() * 0.4));
      });
      return monthData;
    });
  };

  const filteredManagerChartData = generateFilteredManagerData();
  const filteredCustomerTrends = generateFilteredCustomerTrends();

  const weeklyPerformanceData = filteredManagerChartData.weekly;
  const monthlyManagerData = filteredManagerChartData.monthly;
  const targetVsActualData = filteredManagerChartData.targetVsActual;

  const managerRadarData = [
    { metric: 'Lead Time', britto: 88, udhaya: 82 },
    { metric: 'Quality', britto: 92, udhaya: 85 },
    { metric: 'Cost Control', britto: 85, udhaya: 91 },
    { metric: 'On-Time Delivery', britto: 89, udhaya: 87 },
    { metric: 'Supplier Relations', britto: 90, udhaya: 89 },
    { metric: 'Order Volume', britto: 87, udhaya: 84 }
  ];

  const customerTrendData = filteredCustomerTrends;

  const handleChartClick = (data: any, title: string) => {
    setDrilldownData(data);
    setModalTitle(title);
    setIsModalOpen(true);
  };

  const handleRowClick = (row: any) => {
    setDrilldownData(row);
    setModalTitle('Purchase Order Details');
    setIsModalOpen(true);
  };

  return (
    <div className="p-6 space-y-6 bg-gradient-to-br from-slate-50 to-blue-50 min-h-screen">
      {/* Enhanced Header */}
      <div className="flex items-center justify-between bg-white p-6 rounded-xl shadow-sm border">
        <div>
          <h1 className="text-4xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
            Purchase Dashboard
          </h1>
          <p className="text-muted-foreground mt-2">GMNUP INDUSTRIES INDIA PVT LIMITED</p>
          <p className="text-sm text-muted-foreground">Real-time Purchase Analytics & Performance Insights</p>
          {selectedDivision !== 'All' && (
            <div className="mt-3 flex items-center gap-2">
              <Badge variant="outline" className="bg-blue-50 text-blue-700 border-blue-200">
                <Package className="h-3 w-3 mr-1" />
                {selectedDivision} Division
              </Badge>
              <span className="text-sm text-muted-foreground">
                {totalOrders} orders • ₹{totalValue.toLocaleString()} value
              </span>
            </div>
          )}
        </div>
        <div className="flex gap-4">
              <Select value={selectedBuyer} onValueChange={setSelectedBuyer}>
                <SelectTrigger className="w-48 bg-gradient-to-r from-blue-500 to-purple-500 text-white border-none">
                  <SelectValue placeholder="Select Buyer" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="All">All Buyers</SelectItem>
                  {buyersList.map((buyer, index) => (
                <SelectItem key={index} value={buyer.Buyer}>
                  {buyer.Buyer}
                </SelectItem>
                  ))}
                </SelectContent>
              </Select>
          <Select value={selectedDivision} onValueChange={setSelectedDivision}>
            <SelectTrigger className="w-48 bg-gradient-to-r from-blue-500 to-purple-500 text-white border-none">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="All">All Divisions</SelectItem>
              <SelectItem value="Garments">Garments</SelectItem>
              <SelectItem value="Spinning">Spinning</SelectItem>
              <SelectItem value="Socks">Socks</SelectItem>
              <SelectItem value="Gloves">Gloves</SelectItem>
              <SelectItem value="Woven">Woven</SelectItem>
              <SelectItem value="Headwear">Headwear</SelectItem>
            </SelectContent>
          </Select>
          <Select value={selectedTimeframe} onValueChange={setSelectedTimeframe}>
            <SelectTrigger className="w-32 bg-gradient-to-r from-green-500 to-teal-500 text-white border-none">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="Weekly">Weekly</SelectItem>
              <SelectItem value="Monthly">Monthly</SelectItem>
              <SelectItem value="Quarterly">Quarterly</SelectItem>
              <SelectItem value="Yearly">Yearly</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Division Specific Info Panel */}
      {selectedDivision !== 'All' && (
        <Card className="bg-gradient-to-r from-blue-50 to-purple-50 border-blue-200">
          <CardContent className="p-4">
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="text-center">
                <h4 className="font-medium text-blue-800">Primary Materials</h4>
                <p className="text-sm text-blue-600">
                  {selectedDivision === 'Spinning' && 'Cotton Bales, Raw Cotton'}
                  {selectedDivision === 'Socks' && 'Yarn, Elastic, Trims'}
                  {(selectedDivision === 'Gloves' || selectedDivision === 'Woven' || selectedDivision === 'Headwear') && 'Fabric, Trims, Accessories'}
                  {selectedDivision === 'Garments' && 'Labels, Threads, Buttons'}
                </p>
              </div>
              <div className="text-center">
                <h4 className="font-medium text-blue-800">Main UOM</h4>
                <p className="text-sm text-blue-600">
                  {selectedDivision === 'Spinning' && 'Kgs, Tonnes, Bales'}
                  {selectedDivision === 'Socks' && 'Kgs, Meters, Pieces'}
                  {(selectedDivision === 'Gloves' || selectedDivision === 'Woven' || selectedDivision === 'Headwear') && 'Yard, Meter, Pieces'}
                  {selectedDivision === 'Garments' && 'Nos, Cone, Pieces'}
                </p>
              </div>
              <div className="text-center">
                <h4 className="font-medium text-blue-800">Key Customers</h4>
                <p className="text-sm text-blue-600">
                  {customerData.slice(0, 2).map(c => c.name).join(', ')}
                </p>
              </div>
              <div className="text-center">
                <h4 className="font-medium text-blue-800">Top Suppliers</h4>
                <p className="text-sm text-blue-600">
                  {supplierData.slice(0, 2).map(s => s.name.split(' ')[0]).join(', ')}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* KPI Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Orders</CardTitle>
            <ShoppingCart className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalOrders.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">
              <span className="text-green-600 flex items-center">
                <TrendingUp className="h-3 w-3 mr-1" />
                +12% from last month
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Value</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">₹{totalValue.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">
              <span className="text-green-600 flex items-center">
                <TrendingUp className="h-3 w-3 mr-1" />
                +8% from last month
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Avg Lead Time</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{averageLeadTime.toFixed(1)} days</div>
            <p className="text-xs text-muted-foreground">
              <span className="text-red-600 flex items-center">
                <TrendingDown className="h-3 w-3 mr-1" />
                -2 days improved
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">On-Time Delivery</CardTitle>
            <CheckCircle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{onTimeDelivery.toFixed(1)}%</div>
            <p className="text-xs text-muted-foreground">
              <span className="text-green-600 flex items-center">
                <TrendingUp className="h-3 w-3 mr-1" />
                +5% from last month
              </span>
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Advanced KPI Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Payment Outstanding</CardTitle>
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">₹{totalPaymentOutstanding.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">
              <span className={`flex items-center ${totalPaymentOutstanding > 0 ? 'text-red-600' : 'text-green-600'}`}>
                {totalPaymentOutstanding > 0 ? (
                  <AlertTriangle className="h-3 w-3 mr-1" />
                ) : (
                  <CheckCircle className="h-3 w-3 mr-1" />
                )}
                {((totalPaymentOutstanding / totalValue) * 100).toFixed(1)}% of total value
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Advance Payments</CardTitle>
            <DollarSign className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{advancePaymentOrders}</div>
            <p className="text-xs text-muted-foreground">
              <span className="text-blue-600 flex items-center">
                <Package className="h-3 w-3 mr-1" />
                {((advancePaymentOrders / totalOrders) * 100).toFixed(1)}% of total orders
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">GRN to Bill Days</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{avgBillHandoverDays.toFixed(1)} days</div>
            <p className="text-xs text-muted-foreground">
              <span className={`flex items-center ${avgBillHandoverDays > 7 ? 'text-red-600' : 'text-green-600'}`}>
                {avgBillHandoverDays > 7 ? (
                  <TrendingUp className="h-3 w-3 mr-1" />
                ) : (
                  <CheckCircle className="h-3 w-3 mr-1" />
                )}
                Target: ≤7 days
              </span>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Delayed Bills</CardTitle>
            <AlertTriangle className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{delayedBillHandovers}</div>
            <p className="text-xs text-muted-foreground">
              <span className={`flex items-center ${delayedBillHandovers > 0 ? 'text-red-600' : 'text-green-600'}`}>
                {delayedBillHandovers > 0 ? (
                  <AlertTriangle className="h-3 w-3 mr-1" />
                ) : (
                  <CheckCircle className="h-3 w-3 mr-1" />
                )}
                {((delayedBillHandovers / totalOrders) * 100).toFixed(1)}% of orders delayed
              </span>
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Cost Savings KPI Card */}
      {totalCostSavings > 0 && (
        <Card className="bg-gradient-to-r from-green-50 to-emerald-50 border-green-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-lg font-semibold text-green-800 mb-2">💰 Cost Savings Achievement</h3>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div className="text-center">
                    <div className="text-3xl font-bold text-green-600">₹{totalCostSavings.toLocaleString()}</div>
                    <p className="text-sm text-green-700">Total Cost Savings</p>
                    <p className="text-xs text-muted-foreground">{costSavingsData.length} orders contributed</p>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold text-green-600">{avgCostSavingsPercentage.toFixed(1)}%</div>
                    <p className="text-sm text-green-700">Average Savings %</p>
                    <p className="text-xs text-muted-foreground">vs Budget Approved Rate</p>
                  </div>
                  <div className="text-center">
                    <div className="text-3xl font-bold text-green-600">{((costSavingsData.length / totalOrders) * 100).toFixed(1)}%</div>
                    <p className="text-sm text-green-700">Orders with Savings</p>
                    <p className="text-xs text-muted-foreground">{costSavingsData.length} of {totalOrders} orders</p>
                  </div>
                </div>
              </div>
              <div className="text-right">
                <Button 
                  variant="outline"
                  className="bg-green-100 border-green-300 text-green-800 hover:bg-green-200"
                  onClick={() => handleChartClick(costSavingsData, 'Cost Savings Details')}
                >
                  View Details
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Main Dashboard Tabs */}
      <Tabs defaultValue="overview" className="space-y-4">
        <TabsList className="grid w-full grid-cols-9 bg-white shadow-sm">
          <TabsTrigger value="overview" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-blue-500 data-[state=active]:to-purple-500 data-[state=active]:text-white text-xs">
            <BarChart3 className="h-3 w-3" />
            Overview
          </TabsTrigger>
          <TabsTrigger value="managers" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-green-500 data-[state=active]:to-teal-500 data-[state=active]:text-white text-xs">
            <UserCheck className="h-3 w-3" />
            Managers
          </TabsTrigger>
          <TabsTrigger value="customers" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-orange-500 data-[state=active]:to-red-500 data-[state=active]:text-white text-xs">
            <ShoppingBag className="h-3 w-3" />
            Customers
          </TabsTrigger>
          <TabsTrigger value="suppliers" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-purple-500 data-[state=active]:to-pink-500 data-[state=active]:text-white text-xs">
            <Users className="h-3 w-3" />
            Suppliers
          </TabsTrigger>
          <TabsTrigger value="performance" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-indigo-500 data-[state=active]:to-blue-500 data-[state=active]:text-white text-xs">
            <Target className="h-3 w-3" />
            Targets
          </TabsTrigger>
          <TabsTrigger value="quality" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-yellow-500 data-[state=active]:to-orange-500 data-[state=active]:text-white text-xs">
            <Star className="h-3 w-3" />
            Quality
          </TabsTrigger>
          <TabsTrigger value="trends" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-cyan-500 data-[state=active]:to-blue-500 data-[state=active]:text-white text-xs">
            <LucideLineChart className="h-3 w-3" />
            Trends
          </TabsTrigger>
          <TabsTrigger value="advanced" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-red-500 data-[state=active]:to-pink-500 data-[state=active]:text-white text-xs">
            <FileText className="h-3 w-3" />
            Advanced
          </TabsTrigger>
          <TabsTrigger value="data" className="flex items-center gap-1 data-[state=active]:bg-gradient-to-r data-[state=active]:from-gray-500 data-[state=active]:to-slate-500 data-[state=active]:text-white text-xs">
            <Table className="h-3 w-3" />
            Data
          </TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <ProfessionalPieChart 
              data={divisionChartData}
              title="Division Breakdown"
              onClick={(data) => handleChartClick(data, 'Division Analysis')}
              height={350}
            />
            <ProfessionalAreaChart 
              data={timeSeriesData}
              title="Purchase Value Trends"
              onClick={(data) => handleChartClick(data, 'Monthly Trends')}
              height={350}
            />
          </div>
          
          <div className="grid grid-cols-1 xl:grid-cols-1 gap-6">
            <ProfessionalComposedChart 
              data={timeSeriesData}
              title="Purchase Analysis - Value vs Orders"
              onClick={(data) => handleChartClick(data, 'Purchase Analysis')}
              height={400}
            />
          </div>
        </TabsContent>

        <TabsContent value="managers" className="space-y-6">
          {/* Manager Performance KPI Cards */}
          <ManagerKPICards managerData={managerData} />
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <ManagerPerformanceComparison 
              data={monthlyManagerData}
              title="Manager Performance Comparison"
              onClick={(data) => handleChartClick(data, 'Manager Comparison')}
              height={400}
            />
            <ManagerRadarChart 
              data={managerRadarData}
              title="Manager Performance Radar"
              onClick={(data) => handleChartClick(data, 'Manager Radar')}
              height={400}
            />
          </div>
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <TargetVsActualChart 
              data={targetVsActualData}
              title="Target vs Actual Performance"
              onClick={(data) => handleChartClick(data, 'Target Analysis')}
              height={350}
            />
            <WeeklyPerformanceChart 
              data={weeklyPerformanceData}
              title="Weekly Performance Tracking"
              onClick={(data) => handleChartClick(data, 'Weekly Performance')}
              height={350}
            />
          </div>
        </TabsContent>

        <TabsContent value="customers" className="space-y-6">
          {/* Customer KPI Cards */}
          <CustomerKPICards customerData={customerData} />
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <CustomerValueChart 
              data={customerData}
              title="Customer Purchase Value & Orders"
              onClick={(data) => handleChartClick(data, 'Customer Analysis')}
              height={400}
            />
            <CustomerDistribution 
              data={customerData.map(c => ({ name: c.name, value: c.totalValue }))}
              title="Customer Purchase Distribution"
              onClick={(data) => handleChartClick(data, 'Customer Distribution')}
              height={400}
            />
          </div>
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <CustomerPerformanceMatrix 
              data={customerData}
              title="Customer Performance Matrix"
              onClick={(data) => handleChartClick(data, 'Customer Performance')}
              height={350}
            />
            <CustomerLeadTimeAnalysis 
              data={customerData}
              title="Customer Lead Time Analysis"
              onClick={(data) => handleChartClick(data, 'Lead Time Analysis')}
              height={350}
            />
          </div>
          
          <Card>
            <CardHeader>
              <CardTitle>Customer Trend Analysis</CardTitle>
            </CardHeader>
            <CardContent>
              <CustomerTrendAnalysis 
                data={customerTrendData}
                title="Customer Purchase Trends"
                onClick={(data) => handleChartClick(data, 'Customer Trends')}
                height={350}
              />
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="suppliers" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <SupplierPerformanceChart 
              data={supplierData}
              title="Supplier Performance Matrix"
              onClick={(data) => handleChartClick(data, 'Supplier Details')}
              height={400}
            />
            <ProfessionalBarChart 
              data={supplierData.slice(0, 8).map(s => ({ name: s.name, value: s.totalValue }))}
              title="Top Suppliers by Value"
              onClick={(data) => handleChartClick(data, 'Supplier Performance')}
              height={400}
            />
          </div>
          
          <Card>
            <CardHeader>
              <CardTitle>Supplier Performance Rankings</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {supplierData.slice(0, 10).map((supplier, index) => (
                  <div key={supplier.name} 
                       className="flex items-center justify-between p-4 border rounded-lg hover:shadow-md transition-shadow cursor-pointer"
                       onClick={() => handleChartClick(supplier, 'Supplier Details')}>
                    <div className="flex items-center space-x-4">
                      <div className="w-8 h-8 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                        #{index + 1}
                      </div>
                      <div className="flex items-center">
                        {Array.from({ length: 5 }).map((_, i) => (
                          <Star
                            key={i}
                            className={`h-4 w-4 ${
                              i < supplier.rating ? 'text-yellow-400 fill-current' : 'text-gray-300'
                            }`}
                          />
                        ))}
                      </div>
                      <div>
                        <h3 className="font-medium">{supplier.name}</h3>
                        <p className="text-sm text-muted-foreground">
                          {supplier.orders} orders  ₹{supplier.totalValue.toLocaleString()}
                        </p>
                      </div>
                    </div>
                    <div className="text-right">
                      <Badge variant={supplier.qualityScore >= 90 ? 'default' : 'secondary'}>
                        Quality: {supplier.qualityScore}%
                      </Badge>
                      <p className="text-sm text-muted-foreground mt-1">
                        Lead: {supplier.avgLeadTime.toFixed(1)} days • OTD: {supplier.onTimePercentage.toFixed(1)}%
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="performance" className="space-y-6">
          {/* Division Target vs Actual Chart */}
          <DivisionTargetVsActualChart 
            data={filteredData}
            title="Division-wise Monthly Target vs Actual Purchase"
            selectedDivision={selectedDivision}
            height={500}
          />
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <ProfessionalBarChart 
              data={supplierData.slice(0, 8).map(s => ({ name: s.name, value: s.orders }))}
              title="Orders by Supplier"
              onClick={(data) => handleChartClick(data, 'Supplier Orders')}
              height={350}
            />
            <ProfessionalLineChart 
              data={supplierData.slice(0, 8).map(s => ({ month: s.name, value: s.avgLeadTime, orders: s.qualityScore }))}
              title="Lead Time vs Quality Score"
              onClick={(data) => handleChartClick(data, 'Performance Metrics')}
              height={350}
            />
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Best Performer</CardTitle>
                <TrendingUp className="h-4 w-4 text-green-500" />
              </CardHeader>
              <CardContent>
                <div className="text-lg font-bold">{supplierData[0]?.name}</div>
                <p className="text-xs text-muted-foreground">
                  Quality: {supplierData[0]?.qualityScore}% • Lead: {supplierData[0]?.avgLeadTime.toFixed(1)} days
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Fastest Delivery</CardTitle>
                <Clock className="h-4 w-4 text-blue-500" />
              </CardHeader>
              <CardContent>
                <div className="text-lg font-bold">
                  {supplierData.reduce((min, s) => s.avgLeadTime < min.avgLeadTime ? s : min, supplierData[0])?.name}
                </div>
                <p className="text-xs text-muted-foreground">
                  {supplierData.reduce((min, s) => s.avgLeadTime < min.avgLeadTime ? s : min, supplierData[0])?.avgLeadTime.toFixed(1)} days average
                </p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Highest Volume</CardTitle>
                <Package className="h-4 w-4 text-purple-500" />
              </CardHeader>
              <CardContent>
                <div className="text-lg font-bold">
                  {supplierData.reduce((max, s) => s.totalValue > max.totalValue ? s : max, supplierData[0])?.name}
                </div>
                <p className="text-xs text-muted-foreground">
                  ₹{supplierData.reduce((max, s) => s.totalValue > max.totalValue ? s : max, supplierData[0])?.totalValue.toLocaleString()}
                </p>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="quality" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <ProfessionalBarChart 
              data={[
                { name: '90-100%', value: supplierData.filter(s => s.qualityScore >= 90).length },
                { name: '80-89%', value: supplierData.filter(s => s.qualityScore >= 80 && s.qualityScore < 90).length },
                { name: '70-79%', value: supplierData.filter(s => s.qualityScore >= 70 && s.qualityScore < 80).length },
                { name: '<70%', value: supplierData.filter(s => s.qualityScore < 70).length },
              ]}
              title="Quality Score Distribution"
              onClick={(data) => handleChartClick(data, 'Quality Analysis')}
              height={300}
            />

            <Card>
              <CardHeader>
                <CardTitle>Quality Metrics</CardTitle>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid grid-cols-2 gap-4">
                  <div className="text-center p-4 bg-green-50 rounded-lg">
                    <div className="text-2xl font-bold text-green-600">
                      {(supplierData.reduce((sum, s) => sum + s.qualityScore, 0) / supplierData.length).toFixed(1)}%
                    </div>
                    <p className="text-sm text-green-700">Average Quality Score</p>
                  </div>
                  <div className="text-center p-4 bg-blue-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">
                      {supplierData.filter(s => s.qualityScore >= 90).length}
                    </div>
                    <p className="text-sm text-blue-700">Suppliers Above 90%</p>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <h4 className="font-medium">Top Quality Suppliers</h4>
                  {supplierData
                    .sort((a, b) => b.qualityScore - a.qualityScore)
                    .slice(0, 5)
                    .map((supplier, index) => (
                    <div key={supplier.name} className="flex justify-between items-center p-2 hover:bg-muted/50 rounded">
                      <span className="text-sm">{supplier.name}</span>
                      <Badge variant="default" className="bg-green-100 text-green-800">
                        {supplier.qualityScore}%
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Quality Issues</CardTitle>
              <AlertTriangle className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold mb-4">
                {filteredData.filter(item => item.cancelQty > 0).length}
              </div>
              <p className="text-xs text-muted-foreground mb-4">
                <span className="text-red-600">Orders with cancellations</span>
              </p>
              
              <div className="space-y-2">
                <h4 className="font-medium">Recent Quality Issues</h4>
                {filteredData
                  .filter(item => item.cancelQty > 0)
                  .slice(0, 5)
                  .map((item) => (
                  <div key={item.id} className="flex justify-between items-center p-2 bg-red-50 rounded border-l-2 border-red-200">
                    <div>
                      <span className="text-sm font-medium">{item.id}</span>
                      <p className="text-xs text-muted-foreground">{item.supplier}</p>
                    </div>
                    <Badge variant="destructive">
                      {item.cancelQty} {item.uom} cancelled
                    </Badge>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="trends" className="space-y-6">
          <div className="grid grid-cols-1 gap-6">
            <ProfessionalComposedChart 
              data={timeSeriesData}
              title="Purchase Trends"
              onClick={(data) => handleChartClick(data, 'Purchase Trends')}
              height={400}
            />
          </div>
        </TabsContent>

        <TabsContent value="advanced" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <AdvancedPODetails 
              data={filteredData}
              title="Advanced PO Details"
              onClick={(data) => handleChartClick(data, 'Advanced PO Details')}
              height={400}
            />
            <AdvanceDetailsBreakup 
              data={filteredData}
              title="Advance Details Breakup"
              onClick={(data) => handleChartClick(data, 'Advance Details Breakup')}
              height={400}
            />
          </div>
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <GRNToBillBreakup 
              data={filteredData}
              title="GRN to Bill Breakup"
              onClick={(data) => handleChartClick(data, 'GRN to Bill Breakup')}
              height={400}
            />
            <DelayedBillsBreakup 
              data={filteredData}
              title="Delayed Bills Breakup"
              onClick={(data) => handleChartClick(data, 'Delayed Bills Breakup')}
              height={400}
            />
          </div>
          
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <CostSavingsBreakup 
              data={filteredData}
              title="Cost Savings Breakup"
              onClick={(data) => handleChartClick(data, 'Cost Savings Breakup')}
              height={400}
            />
          </div>
        </TabsContent>

        <TabsContent value="data" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <DetailedDataGrid 
              data={filteredData}
              title="Purchase Order Data"
              onClick={(data) => handleRowClick(data)}
              height={400}
            />
          </div>
        </TabsContent>
      </Tabs>

      {/* Drilldown Modal */}
      {isModalOpen && (
        <DrilldownModal 
          data={drilldownData}
          title={modalTitle}
          onClose={() => setIsModalOpen(false)}
        />
      )}
    </div>
  );
};