<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Direct Labour Strength</title>

  <!-- Tailwind CSS -->
  <script src="https://cdn.tailwindcss.com"></script>
  <!-- jQuery -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <!-- Font Awesome CSS -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <!-- Date picker CDN -->
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script type="text/javascript" src="https://cdn.jsdelivr.net/jquery/latest/jquery.min.js"></script>
  <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

  <style>
    body {
      background-color: #f8fafc;
      font-family: 'Inter', sans-serif;
    }
    .card {
      background: white;
      border-radius: 0.75rem;
      box-shadow: 0 1px 3px rgba(0,0,0,0.05);
      padding: 1.25rem;
    }
    .kpi-spark {
      height: 20px; /* Reduced height */
      width: 60px;  /* Reduced width */
      max-width: 100%; /* Ensure it doesn't exceed container */
      overflow: hidden; /* Prevent overflow */
    }
    .kpi-content {
      display: flex;
      justify-content: space-between;
      align-items: center;
      width: 100%;
    }
    .table-row {
      transition: background 0.2s ease;
    }
    .table-row:hover {
      background-color: #f9fafb;
    }
    .chart-container {
      position: relative;
      height: 360px;
      width: 100%;
    }
    .chart-container > canvas {
      width: 100% !important;
      height: 100% !important;
      display: block;
    }
    .accordion-row { font-size: 13px; }
    .level-1 { background-color: #eaf2ff; }
    .level-2 { background-color: #f3f7ff; }
    .level-3 { background-color: #ffffff; }
    .level-4 { background-color: #ffffff; }
    .caret { display: inline-block; width: 14px; color: #0b3b91; }
    .name-cell { display: flex; align-items: center; gap: 8px; }
    .badge { display: inline-block; padding: 2px 8px; border-radius: 12px; font-size: 11px; line-height: 16px; }
    .badge-red { background-color: #ffe4e6; color: #b91c1c; }
    .badge-yellow { background-color: #fff7cc; color: #7a5d00; }
    .dot-red { width: 8px; height: 8px; background: #dc2626; border-radius: 9999px; display: inline-block; }
    .muted { color: #6b7280; }
    .spinner {
        display: inline-block;
        width: 1.5rem;
        height: 1.5rem;
        border: 3px solid rgba(255, 255, 255, 0.3);
        border-top-color: #fff;
        border-radius: 50%;
        animation: spin 1s ease-in-out infinite;
    }
    @keyframes spin {
        to { transform: rotate(360deg); }
    }
  </style>
</head>

<body class="min-h-screen flex flex-col">
  <!-- Top Bar -->
  @include('LifeAtGainup.Production.header')

  <!-- Breadcrumb -->
  <div class="px-6 py-3 text-sm text-gray-600">
    <a id="productionBreadcrumb" href="/production_gar_dashboard" class="text-blue-700 hover:underline">Production Dashboard</a> 
    <span> &gt; </span>
    <span class="text-gray-800 font-medium">Direct Labour Strength</span>
  </div>

  <!-- KPI Cards -->
  <div class="px-6 grid grid-cols-5 gap-4">
    <!-- Strength -->
    <div class="card cursor-pointer" id="kpi1">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Strength</div>
          <div class="text-3xl font-bold mt-1" id="strengthVal">455</div>
          <div class="text-xs text-gray-500 mt-1">▼ 2.2% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 480</div>
        </div>
        <canvas id="spark1" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Absenteeism -->
    <div class="card cursor-pointer" id="kpi2" style="pointer-events: auto; transition: none; box-shadow: none !important; background-color: #ddd8d8ff !important;">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Absenteeism</div>
          <div class="text-3xl font-bold mt-1" id="absentVal">3.5%</div>
          <div class="text-xs text-red-500 mt-1">▲ 10.5% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 4.0%</div>
        </div>
        <canvas id="spark2" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Attrition -->
    <div class="card cursor-pointer" id="kpi3" style="pointer-events: auto; transition: none; box-shadow: none !important; background-color: #ddd8d8ff !important;">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Attrition</div>
          <div class="text-3xl font-bold mt-1" id="attritionVal">1.2%</div>
          <div class="text-xs text-red-500 mt-1">▲ 5.2% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 1.5%</div>
        </div>
        <canvas id="spark3" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Recruitment -->
    <div class="card cursor-pointer" id="kpi4" style="pointer-events: auto; transition: none; box-shadow: none !important; background-color: #ddd8d8ff !important;">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Recruitment</div>
          <div class="text-3xl font-bold mt-1" id="recruitVal">15</div>
          <div class="text-xs text-green-500 mt-1">▼ 5.0% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 20</div>
        </div>
        <canvas id="spark4" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Skill Balance Ratio -->
    <div class="card flex flex-col justify-center" style="pointer-events: auto; transition: none; box-shadow: none !important; background-color: #ddd8d8ff !important;">
      <div class="text-sm text-gray-600">Skill Balance Ratio</div>
      <div class="text-xl font-semibold mt-1">1 : 5 : 10 : 4</div>
      <div class="text-xs text-gray-500 mt-1">vs ideal 1 : 4 : 8 : 2</div>
    </div>
  </div>

  <!-- Charts -->
  <div class="px-6 mt-6 grid grid-cols-2 gap-4">
    <div class="card">
      <div class="flex flex-row items-center mb-1 gap-2">
          <div class="font-semibold mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Overall Trend for Recruitment</div>
          <div class="text-sm text-gray-500 mb-2">Time-series data for the selected labour metric or category.</div>
      </div>    
      <div class="chart-container"><canvas id="lineChart"></canvas></div>
    </div>

    <div class="card">
      <div class="font-semibold mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Direct Labour Categories</div>
      <div class="text-sm text-gray-500 mb-3 hidden">Breakdown by labour category for Recruitment. Click 'Tailor' to drill down.</div>
      <div id="barCrumb" class="text-xs text-[var(--muted)] mt-2">ALL Departments</div>
      <div class="chart-container"><canvas id="barChart" class="cursor-pointer"></canvas></div>
    </div>
  </div>

  <!-- Line-wise Drilldown -->
  <div class="px-6 mt-6 mb-10">
    <div class="card">
      <div class="flex flex-row items-center mb-3 gap-2">
        <div class="font-semibold text-gray-700 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Line-wise Drilldown</div>
         <div class="text-sm text-gray-500 mb-1">Aggregated labour KPIs. Click on a unit or zone to expand.</div>
        <button class="text-sm text-gray-500 hidden">⤓</button>
      </div>
     

      <table class="w-full text-lg text-gray-700" id="drilldownTable">
        <thead class="text-gray-500 border-b">
          <tr>
            <th class="text-left py-2">Name</th>
            <th class="text-left py-2">Strength</th>
            <th class="text-left py-2">Absenteeism (%)</th>
            <th class="text-left py-2">Attrition (%)</th>
            <th class="text-left py-2">Skill Balance Ratio</th>
          </tr>
        </thead>
        <tbody>
          
        </tbody>
      </table>
    </div>
  </div>

  <script>
    let apiData = null;
    let selectedRangeLabel = "Last Week";
	let selectedfromdate = null;
	let selectedenddate = null;
    const useAccordionTable = true;

    // KPI Sparkline Charts
    const sparkOptions = {
      responsive: true,
      maintainAspectRatio: false,
      elements: {
        line: { tension: 0.4, borderWidth: 2 },
        point: { radius: 0 }
      },
      plugins: { legend: { display: false } },
      scales: { x: { display: false }, y: { display: false } }
    };

    const sparkData = [
      { id: 'spark1', color: '#2563eb' },
      { id: 'spark2', color: '#ef4444' },
      { id: 'spark3', color: '#22c55e' },
      { id: 'spark4', color: '#10b981' },
    ];

    sparkData.forEach(s => {
      new Chart(document.getElementById(s.id), {
        type: 'line',
        data: {
          labels: [1, 2, 3, 4, 5, 6, 7],
          datasets: [{
            data: [3, 3.2, 3.3, 3.4, 3.8, 4, 4.5],
            borderColor: s.color,
            fill: false
          }]
        },
        options: sparkOptions
      });
    });

    // Process API data to extract required information
    function processApiData(response) {
      if (!response || !response.units || !Array.isArray(response.units)) return null;

      const departments = new Set();
      const designations = new Set();
      const departmentCounts = {};
      const designationCountsPerDept = {};
      const trendData = {};
      const tableData = [];

      response.units.forEach(unit => {
        const unitData = {
          name: unit.Unit,
          id: unit.Unit_Code || unit.Unit,
          level: 1,
          totalEmpCnt: unit.Total_Emp_Cnt || 0,
          absenteeism: null,
          attrition: null,
          skillBalance: null,
          children: []
        };

        unit.zones.forEach(zone => {
          const zoneData = {
            name: `Zone ${zone.Zone}`,
            id: `${unitData.id}-z${zone.Zone}`,
            level: 2,
            parent: unitData.id,
            totalEmpCnt: zone.Total_Emp_Cnt || 0,
            absenteeism: null,
            attrition: null,
            skillBalance: null,
            children: []
          };

          zone.data.forEach(item => {
            departments.add(item.Department);
            designations.add(item.Designation);
            departmentCounts[item.Department] = (departmentCounts[item.Department] || 0) + parseInt(item.Emp_Cnt || 0);

            if (!designationCountsPerDept[item.Department]) {
              designationCountsPerDept[item.Department] = {};
            }
            designationCountsPerDept[item.Department][item.Designation] = 
              (designationCountsPerDept[item.Department][item.Designation] || 0) + parseInt(item.Emp_Cnt || 0);

            if (!trendData[item.Edate]) {
              trendData[item.Edate] = {};
            }
            if (!trendData[item.Edate][item.Department]) {
              trendData[item.Edate][item.Department] = {};
            }
            if (!trendData[item.Edate][item.Department][item.Designation]) {
              trendData[item.Edate][item.Department][item.Designation] = {};
            }
            const grade = item.Grade || '-';
            if (!trendData[item.Edate][item.Department][item.Designation][grade]) {
              trendData[item.Edate][item.Department][item.Designation][grade] = 0;
            }
            trendData[item.Edate][item.Department][item.Designation][grade] += parseInt(item.Emp_Cnt || 0);

            zoneData.children.push({
              name: `${item.Designation} (${grade})`,
              id: `${zoneData.id}-l${item.Line_No}`,
              level: 3,
              parent: zoneData.id,
              totalEmpCnt: parseInt(item.Emp_Cnt || 0),
              absenteeism: null,
              attrition: null,
              skillBalance: null,
              designation: item.Designation,
              department: item.Department,
              grade: grade,
              lineNo: item.Line_No,
              unit: item.Unit,
              zone: item.Zone
            });
          });

          unitData.children.push(zoneData);
        });

        tableData.push(unitData);
      });

      return {
        departments: Array.from(departments),
        designations: Array.from(designations),
        departmentCounts,
        designationCountsPerDept,
        trendData,
        tableData
      };
    }

    // Update fetchData to process and update charts/table
    function fetchData(type = "", fromDate = "", toDate = "") {
      $("#loadSpinner").show();

      $.ajax({
        url: '/directlabour_details',
        method: 'GET',
        data: {
          type: type,
          fromdate: fromDate,
          enddate: toDate,
        },
        success: function(response) {
          $("#loadSpinner").hide();
          console.log("API Response:", response);
          if (response && response.units && Array.isArray(response.units) && response.units.length > 0) {
            apiData = processApiData(response);
            updateBarLevel('root');
            updateLineChart();
            updateDrilldownTable();
            updateStrengthKPI();
          } else {
            console.warn("No valid data in response:", response);
            apiData = null;
          }
        },
        error: function(jqXHR, textStatus, errorThrown) {
          console.error("Failed to fetch data:", textStatus, errorThrown, jqXHR.responseText);
          $("#loadSpinner").hide();
          apiData = null;
        }
      });
    }

    // Dynamic BAR_DRILL structure
    function getBarDrillData() {
      if (!apiData) return { root: { labels: [], data: [] } };

      const barDrill = {
        root: {
          labels: apiData.departments,
          data: apiData.departments.map(d => apiData.departmentCounts[d] || 0)
        }
      };

      apiData.departments.forEach(dept => {
        const desigs = Object.keys(apiData.designationCountsPerDept[dept] || {});
        barDrill[dept] = {
          labels: desigs,
          data: desigs.map(d => apiData.designationCountsPerDept[dept][d] || 0)
        };
      });

      return barDrill;
    }

    // Update Bar Chart
    // Plugin to enforce a minimum visible bar height in pixels
    const MIN_BAR_PX = 10;
    const minBarPlugin = {
      id: 'minBarPlugin',
      afterDatasetsDraw(chart, args, pluginOptions) {
        const { ctx } = chart;
        chart.data.datasets.forEach((ds, dsIndex) => {
          const meta = chart.getDatasetMeta(dsIndex);
          if (!meta || meta.type !== 'bar') return;
          meta.data.forEach((bar) => {
            if (!bar) return;
            const props = bar.getProps(['x','y','base','width','height','options'], true);
            const actualH = Math.abs(props.base - props.y);
            if (actualH > 0 && actualH < MIN_BAR_PX) {
              const isPositive = props.y < props.base;
              const drawY = isPositive ? props.base - MIN_BAR_PX : props.base;
              ctx.save();
              ctx.fillStyle = props.options.backgroundColor || '#0B3B91';
              ctx.fillRect(props.x - props.width / 2, drawY, props.width, MIN_BAR_PX);
              ctx.restore();
            }
          });
        });
      }
    };
    const barCtx = document.getElementById('barChart').getContext('2d');
    const barChart = new Chart(barCtx, {
      type: 'bar',
      data: {
        labels: [],
        datasets: [{
          data: [],
          backgroundColor: '#0B3B91',
          minBarLength: 10
        }]
      },
      plugins: [minBarPlugin],
      options: {
        maintainAspectRatio: false,
        elements: { bar: { minBarLength: 10 } },
        plugins: { legend: { display: false } },
        scales: {
          x: { grid: { display: false } },
          y: { beginAtZero: true, grid: { color: '#f1f5f9' } }
        }
      }
    });

    function updateBarLevel(key) {
      const node = getBarDrillData()[key];
      if (!node) return;
      barChart.data.labels = node.labels;
      barChart.data.datasets[0].data = node.data;
      barChart.data.datasets[0].backgroundColor = node.labels.map(() => '#0B3B91');
      barChart.data.datasets[0].minBarLength = barChart.data.datasets[0].minBarLength || 10;
      barChart.update();
    }

    // Update Line Chart
    const lineCtx = document.getElementById('lineChart').getContext('2d');
    const lineChart = new Chart(lineCtx, {
      type: 'line',
      data: {
        labels: [],
        datasets: [{
          label: 'Recruitment',
          data: [],
          borderColor: '#2563eb',
          backgroundColor: '#2563eb33',
          tension: 0.4,
          fill: true,
          pointRadius: 0,
          pointHoverRadius: 7,
          pointBackgroundColor: '#2563eb',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 2
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: { display: false },
          tooltip: {
            enabled: true,
            mode: 'index',
            intersect: false,
            displayColors: false
          }
        },
        interaction: { mode: 'index', intersect: false },
        scales: {
          x: { grid: { color: '#f1f5f9' } },
          y: { grid: { color: '#f1f5f9' }, beginAtZero: true }
        }
      }
    });

    function updateLineChart(selectedDepartment = null, selectedDesignation = null) {
      if (!apiData) return;
      const dates = Object.keys(apiData.trendData).sort();
      lineChart.data.labels = dates.map(date => moment(date).format('DD-MM-YY'));
      lineChart.data.datasets[0].data = dates.map(date => {
        let sum = 0;
        const dateData = apiData.trendData[date] || {};
        if (selectedDepartment && selectedDesignation) {
          const deptData = dateData[selectedDepartment] || {};
          const desData = deptData[selectedDesignation] || {};
          sum = Object.values(desData).reduce((s, c) => s + (c || 0), 0);
        } else if (selectedDepartment) {
          const deptData = dateData[selectedDepartment] || {};
          sum = Object.values(deptData).reduce((sd, des) => sd + Object.values(des).reduce((s, c) => s + (c || 0), 0), 0);
        } else {
          sum = Object.values(dateData).reduce((sd, dept) => sd + Object.values(dept).reduce((s, des) => s + Object.values(des).reduce((sc, c) => sc + (c || 0), 0), 0), 0);
        }
        return sum;
      });
      lineChart.update();

      // Update simple sparkline canvases with shaded fill
      const series = lineChart.data.datasets[0].data;
      renderSparkline('spark1', series, '#22c55e');
      renderSparkline('spark2', series, '#22c55e');
      renderSparkline('spark3', series, '#22c55e');
      renderSparkline('spark4', series, '#22c55e');
    }

    // Lightweight canvas sparkline drawer (no Chart.js)
    function renderSparkline(canvasId, series, strokeColor = '#22c55e') {
      const canvas = document.getElementById(canvasId);
      if (!canvas || !series) return;

      const ctx = canvas.getContext('2d');
      if (!canvas.style.width) canvas.style.width = '100%';
      if (!canvas.style.height) canvas.style.height = '36px';
      const parentW = canvas.parentElement ? canvas.parentElement.clientWidth : 120;
      const W = canvas.clientWidth || parentW || 120;
      const H = canvas.clientHeight || 36;
      const dpr = window.devicePixelRatio || 1;
      canvas.width = Math.max(100, W) * dpr;
      canvas.height = Math.max(40, H) * dpr;
      ctx.scale(dpr, dpr);

      const data = series.slice(-12);
      ctx.clearRect(0, 0, canvas.width, canvas.height);

      const pad = 2;
      const w = (canvas.width / dpr) - pad * 2;
      const h = (canvas.height / dpr) - pad * 2;
      const min = Math.min(...data);
      const max = Math.max(...data);
      const range = max - min || 1;

      const stepX = data.length > 1 ? w / (data.length - 1) : w;

      const pts = data.map((v, i) => {
        const x = pad + i * stepX;
        const y = pad + h - ((v - min) / range) * h;
        return { x, y };
      });

      const grad = ctx.createLinearGradient(0, pad, 0, pad + h);
      grad.addColorStop(0, 'rgba(34, 197, 94, 0.25)');
      grad.addColorStop(1, 'rgba(34, 197, 94, 0.05)');

      ctx.beginPath();
      if (pts.length) {
        ctx.moveTo(pts[0].x, pad + h);
        pts.forEach(p => ctx.lineTo(p.x, p.y));
        ctx.lineTo(pts[pts.length - 1].x, pad + h);
        ctx.closePath();
        ctx.fillStyle = grad;
        ctx.fill();
      }

      ctx.beginPath();
      pts.forEach((p, i) => {
        if (i === 0) ctx.moveTo(p.x, p.y);
        else ctx.lineTo(p.x, p.y);
      });
      ctx.strokeStyle = strokeColor;
      ctx.lineWidth = 2;
      ctx.stroke();
    }

    // Update Strength KPI
    function updateStrengthKPI(selectedDepartment = null, selectedDesignation = null) {
      if (!apiData) return;
      let totalStrength = 0;
      if (selectedDepartment && selectedDesignation) {
        totalStrength = apiData.tableData.reduce((sum, unit) => {
          return sum + unit.children.reduce((zs, zone) => {
            return zs + zone.children.reduce((ds, item) => {
              if (item.department === selectedDepartment && item.designation === selectedDesignation) {
                return ds + (item.totalEmpCnt || 0);
              }
              return ds;
            }, 0);
          }, 0);
        }, 0);
      } else if (selectedDepartment) {
        totalStrength = apiData.departmentCounts[selectedDepartment] || 0;
      } else {
        totalStrength = apiData.tableData.reduce((s, u) => s + (u.totalEmpCnt || 0), 0);
      }
      document.getElementById('strengthVal').textContent = totalStrength;
    }

    // Update Drilldown Table with full hierarchy
    function updateDrilldownTable(selectedDepartment = null, selectedDesignation = null) {
      if (!apiData || !useAccordionTable) return;
      const tbody = document.querySelector('#drilldownTable tbody');
      tbody.innerHTML = '';

      apiData.tableData.forEach(unit => {
        const filteredZones = [];

        unit.children.forEach(zone => {
          const filteredDesignations = zone.children.filter(item => 
            (!selectedDepartment || item.department === selectedDepartment) &&
            (!selectedDesignation || item.designation === selectedDesignation)
          );

          if (filteredDesignations.length > 0) {
            filteredZones.push({ zone, filteredDesignations });
          }
        });

        if (filteredZones.length > 0) {
          const unitRow = document.createElement('tr');
          unitRow.className = `accordion-row level-1 cursor-pointer`;
          unitRow.dataset.toggle = 'true';
          unitRow.dataset.level = '1';
          unitRow.dataset.id = unit.id;
          unitRow.innerHTML = `
            <td class="p-2 name-cell">
              <i class="fa-solid fa-chevron-right caret-icon"></i>
              <span class="font-medium">${unit.name}</span>
              <span class="badge badge-red">2 WIP</span>
              <span class="badge badge-yellow">2 Stops</span>
            </td>
            <td class="py-2">${filteredZones.reduce((sum, z) => sum + z.filteredDesignations.reduce((s, d) => s + d.totalEmpCnt, 0), 0)}</td>
            <td class="py-2">${unit.absenteeism || '-'}</td>
            <td class="py-2">${unit.attrition || '-'}</td>
            <td class="py-2">${unit.skillBalance || '-'}</td>
          `;
          tbody.appendChild(unitRow);

          filteredZones.forEach(({ zone, filteredDesignations }) => {
            const zoneRow = document.createElement('tr');
            zoneRow.className = `accordion-row level-2 hidden`;
            zoneRow.dataset.toggle = 'true';
            zoneRow.dataset.level = '2';
            zoneRow.dataset.id = zone.id;
            zoneRow.dataset.parent = unit.id;
            zoneRow.innerHTML = `
              <td class="py-2 pl-6 name-cell">
                <i class="fa-solid fa-chevron-right caret-icon"></i>
                <span class="font-medium">${zone.name}</span>
                <span class="badge badge-red">6 WIP</span>
                <span class="badge badge-yellow">8 Stops</span>
              </td>
              <td class="py-2">${filteredDesignations.reduce((sum, d) => sum + d.totalEmpCnt, 0)}</td>
              <td class="py-2">${zone.absenteeism || '-'}</td>
              <td class="py-2">${zone.attrition || '-'}</td>
              <td class="py-2">${zone.skillBalance || '-'}</td>
            `;
            tbody.appendChild(zoneRow);

            const lineMap = {};
            filteredDesignations.forEach(d => {
              const ln = (d.lineNo || 'NA').toString();
              if (!lineMap[ln]) lineMap[ln] = [];
              lineMap[ln].push(d);
            });

            Object.keys(lineMap).forEach(lineNo => {
              const lineId = `${zone.id}-line-${String(lineNo).replace(/\s+/g,'_').toLowerCase()}`;
              const lineItems = lineMap[lineNo];
              const lineTotal = lineItems.reduce((s, it) => s + (it.totalEmpCnt || 0), 0);

              const lineRow = document.createElement('tr');
              lineRow.className = `accordion-row level-3 hidden`;
              lineRow.dataset.toggle = 'true';
              lineRow.dataset.level = '3';
              lineRow.dataset.id = lineId;
              lineRow.dataset.parent = zone.id;
              lineRow.innerHTML = `
                <td class="py-2 pl-12 name-cell">
                  <i class="fa-solid fa-chevron-right caret-icon"></i>
                  <span class="font-medium">Line ${lineNo}</span>
                </td>
                <td class="py-2">${lineTotal}</td>
                <td class="py-2">-</td>
                <td class="py-2">-</td>
                <td class="py-2">-</td>
              `;
              tbody.appendChild(lineRow);

              const deptTotals = {};
              lineItems.forEach(it => {
                const dept = (it.department || 'Unknown').toString();
                if (!deptTotals[dept]) deptTotals[dept] = 0;
                deptTotals[dept] += (it.totalEmpCnt || 0);
              });

              Object.keys(deptTotals).forEach(deptName => {
                const deptRow = document.createElement('tr');
                deptRow.className = `accordion-row level-4 hidden`;
                deptRow.dataset.level = '4';
                deptRow.dataset.parent = lineId;
                deptRow.innerHTML = `
                  <td class="py-2 pl-16 name-cell"><span></span> ${deptName}</td>
                  <td class="py-2">${deptTotals[deptName]}</td>
                  <td class="py-2">-</td>
                  <td class="py-2">-</td>
                  <td class="py-2">-</td>
                `;
                tbody.appendChild(deptRow);
              });
            });
          });
        }
      });

      // Icons use Font Awesome; no init required
    }

    // Bar Chart Drilldown Logic
    let barDrillPath = ['root'];
    let selectedBarIndex = null;
    const barCrumbEl = document.getElementById('barCrumb');

    function setBarCrumb() {
      const parts = [];
      parts.push(`<span class="text-blue-600 cursor-pointer" data-idx="0">All Departments</span>`);

      let parentLabel = null;
      if (barDrillPath.length > 1) {
        parentLabel = (barDrillPath[1] || '').toString().toUpperCase();
      } else if (barDrillPath.length === 1 && selectedBarIndex !== null) {
        const labels = (barChart && barChart.data && Array.isArray(barChart.data.labels)) ? barChart.data.labels : [];
        if (labels[selectedBarIndex] !== undefined) {
          parentLabel = String(labels[selectedBarIndex]).toUpperCase();
        }
      }

      if (parentLabel) {
        let parentHtml = `<span class="text-blue-600 cursor-pointer" data-idx="1">${parentLabel}</span>`;
        if (barDrillPath.length > 1 && selectedBarIndex !== null) {
          const node = getBarDrillData()[barDrillPath[1]];
          const child = node && node.labels && node.labels[selectedBarIndex] ? node.labels[selectedBarIndex] : '';
          if (child) {
            parentHtml = `${parentHtml} <span class="text-slate-400"> > </span> <span>${child}</span>`;
          }
        }
        parts.push(parentHtml);
      }

      const html = parts.length === 1
        ? parts[0]
        : parts[0] + ' <span class="text-slate-400">›</span> ' + parts.slice(1).join('');
      barCrumbEl.innerHTML = html;
    }

    document.getElementById('barChart').addEventListener('click', function (evt) {
      const points = barChart.getElementsAtEventForMode(evt, 'nearest', { intersect: true }, false);
      if (!points.length) return;
      const idx = points[0].index;
      const label = barChart.data.labels[idx];
      const barDrill = getBarDrillData();

      if (barDrillPath.length === 1 && barDrill[label]) {
        barDrillPath.push(label);
        setBarCrumb();
        updateBarLevel(label);
        selectedBarIndex = null;
        updateLineChart(label);
        updateDrilldownTable(label);
        updateStrengthKPI(label);
      } else if (barDrillPath.length === 1 && !barDrill[label]) {
        selectedBarIndex = idx;
        barChart.data.datasets[0].backgroundColor = barChart.data.labels.map((_, i) => (i === idx ? '#0B3B91' : '#e5e7eb'));
        barChart.update();

        const selectedDepartment = barChart.data.labels[idx];
        setBarCrumb();
        updateLineChart(selectedDepartment);
        updateDrilldownTable(selectedDepartment);
        updateStrengthKPI(selectedDepartment);
      } else if (barDrillPath.length === 2) {
        highlightSecondLevel(idx);
      }
    });

    function highlightSecondLevel(idx) {
      const node = getBarDrillData()[barDrillPath[1]];
      if (!node) return;
      selectedBarIndex = idx;
      barChart.data.datasets[0].backgroundColor = barChart.data.labels.map((_, i) => (i === idx ? '#0B3B91' : '#e5e7eb'));
      barChart.update();

      const selectedDepartment = barDrillPath[1];
      const selectedDesignation = node.labels[idx];
      updateLineChart(selectedDepartment, selectedDesignation);
      updateDrilldownTable(selectedDepartment, selectedDesignation);
      updateStrengthKPI(selectedDepartment, selectedDesignation);
      setBarCrumb();
    }

    barCrumbEl.addEventListener('click', function(e) {
      const el = e.target.closest('span[data-idx]');
      if (!el) return;
      const idx = parseInt(el.dataset.idx);

      if (idx === barDrillPath.length - 1 && selectedBarIndex !== null) {
        selectedBarIndex = null;
        setBarCrumb();
        const key = barDrillPath[barDrillPath.length - 1];
        updateBarLevel(key === 'root' ? 'root' : key);
        const selectedDepartment = (barDrillPath.length > 1) ? key : null;
        updateLineChart(selectedDepartment);
        updateDrilldownTable(selectedDepartment);
        updateStrengthKPI(selectedDepartment);
        return;
      }

      if (idx < barDrillPath.length - 1) {
        barDrillPath = barDrillPath.slice(0, idx + 1);
        selectedBarIndex = null;
        setBarCrumb();
        const key = barDrillPath[barDrillPath.length - 1];
        updateBarLevel(key === 'root' ? 'root' : key);
        const selectedDepartment = (barDrillPath.length > 1) ? key : null;
        updateLineChart(selectedDepartment);
        updateDrilldownTable(selectedDepartment);
        updateStrengthKPI(selectedDepartment);
      }
    });

    function resetBarDrill() {
      barDrillPath = ['root'];
      selectedBarIndex = null;
      setBarCrumb();
      updateBarLevel('root');
      updateLineChart();
      updateDrilldownTable();
      updateStrengthKPI();
    }

    // Function to get URL parameters
    function getUrlParameter(name) {
        name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
        const regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
        const results = regex.exec(location.search);
        return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
    }

    // Parse and apply dates from URL on page load
    function applyDatesFromUrl() {
        const fromDateParam = getUrlParameter('fromdate');
        const endDateParam = getUrlParameter('enddate');

        if (fromDateParam && endDateParam) {
          const initialRouteStart = moment(fromDateParam, 'YYYY-MM-DD');
          const initialRouteEnd = moment(endDateParam, 'YYYY-MM-DD');
		  selectedfromdate = initialRouteStart;
		  selectedenddate = initialRouteEnd;
          // Set initial breadcrumb link with initial dates
          const initialBreadcrumbUrl = `/production_gar_dashboard?fromdate=${initialRouteStart.format('YYYY-MM-DD')}&enddate=${initialRouteEnd.format('YYYY-MM-DD')}`;
          $('#productionBreadcrumb').attr('href', initialBreadcrumbUrl);
        }
    }

    $(document).ready(function() {
      applyDatesFromUrl();
      
      $("#kpi1").on('click', () => resetBarDrill());
      $("#kpi2").on('click', () => resetBarDrill());
      $("#kpi3").on('click', () => resetBarDrill());
      $("#kpi4").on('click', () => resetBarDrill());

      if (useAccordionTable) {
        // Helper: hide all descendants for a given parent id and reset their carets
        function hideDescendants(rootId) {
          const stack = [rootId];
          while (stack.length) {
            const pid = stack.pop();
            const rows = Array.from(document.querySelectorAll(`#drilldownTable tbody tr[data-parent="${pid}"]`));
            rows.forEach(r => {
              r.classList.add('hidden');
              const caret = r.querySelector('.caret-icon');
              if (caret) {
                caret.classList.remove('fa-chevron-down');
                caret.classList.add('fa-chevron-right');
              }
              const cid = r.getAttribute('data-id');
              if (cid) stack.push(cid);
            });
          }
        }

        $('#drilldownTable tbody').on('click', 'tr[data-toggle="true"]', function() {
          const id = this.getAttribute('data-id');
          const caretEl = this.querySelector('.caret-icon');
          const isExpanded = caretEl && caretEl.classList.contains('fa-chevron-down');
          const children = Array.from(document.querySelectorAll(`#drilldownTable tbody tr[data-parent="${id}"]`));

          if (caretEl) {
            caretEl.classList.toggle('fa-chevron-right', isExpanded);
            caretEl.classList.toggle('fa-chevron-down', !isExpanded);
          }

          if (children.length) {
            const show = !isExpanded;
            if (show) {
              children.forEach(row => row.classList.remove('hidden'));
            } else {
              hideDescendants(id);
            }
          }
        });
      }

      $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: moment(),
        endDate: moment(),
        maxDate: moment(),
        ranges: {
          'Today': [moment(), moment()],
          'Last Week': [moment().subtract(6, 'days'), moment()],
          'This Month': [moment().startOf('month'), moment().endOf('month')]
        },
        locale: {
          format: 'DD-MM-YY'
        }
      }, function(start, end, label) {
        const startFormatted = start.format('YYYY-MM-DD');
        const endFormatted = end.format('YYYY-MM-DD');
		selectedfromdate = startFormatted;
		selectedenddate = endFormatted;
        const displayStartFormatted = start.format('DD-MM-YY');
        const displayEndFormatted = end.format('DD-MM-YY');
        selectedRangeLabel = label;

        const titleText = `
          Direct Labour Strength for - 
          <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
              GARMENTS
          </span>
          <span class="ml-4" style="color: #fff; font-weight: 500;">
              ${displayStartFormatted} - ${displayEndFormatted}
          </span>
        `;

        $('.dashboard_title').html(titleText);
        $('#dateRangePicker').val(`${displayStartFormatted} - ${displayEndFormatted}`);
        // Update breadcrumb with selected date range
        const breadcrumbUrl = `/production_gar_dashboard?fromdate=${startFormatted}&enddate=${endFormatted}`;
        $('#productionBreadcrumb').attr('href', breadcrumbUrl);
        fetchData(selectedRangeLabel, startFormatted, endFormatted);
      });
      

      const initialStart = moment();
      const initialEnd = moment();
      const initialStartFormatted = initialStart.format('YYYY-MM-DD');
      const initialEndFormatted = initialEnd.format('YYYY-MM-DD');
	  selectedfromdate = initialStartFormatted;
	  selectedenddate = initialEndFormatted;
      const initialDisplayStartFormatted = initialStart.format('DD-MM-YY');
      const initialDisplayEndFormatted = initialEnd.format('DD-MM-YY');
      $('.dashboard_title').html(`
        Direct Labour Strength for - 
          <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
            GARMENTS
          </span>
          <span class="ml-4" style="color: #fff; font-weight: 500;">
            ${initialDisplayStartFormatted} - ${initialDisplayEndFormatted}
          </span>
      `);
      selectedRangeLabel = 'Today';
      $('#dateRangePicker').val('Today');
    
      fetchData(selectedRangeLabel, initialStartFormatted, initialEndFormatted);

      $('#HistoricalButton').on('click', function () {
        window.location.href = `/history_dashboard/Labour?fromdate=${selectedfromdate}&enddate=${selectedenddate}`;
      });
    });
  </script>
</body>
</html>