<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Direct Labour Strength</title>

  <!-- Tailwind CSS -->
  <script src="https://cdn.tailwindcss.com"></script>
  <!-- jQuery -->
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <!-- Font Awesome CSS -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

  <style>
    body {
      background-color: #f8fafc;
      font-family: 'Inter', sans-serif;
    }
    .card {
      background: white;
      border-radius: 0.75rem;
      box-shadow: 0 1px 3px rgba(0,0,0,0.05);
      padding: 1.25rem;
    }
    .kpi-spark {
      height: 20px; /* Reduced height */
      width: 60px;  /* Reduced width */
      max-width: 100%; /* Ensure it doesn't exceed container */
      overflow: hidden; /* Prevent overflow */
    }
    .kpi-content {
      display: flex;
      justify-content: space-between;
      align-items: center;
      width: 100%;
    }
    .table-row {
      transition: background 0.2s ease;
    }
    .table-row:hover {
      background-color: #f9fafb;
    }
    .chart-container {
      position: relative;
      height: 260px;
    }
  </style>
</head>

<body class="min-h-screen flex flex-col">
  <!-- Top Bar -->
  <div class="bg-[#0B3B91] text-white px-6 py-3 flex items-center justify-between">
    <div class="flex items-center space-x-2 text-lg font-semibold">
      <span>Direct Labour Strength</span>
    </div>
    <div class="flex items-center space-x-3">
      <div class="bg-[#e6ecf8] text-[#0B3B91] px-5 py-2.5 rounded-md text-sm font-medium border border-[#cdd8f0] flex items-center justify-between w-52 shadow-sm cursor-default">
        <span>Today</span>
        <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" 
             stroke-width="1.5" stroke="currentColor" 
             class="w-4 h-4 text-[#0B3B91]">
          <path stroke-linecap="round" stroke-linejoin="round" d="M6 9l6 6 6-6" />
        </svg>
      </div>           
      <button class="bg-[#0B3B91] border border-white/30 rounded-md p-2">
        <svg xmlns="http://www.w3.org/2000/svg" 
             fill="none" viewBox="0 0 24 24" stroke="currentColor" 
             class="w-5 h-5 text-white">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                d="M14.857 17.243A4.001 4.001 0 0 1 12 21a4.001 4.001 0 0 1-2.857-3.757M18 8a6 6 0 1 0-12 0c0 7-3 9-3 9h18s-3-2-3-9z" />
        </svg>
      </button>
      
    </div>
  </div>

  <!-- Breadcrumb -->
  <div class="px-6 py-3 text-sm text-gray-500">
    Dashboard / <span class="text-gray-800 font-medium">Direct Labour Strength</span>
  </div>

  <!-- KPI Cards -->
  <div class="px-6 grid grid-cols-5 gap-4">
    <!-- Strength -->
    <div class="card cursor-pointer" id="kpi1">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Strength</div>
          <div class="text-3xl font-bold mt-1" id="strengthVal">455</div>
          <div class="text-xs text-gray-500 mt-1">▼ 2.2% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 480</div>
        </div>
        <canvas id="spark1" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Absenteeism -->
    <div class="card cursor-pointer" id="kpi2">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Absenteeism</div>
          <div class="text-3xl font-bold mt-1" id="absentVal">3.5%</div>
          <div class="text-xs text-red-500 mt-1">▲ 10.5% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 4.0%</div>
        </div>
        <canvas id="spark2" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Attrition -->
    <div class="card cursor-pointer" id="kpi3">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Attrition</div>
          <div class="text-3xl font-bold mt-1" id="attritionVal">1.2%</div>
          <div class="text-xs text-red-500 mt-1">▲ 5.2% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 1.5%</div>
        </div>
        <canvas id="spark3" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Recruitment -->
    <div class="card cursor-pointer" id="kpi4">
      <div class="kpi-content">
        <div>
          <div class="text-sm text-gray-600">Recruitment</div>
          <div class="text-3xl font-bold mt-1" id="recruitVal">15</div>
          <div class="text-xs text-green-500 mt-1">▼ 5.0% vs last month</div>
          <div class="text-xs text-gray-500 mt-1">Target: 20</div>
        </div>
        <canvas id="spark4" class="kpi-spark"></canvas>
      </div>
    </div>
    <!-- Skill Balance Ratio -->
    <div class="card flex flex-col justify-center">
      <div class="text-sm text-gray-600">Skill Balance Ratio</div>
      <div class="text-xl font-semibold mt-1">1 : 5 : 10 : 4</div>
      <div class="text-xs text-gray-500 mt-1">vs ideal 1 : 4 : 8 : 2</div>
    </div>
  </div>

  <!-- Charts -->
  <div class="px-6 mt-6 grid grid-cols-2 gap-4">
    <div class="card">
      <div class="font-semibold text-gray-700 mb-1">Overall Trend for Recruitment</div>
      <div class="text-sm text-gray-500 mb-3">Time-series data for the selected labour metric or category.</div>
      <div class="chart-container"><canvas id="lineChart"></canvas></div>
    </div>

    <div class="card">
      <div class="font-semibold text-gray-700 mb-1">Direct Labour Categories</div>
      <div class="text-sm text-gray-500 mb-3 hidden">Breakdown by labour category for Recruitment. Click 'Tailor' to drill down.</div>
      <div id="barCrumb" class="text-xs text-[var(--muted)] mt-2">ALL Categories</div>
      <div class="chart-container"><canvas id="barChart" class="cursor-pointer"></canvas></div>
    </div>
  </div>

  <!-- Line-wise Drilldown -->
  <div class="px-6 mt-6 mb-10">
    <div class="card">
      <div class="flex justify-between items-center mb-3">
        <div class="font-semibold text-gray-700">Line-wise Drilldown</div>
        <button class="text-sm text-gray-500">⤓</button>
      </div>
      <div class="text-sm text-gray-500 mb-3">Aggregated labour KPIs. Click on a unit or zone to expand.</div>

      <table class="w-full text-sm text-gray-700" id="drilldownTable">
        <thead class="text-gray-500 border-b">
          <tr>
            <th class="text-left py-2">Name</th>
            <th class="text-left py-2">Strength</th>
            <th class="text-left py-2">Absenteeism (%)</th>
            <th class="text-left py-2">Attrition (%)</th>
            <th class="text-left py-2">Skill Balance Ratio</th>
          </tr>
        </thead>
        <tbody>
          <tr class="table-row cursor-pointer drilldown" data-target="#unit1">
            <td class="py-2">▶ Unit 1</td>
            <td>240 <span class="text-red-500 text-xs">▼26</span></td>
            <td>3.0%</td>
            <td>1.5%</td>
            <td>1 : 1 : 9 : 0</td>
          </tr>
          <tr id="unit1" class="hidden bg-gray-50 text-gray-600 text-sm">
            <td colspan="5" class="py-2 pl-8">Zones: A1, A2, A3...</td>
          </tr>

          <tr class="table-row cursor-pointer drilldown" data-target="#unit2">
            <td class="py-2">▶ Unit 2</td>
            <td>238 <span class="text-red-500 text-xs">▼26</span></td>
            <td>3.0%</td>
            <td>1.5%</td>
            <td>2 : 7 : 14 : 3</td>
          </tr>
          <tr id="unit2" class="hidden bg-gray-50 text-gray-600 text-sm">
            <td colspan="5" class="py-2 pl-8">Zones: B1, B2, B3...</td>
          </tr>
        </tbody>
      </table>
    </div>
  </div>

  <script>
    // Icons now use Font Awesome; no initialization required
    // KPI Sparkline Charts
    const sparkOptions = {
      responsive: true,
      maintainAspectRatio: false, // Allow custom sizing
      elements: {
        line: { tension: 0.4, borderWidth: 2 },
        point: { radius: 0 }
      },
      plugins: { legend: { display: false } },
      scales: { x: { display: false }, y: { display: false } }
    };

    const sparkData = [
      { id: 'spark1', color: '#2563eb' },
      { id: 'spark2', color: '#ef4444' },
      { id: 'spark3', color: '#22c55e' },
      { id: 'spark4', color: '#10b981' },
    ];

    sparkData.forEach(s => {
      new Chart(document.getElementById(s.id), {
        type: 'line',
        data: {
          labels: [1, 2, 3, 4, 5, 6, 7],
          datasets: [{
            data: [3, 3.2, 3.3, 3.4, 3.8, 4, 4.5],
            borderColor: s.color,
            fill: false
          }]
        },
        options: sparkOptions
      });
    });

    // Main Line Chart
    const lineCtx = document.getElementById('lineChart').getContext('2d');
    const lineChart = new Chart(lineCtx, {
      type: 'line',
      data: {
        labels: ['Day 1', 'Day 2', 'Day 3', 'Day 4', 'Day 5', 'Day 6', 'Day 7'],
        datasets: [{
          label: 'Recruitment',
          data: [0, 1, 2, 4, 5, 8, 16],
          borderColor: '#2563eb',
          backgroundColor: '#2563eb33',
          tension: 0.4,
          fill: true,
          pointRadius: 0,
          pointHoverRadius: 7,
          pointBackgroundColor: '#2563eb',
          pointBorderColor: '#ffffff',
          pointBorderWidth: 2
        }]
      },
      options: {
        responsive: true,
        plugins: {
          legend: { display: false },
          tooltip: {
            enabled: true,
            mode: 'index',
            intersect: false,
            displayColors: false
          }
        },
        interaction: { mode: 'index', intersect: false },
        scales: {
          x: { grid: { color: '#f1f5f9' } },
          y: { grid: { color: '#f1f5f9' }, beginAtZero: true }
        }
      }
    });

    // Bar Chart
    const barCtx = document.getElementById('barChart').getContext('2d');
    const barChart = new Chart(barCtx, {
      type: 'bar',
      data: {
        labels: ['Auditor', 'Cutting', 'Helper', 'Tailor', 'Packers', 'Warehouse'],
        datasets: [{
          data: [2, 3, 5, 15, 5, 4],
          backgroundColor: '#0B3B91'
        }]
      },
      options: {
        plugins: { legend: { display: false } },
        scales: {
          x: { grid: { display: false } },
          y: { beginAtZero: true, grid: { color: '#f1f5f9' } }
        }
      }
    });

    // Bar chart drilldown data
    const BAR_DRILL = {
      root: {
        labels: ['Auditor','Cutting','Helper','Tailor','Packers','Warehouse'],
        data:   [2,3,5,15,5,4]
      },
      Tailor: {
        title: 'Tailor Grades',
        labels: ['A*','A','B','C'],
        data: [10,50,100,40],
        trend: [45,48,50,55,60,80,100],
        kpi: { strength: 455, absent: '3.2%', attrition: '1.1%', recruit: 12 },
        table: [
          ['A* Grade', '120', '2.8%', '1.0%', '1 : 1 : 10 : 1'],
          ['A Grade', '300', '3.1%', '1.2%', '2 : 6 : 12 : 3']
        ]
      },
      Cutting: {
        title: 'Cutting Types',
        labels: ['Manual','Auto'],
        data: [4,6],
        trend: [2,2,3,3,4,5,6],
        kpi: { strength: 430, absent: '3.8%', attrition: '1.3%', recruit: 10 },
        table: [
          ['Manual', '180', '3.6%', '1.4%', '1 : 2 : 8 : 1'],
          ['Auto', '200', '3.5%', '1.2%', '1 : 3 : 9 : 1']
        ]
      }
    };

    let barDrillPath = ['root'];
    let selectedBarIndex = null; // selected index at second level
    const barCrumbEl = document.getElementById('barCrumb');

    function setBarCrumb(){
      const names = barDrillPath.map((k,i)=> i===0 ? 'All Categories' : k);
      barCrumbEl.innerHTML = names.map((n,i)=>`<span class="${i<names.length-1?'text-blue-600 cursor-pointer':''}" data-idx="${i}">${n}</span>`).join(' <span class="text-slate-400">›</span> ');
    }

    function updateBarLevel(key){
      const node = BAR_DRILL[key];
      if (!node) return;
      barChart.data.labels = node.labels;
      barChart.data.datasets[0].data = node.data;
      // reset selection and colors
      selectedBarIndex = null;
      barChart.data.datasets[0].backgroundColor = node.labels.map(()=> '#0B3B91');
      barChart.update();
      if (node.trend){
        lineChart.data.datasets[0].data = node.trend;
        lineChart.update();
      }

      if (node.kpi){
        document.getElementById('strengthVal').textContent = node.kpi.strength;
        document.getElementById('absentVal').textContent = node.kpi.absent;
        document.getElementById('attritionVal').textContent = node.kpi.attrition;
        document.getElementById('recruitVal').textContent = node.kpi.recruit;
      }

      if (node.table){
        const tbody = document.querySelector('#drilldownTable tbody');
        if (tbody){
          tbody.innerHTML = '';
          node.table.forEach(r=>{
            const tr = document.createElement('tr');
            tr.className = 'table-row cursor-pointer drilldown';
            tr.innerHTML = `<td class="py-2">▶ ${r[0]}</td><td>${r[1]}</td><td>${r[2]}</td><td>${r[3]}</td><td>${r[4]}</td>`;
            tbody.appendChild(tr);
          })
        }
      }
    }

    // Click on bars to drill in
    function highlightSecondLevel(idx){
      const node = BAR_DRILL[barDrillPath[1]];
      if (!node) return;
      selectedBarIndex = idx;
      const navy = '#0B3B91';
      const pale = '#e5e7eb';
      barChart.data.datasets[0].backgroundColor = barChart.data.labels.map((_,i)=> i===idx ? navy : pale);
      barChart.update();

      // Update trend, KPIs and table
      const trendVariants = [
        [20,25,30,35,40,60,70],
        [30,35,40,45,50,70,90],
        [45,48,50,55,60,80,100],
        [25,28,30,35,40,55,65]
      ];
      lineChart.data.datasets[0].data = trendVariants[idx % trendVariants.length];
      lineChart.update();
      document.getElementById('strengthVal').textContent = 400 + idx*20;
      document.getElementById('absentVal').textContent = (3 + idx*0.2).toFixed(1) + '%';
      document.getElementById('attritionVal').textContent = (1 + idx*0.1).toFixed(1) + '%';
      document.getElementById('recruitVal').textContent = 10 + idx*2;
      const tbody = document.querySelector('#drilldownTable tbody');
      if (tbody){
        tbody.innerHTML = '';
        [['Segment '+barChart.data.labels[idx], 200+idx*20, '3.'+idx+'%', '1.'+idx+'%', '1 : 4 : 8 : 2']].forEach(r=>{
          const tr = document.createElement('tr');
          tr.className = 'table-row cursor-pointer drilldown';
          tr.innerHTML = `<td class=\"py-2\">▶ ${r[0]}</td><td>${r[1]}</td><td>${r[2]}</td><td>${r[3]}</td><td>${r[4]}</td>`;
          tbody.appendChild(tr);
        })
      }
    }

    document.getElementById('barChart').addEventListener('click', function(evt){
      const points = barChart.getElementsAtEventForMode(evt,'nearest',{intersect:true},false);
      if (!points.length) return;
      const idx = points[0].index;
      const label = barChart.data.labels[idx];
      if (barDrillPath.length === 1 && BAR_DRILL[label]){
        barDrillPath.push(label);
        setBarCrumb();
        updateBarLevel(label);
      } else if (barDrillPath.length === 1 && !BAR_DRILL[label]) {
        // Root level selection when no drill node exists
        selectedBarIndex = idx;
        const navy = '#0B3B91';
        const pale = '#e5e7eb';
        barChart.data.datasets[0].backgroundColor = barChart.data.labels.map((_,i)=> i===idx ? navy : pale);
        barChart.update();

        // mock different trends per root category index
        const rootTrendVariants = [
          [10,12,13,15,16,17,18],
          [12,13,15,16,18,19,20],
          [14,15,16,18,19,20,22],
          [16,18,20,22,24,26,28],
          [12,13,14,15,15,16,17],
          [11,12,13,14,14,15,16]
        ];
        lineChart.data.datasets[0].data = rootTrendVariants[idx % rootTrendVariants.length];
        lineChart.update();

        // update KPIs simple mock based on selection
        document.getElementById('strengthVal').textContent = 380 + idx*15;
        document.getElementById('absentVal').textContent = (2.8 + idx*0.3).toFixed(1) + '%';
        document.getElementById('attritionVal').textContent = (0.9 + idx*0.1).toFixed(1) + '%';
        document.getElementById('recruitVal').textContent = 8 + idx*2;

        const tbody = document.querySelector('#drilldownTable tbody');
        if (tbody){
          tbody.innerHTML = '';
          [[label+' Segment', 180+idx*25, (2.8+idx*0.3).toFixed(1)+'%', (0.9+idx*0.1).toFixed(1)+'%', '1 : 4 : 8 : 2']].forEach(r=>{
            const tr = document.createElement('tr');
            tr.className = 'table-row cursor-pointer drilldown';
            tr.innerHTML = `<td class=\"py-2\">▶ ${r[0]}</td><td>${r[1]}</td><td>${r[2]}</td><td>${r[3]}</td><td>${r[4]}</td>`;
            tbody.appendChild(tr);
          })
        }
      } else if (barDrillPath.length === 2) {
        highlightSecondLevel(idx);
      }
    });

    // Click breadcrumb to drill out
    barCrumbEl.addEventListener('click', function(e){
      const el = e.target.closest('span[data-idx]');
      if (!el) return;
      const idx = parseInt(el.dataset.idx);
      if (idx === barDrillPath.length-1) return;
      barDrillPath = barDrillPath.slice(0, idx+1);
      setBarCrumb();
      const key = barDrillPath[barDrillPath.length-1];
      updateBarLevel(key === 'root' ? 'root' : key);
      // when drilling out to second level, keep last selection cleared
      if (barDrillPath.length === 1){
        selectedBarIndex = null;
      }
    });

    // Reset bar drill when KPI changes
    function resetBarDrill(){
      barDrillPath = ['root'];
      setBarCrumb();
      updateBarLevel('root');
    }

    // Mock datasets per KPI
    const KPI_MOCK = {
      kpi1: {
        line: [3,4,5,6,7,6,8],
        bar: [4,5,6,10,6,5],
        table: [
          ['Unit 1', '240', '3.0%', '1.5%', '1 : 1 : 9 : 0'],
          ['Unit 2', '238', '3.1%', '1.4%', '2 : 7 : 14 : 3']
        ]
      },
      kpi2: {
        line: [5,5,4.5,4,3.8,3.6,3.5],
        bar: [1,2,3,7,2,1],
        table: [
          ['Unit 1', '220', '3.8%', '1.6%', '1 : 2 : 8 : 1'],
          ['Unit 2', '215', '3.6%', '1.4%', '2 : 6 : 13 : 3']
        ]
      },
      kpi3: {
        line: [1.2,1.1,1.3,1.4,1.2,1.1,1.2],
        bar: [2,2,3,6,2,2],
        table: [
          ['Unit 1', '230', '3.1%', '1.2%', '1 : 1 : 8 : 1'],
          ['Unit 2', '225', '3.0%', '1.3%', '2 : 5 : 12 : 3']
        ]
      },
      kpi4: {
        line: [0,1,2,4,5,8,16],
        bar: [2,3,5,15,5,4],
        table: [
          ['Unit 1', '240', '3.0%', '1.5%', '1 : 1 : 9 : 0'],
          ['Unit 2', '238', '3.0%', '1.5%', '2 : 7 : 14 : 3']
        ]
      }
    };

    function updateDashboardForKPI(key){
      const mock = KPI_MOCK[key];
      if (!mock) return;

      // Update line chart
      lineChart.data.datasets[0].data = mock.line;
      lineChart.update();

      // Update bar chart
      barChart.data.datasets[0].data = mock.bar;
      barChart.update();

      // Update drilldown table body
      const tbody = document.querySelector('#drilldownTable tbody');
      if (tbody){
        tbody.innerHTML = '';
        mock.table.forEach(row => {
          const tr = document.createElement('tr');
          tr.className = 'table-row cursor-pointer drilldown';
          tr.innerHTML = `<td class="py-2">▶ ${row[0]}</td><td>${row[1]} <span class="text-red-500 text-xs">▼26</span></td><td>${row[2]}</td><td>${row[3]}</td><td>${row[4]}</td>`;
          tbody.appendChild(tr);
        });
      }
    }

    // KPI click bindings
    $(document).ready(function() {
      $("#kpi1").on('click', ()=> updateDashboardForKPI('kpi1'));
      $("#kpi2").on('click', ()=> updateDashboardForKPI('kpi2'));
      $("#kpi3").on('click', ()=> updateDashboardForKPI('kpi3'));
      $("#kpi4").on('click', ()=> updateDashboardForKPI('kpi4'));

      // reset bar drill when KPI toggles
      $("#kpi1,#kpi2,#kpi3,#kpi4").on('click', resetBarDrill);

      $(".drilldown").click(function() {
        const target = $(this).data("target");
        $(target).toggleClass("hidden");
      });
    });
  </script>
</body>
</html>