<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Production Socks Overview</title>
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- jQuery CDN -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    
    <!-- Chart.js CDN for sparklines -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>


    <!-- Date picker CDN -->
    <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
    <script type="text/javascript" src="https://cdn.jsdelivr.net/jquery/latest/jquery.min.js"></script>
    <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
    <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
    
    <!-- Font Awesome CSS -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

    
    <!-- Custom Tailwind Configuration -->
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'background': 'rgb(245, 246, 247)',
                        'foreground': 'rgb(31, 41, 55)',
                        'card': 'rgb(255, 255, 255)',
                        'card-foreground': 'rgb(31, 41, 55)',
                        'primary': 'rgb(15, 61, 126)',
                        'primary-foreground': 'rgb(248, 250, 252)',
                        'primary-100': 'rgb(219, 234, 254)',
                        'primary-50': 'rgb(239, 246, 255)',
                        'primary-25': 'rgb(248, 250, 252)',
                        'secondary': 'rgb(241, 245, 249)',
                        'muted': 'rgb(241, 245, 249)',
                        'muted-foreground': 'rgb(100, 116, 139)',
                        'border': 'rgb(226, 232, 240)',
                        'input': 'rgb(226, 232, 240)',
                        'ring': 'rgb(15, 61, 126)',
                    },
                    fontFamily: {
                        'body': ['Inter', 'system-ui', 'sans-serif'],
                    }
                }
            }
        }
    </script>
    
    <style>
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Oxygen', 'Ubuntu', 'Cantarell', 'Fira Sans', 'Droid Sans', 'Helvetica Neue', sans-serif;
        }
        
        /* Custom scrollbar */
        .custom-scrollbar::-webkit-scrollbar {
            width: 6px;
        }
        .custom-scrollbar::-webkit-scrollbar-track {
            background: #f1f5f9;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 3px;
        }
        
        /* Sparkline container */
        .sparkline-container {
            width: 80px;
            height: 40px;
        }
        
        /* Tooltip styles */
        .tooltip {
            position: absolute;
            z-index: 1000;
            background: rgb(15, 23, 42);
            color: white;
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 12px;
            pointer-events: none;
            opacity: 0;
            transition: opacity 0.2s;
        }
        
        .tooltip.show {
            opacity: 1;
        }

        #tooltip {
            position: absolute;
            z-index: 9999;
            background: #fff;
            color: #333;
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
            white-space: nowrap;
            display: none;
        }

        #tooltip.show {
            display: block;
        }
        
        /* Dropdown styles */
        .dropdown-menu {
            display: none;
            position: absolute;
            top: 100%;
            right: 0;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            z-index: 50;
            min-width: 200px;
            max-height: 240px;
            overflow-y: auto;
        }
        
        .dropdown-menu.show {
            display: block;
        }
        
        /* Animation for row highlighting */
        .highlight-row {
            background-color: rgb(219, 234, 254) !important;
            transition: background-color 0.5s ease;
        }
        
        /* Date picker custom styles */
        .date-picker input {
            background: rgba(248, 250, 252, 0.1);
            border: 1px solid rgba(248, 250, 252, 0.2);
            color: white;
        }
        
        .date-picker input::placeholder {
            color: rgba(248, 250, 252, 0.7);
        }
        
        /* Loading skeleton animation */
        @keyframes skeleton-loading {
            0% { background-color: #e2e8f0; }
            50% { background-color: #f1f5f9; }
            100% { background-color: #e2e8f0; }
        }
        
        .skeleton {
            animation: skeleton-loading 2s ease-in-out infinite;
            border-radius: 4px;
        }
        
        /* Status indicators */
        .status-dot-red {
            width: 8px;
            height: 8px;
            background-color: #ef4444;
            border-radius: 50%;
        }
        
        .status-dot-yellow {
            width: 8px;
            height: 8px;
            background-color: #facc15;
            border-radius: 50%;
        }

        /* Tailwind-based spinner */
        .spinner {
            display: inline-block;
            width: 1.5rem;
            height: 1.5rem;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-top-color: #fff;
            border-radius: 50%;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .active-card {
            background-color: #e6f3ff !important;
            border: 2px solid #2e66beff !important;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1) !important;
        }

    </style>
</head>
<body class="h-full bg-background text-foreground antialiased">
    <!-- Tooltip container -->
    <div id="tooltip" class="tooltip"></div>
    
    <div class="flex min-h-screen w-full flex-col">
       @include('LifeAtGainup.production.header')

        <!-- Main Content -->
        <main class="flex flex-1 flex-col gap-4 p-4 md:gap-8 md:p-8">
            <!-- KPI Header -->
            <div class="grid gap-4 md:grid-cols-2 lg:grid-cols-4 xl:grid-cols-6 mb-4" id="kpiGrid">
                <!-- KPI cards will be dynamically generated -->
            </div>

            <!-- Drilldown Table -->
            <div class="grid gap-4 md:gap-8 grid-cols-1">
                <div class="bg-card border border-border rounded-lg shadow-sm">
                    <!-- Card Header -->
                    <div class="flex flex-row items-start justify-between p-6 border-b border-border">
                        <div>
                            <h3 class="text-2xl font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Daily snapshot</h3>
                            <p class="text-sm text-muted-foreground mt-1.5">
                                Aggregated production data. Click on a unit or zone to expand.
                            </p>
                        </div>
                        <div class="flex items-center gap-2">
                            <!-- Filter Button -->
                            <div class="relative">
                                <button class="inline-flex items-center justify-center rounded-md border border-input bg-background hover:bg-muted h-10 px-4 py-2 text-sm font-medium transition-colors" id="filterBtn">
                                    <i class="fa-solid fa-filter mr-2 h-4 w-4"></i>
                                    Filters
                                    <span class="ml-2 bg-secondary text-secondary-foreground rounded-full h-5 w-5 text-xs flex items-center justify-center hidden" id="filterBadge">0</span>
                                </button>
                                
                                <!-- Filter Dropdown -->
                                <div class="dropdown-menu custom-scrollbar" id="filterDropdown">
                                    <div class="p-4">
                                        <div class="mb-4 relative">
                                            <div class="custom-dropdown" id="buyerDropdown">
                                                <button type="button" 
                                                    class="w-full border border-input rounded-md p-2 text-sm flex justify-between items-center bg-white"
                                                    id="buyerDropdownBtn">
                                                    <span id="buyerSelected" style="font-size: 10px;">Select Buyers</span>
                                                    <i class="fa-solid fa-chevron-down h-4 w-4"></i>
                                                </button>
                                                <ul class="absolute hidden bg-white border border-input rounded-md mt-1 w-full max-h-40 overflow-y-auto z-10" 
                                                    id="buyerDropdownMenu">
                                                    <!-- Populated dynamically -->
                                                </ul>
                                            </div>
                                        </div>

                                        <div class="mb-4 relative">
                                            <div class="custom-dropdown" id="ocnDropdown">
                                                <button type="button" 
                                                    class="w-full border border-input rounded-md p-2 text-sm flex justify-between items-center bg-white"
                                                    id="ocnDropdownBtn">
                                                    <span id="ocnSelected" style="font-size: 11px;">Select OCNs</span>
                                                    <i class="fa-solid fa-chevron-down h-4 w-4"></i>
                                                </button>
                                                <ul class="absolute hidden bg-white border border-input rounded-md mt-1 w-full max-h-40 overflow-y-auto z-10" 
                                                    id="ocnDropdownMenu">
                                                    <li class="px-3 py-2">Select at least one buyer</li>
                                                </ul>
                                            </div>
                                        </div>
                                        
                                        <button class="w-full bg-secondary hover:bg-secondary/80 text-sm py-2 px-4 rounded-md transition-colors" id="resetFiltersBtn">
                                            <i class="fa-solid fa-xmark mr-2 h-4 w-4 inline"></i>
                                            Reset All Filters
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Export Button -->
                            <button class="hidden inline-flex items-center justify-center rounded-md border border-input bg-background hover:bg-muted h-10 w-10 transition-colors" title="Export data">
                                <i class="fa-solid fa-upload h-4 w-4"></i>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Table Content -->
                    <div class="p-6">
                        <div class="relative w-full overflow-auto">
                            <table class="w-full caption-bottom text-sm">
                                <thead>
                                    <tr class="border-b border-border">
                                        <th class="h-12 px-4 text-left align-middle font-medium text-muted-foreground w-[350px]">Unit</th>
                                        <th class="h-12 px-4 text-left align-middle font-medium text-muted-foreground">Buyer/OCN</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Running Mch</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Cycle Time</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">No of opr</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Direct Labour</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">WIP</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Knit Qty</th>
                                        <th class="h-12 px-4 text-center align-middle font-medium text-muted-foreground w-[120px]">Knit Qty Trend</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Earned Minutes</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">OEE%</th>
                                        <th class="h-12 px-4 text-center align-middle font-medium text-muted-foreground w-[120px]">OEE Trend</th>
                                    </tr>
                                </thead>
                                <tbody id="drilldownTableBody">
                                    <!-- Table rows will be dynamically generated -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Notification Side Panel (hidden by default) -->
    <div class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden" id="notificationOverlay">
        <div class="fixed right-0 top-0 h-full w-80 bg-white shadow-lg transform translate-x-full transition-transform duration-300" id="notificationPanel">
            <div class="p-6 border-b border-border">
                <div class="flex items-center justify-between">
                    <h2 class="text-lg font-semibold">Production Alerts</h2>
                    <button class="text-muted-foreground hover:text-foreground" id="closeNotifications">
                        <i class="fa-solid fa-xmark h-5 w-5"></i>
                    </button>
                </div>
            </div>
            <div class="p-6">
                <p class="text-muted-foreground">No new alerts at this time.</p>
            </div>
        </div>
    </div>

<script>

    // Icons now use Font Awesome; no initialization required

    // Mock Data
    const mockKpiData = {
        backFeedAvail: {
            label: "SM Stock",
            value: 87,
            yesterday: 4,
            target: 100,
            unit: "",
            description: "Available back feed stock",
            trendData: [
                { value: 80 }, { value: 82 }, { value: 83 }, 
                { value: 84 }, { value: 85 }, { value: 86 }, { value: 87 }
            ]
        },
        capacityUtilization: {
            label: "Cap Utilization",
            value: 85,
            yesterday: 4,
            target: 95.0,
            unit: "%",
            description: "Current capacity utilization percentage",
            trendData: [
                { value: 87.5 }, { value: 88.1 }, { value: 88.8 }, 
                { value: 89.2 }, { value: 90.1 }, { value: 90.8 }, { value: 91.4 }
            ]
        },
        cuttingOutput: {
            label: "Cutting Output",
            value: 100,
            yesterday: 5,
            target: 14000,
            unit: "",
            description: "Total pieces cut today",
            trendData: [
                { value: 12200 }, { value: 12400 }, { value: 12650 }, 
                { value: 12834 }, { value: 13000 }, { value: 13120 }, { value: 13247 }
            ]
        },
        directLabourStrength: {
            label: "Direct Labour",
            value: 455,
            yesterday: 5,
            target: 1250,
            unit: "",
            description: "Number of direct labour workers present",
            trendData: [
                { value: 1235 }, { value: 1240 }, { value: 1242 }, 
                { value: 1238 }, { value: 1245 }, { value: 1246 }, { value: 1247 }
            ]
        },
        efficiency: {
            label: "Efficiency",
            value: 80,
            yesterday: 4,
            target: 90.0,
            unit: "%",
            description: "Overall production efficiency",
            trendData: [
                { value: 83.2 }, { value: 84.1 }, { value: 84.8 }, 
                { value: 85.1 }, { value: 86.2 }, { value: 86.9 }, { value: 87.3 }
            ]
        },
        output: {
            label: "Knitting prod",
            value: 120,
            yesterday: 8,
            target: 13000,
            unit: "",
            description: "Total pieces produced today",
            trendData: [
                { value: 11200 }, { value: 11400 }, { value: 11600 }, 
                { value: 11823 }, { value: 12100 }, { value: 12300 }, { value: 12547 }
            ]
        },
        lineStopMins: {
            label: "Line Stop Minutes",
            value: 200,
            yesterday: 4,
            target: 150,
            unit: "",
            description: "Total minutes lost due to line stops",
            trendData: [
                { value: 245 }, { value: 230 }, { value: 218 }, 
                { value: 203 }, { value: 195 }, { value: 189 }, { value: 184 }
            ]
        },
        ordersOnTrack: {
            label: "Orders On Track",
            value: 92,
            yesterday: 13,
            target: 98.0,
            unit: "%",
            description: "Percentage of orders on schedule",
            trendData: [
                { value: 91.2 }, { value: 91.8 }, { value: 92.1 }, 
                { value: 92.8 }, { value: 93.4 }, { value: 93.8 }, { value: 94.2 }
            ]
        },
        packingOutput: {
            label: "Pack Output",
            value: 215,
            yesterday: 3,
            target: 12500,
            unit: "",
            description: "Total pieces packed today",
            trendData: [
                { value: 10800 }, { value: 11000 }, { value: 11150 }, 
                { value: 11234 }, { value: 11450 }, { value: 11650 }, { value: 11847 }
            ]
        },
        wip: {
            label: "WIP",
            value: 9000,
            yesterday: 4,
            target: 2500,
            unit: "",
            description: "Work in progress on production lines",
            trendData: [
                { value: 3200 }, { value: 3150 }, { value: 3080 }, 
                { value: 3012 }, { value: 2950 }, { value: 2890 }, { value: 2847 }
            ]
            },
        merchandiser: {
            label: "Merchandiser",
            value: 95,
            yesterday: 5,
            target: 100,
            unit: "%",
            description: "Merchandiser performance metric",
            trendData: [
                { value: 88 }, { value: 90 }, { value: 91 }, 
                { value: 92 }, { value: 93 }, { value: 94 }, { value: 95 }
            ]
        },
        quality: {
            label: "Quality",
            value: 98,
            yesterday: 2,
            target: 99.5,
            unit: "%",
            description: "Quality assurance percentage",
            trendData: [
                { value: 96.5 }, { value: 97.0 }, { value: 97.2 }, 
                { value: 97.5 }, { value: 97.8 }, { value: 98.0 }, { value: 98.2 }
            ]
        }
    };

    // Chart utilities
    function createSparkline(containerId, data, color = '#3b82f6') {
        const canvas = document.getElementById(containerId);
        if (!canvas) return;
        
        const ctx = canvas.getContext('2d');
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: data.map((_, i) => i),
                datasets: [{
                    data: data.map(d => d.value),
                    borderColor: color,
                    backgroundColor: color + '20',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4,
                    pointRadius: 0,
                    pointHoverRadius: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    x: { display: false },
                    y: { display: false }
                },
                plugins: {
                    legend: { display: false },
                    tooltip: { enabled: false }
                },
                interaction: {
                    intersect: false
                },
                elements: {
                    point: { radius: 0 }
                }
            }
        });
    }

    // Initialize KPI Cards
    function initializeKpiCards() {
        const kpiGrid = $('#kpiGrid');
        kpiGrid.empty();
        
        const kpiOrder = [
            'cuttingOutput', 'backFeedAvail', 'output', 'lineStopMins', 'wip',
            'efficiency', 'directLabourStrength', 'packingOutput', 'ordersOnTrack', 'capacityUtilization', 'merchandiser', 'quality'
        ];
        
        kpiOrder.forEach(key => {
            const kpi = mockKpiData[key];
            if (!kpi) return;
            
            const diff = kpi.yesterday;
            const comparisonColor = diff >= 0 ? 'text-green-500' : 'text-red-500';
            const arrowIcon = diff >= 0 ? 'arrow-up' : 'arrow-down';
            
            let comparisonText;
            if (selectedRangeLabel === 'Today') comparisonText = 'vs yesterday';
            else if (selectedRangeLabel === 'Last Week') comparisonText = 'vs last week';
            else comparisonText = 'vs this month';
            
            const formatValue = value => {
            if (key === 'lineStopMins') return Math.round(value).toLocaleString();
            if (kpi.unit === '%') return `${value.toFixed(1)}${kpi.unit}`;
            return value.toLocaleString();
            };
            
            const trend = kpi.trendData;
            const isIncreasing = trend.length > 1 && trend[trend.length - 1].value > trend[0].value;
            const chartColor = isIncreasing ? '#22c55e' : (trend[trend.length - 1].value < trend[0].value ? '#ef4444' : '#3b82f6');
            
            // Segmented mini-panels for KPI cards
            let segmentedControl = '';
            if (key === 'output') {
                segmentedControl = `
                <div class="flex bg-gray-50 rounded-md border border-gray-200 px-2 py-1 gap-3 absolute top-10 left-3 ">
                    <div class="text-xs text-primary font-medium"><i class="fa-solid fa-boxes-stacked"></i></div>
                    <div class="text-xs  font-bold" id="output-sum" style="font-size: 14px;" title="Output">0</div>
                    <span class="text-xs text-gray-600 font-medium">|</span>
                    <div class="text-xs text-primary font-medium"><i class="fa-solid fa-clock"></i></div>
                    <div class="text-xs  font-bold" id="earned-minutes-sum" style="font-size: 14px;" title="Earned Minutes">0</div>
                </div>
                `;
            } else if (key === 'directLabourStrength') {
                segmentedControl = `
                <div class="flex bg-gray-50 rounded-md  px-2 py-1 gap-3 absolute top-10 left-3 ">
                    <div class="flex items-center gap-1 text-xs text-primary" title="Tailors">
                        <i class="fa-solid fa-users"></i>
                    </div>
                    <div class="text-xs font-bold" id="labour-tailor-sum" style="font-size: 14px;">0</div>
                    <span class="text-xs text-gray-600 font-medium">|</span>
                    <div class="flex items-center gap-1 text-xs text-primary" title="Direct Labour">
                        <i class="fa-solid fa-user"></i>
                    </div>
                    <div class="text-xs font-bold" id="labour-direct-sum" style="font-size: 14px;">0</div>
                    <span class="text-xs text-gray-600 font-medium">|</span>
                    <div class="flex items-center gap-1 text-xs text-primary" title="Unallocated">
                        <i class="fa-regular fa-circle"></i>
                    </div>
                    <div class="text-xs font-bold" id="labour-unalloc-sum" style="font-size: 14px;">00</div>
                </div>
                `;
            }

            const cardHtml = `
            <div class="bg-card border border-border rounded-lg shadow-sm flex-1 relative ${['output', 'lineStopMins'].includes(key) ? 'cursor-pointer hover:bg-muted transition-colors' : ''} ${key === 'output' ? 'active-card' : ''}" data-type="${key}" ${!['output', 'lineStopMins'].includes(key) ? 'style="pointer-events: none; cursor: default; background-color: #ddd8d8ff !important;"' : ''}>
                ${segmentedControl}
                <div class="p-4">
                <div class="flex flex-col md:flex-row md:items-stretch h-full">
                    <div class="flex-1 flex flex-col ">
                    <div>
                        <div class="flex items-center gap-1.5">
                        <p class="text-sm font-medium text-muted-foreground">${kpi.label}</p>
                        <i class="fa-solid fa-circle-info h-3 w-3 text-muted-foreground" data-tooltip="${kpi.description}"></i>
                        </div>
                        <p class="text-2xl font-bold  ">${formatValue(kpi.value)}</p>
                    </div>
                    <div>
                        <div class="flex items-center text-xs text-muted-foreground">
                        <i class="fa-solid ${arrowIcon === 'arrow-up' ? 'fa-arrow-up' : 'fa-arrow-down'} h-4 w-4 ${comparisonColor}"></i>
                        <span class="ml-1 ${comparisonColor}">${Math.abs(kpi.yesterday)}</span> 
                        <span class="ml-1">${comparisonText}</span>
                        </div>
                        <p class="text-xs text-purple-300 mt-1">
                        Target: ${formatValue(kpi.target)}
                        </p>
                    </div>
                    </div>
                    <div class="w-full h-12 md:w-24 md:h-auto mt-2 md:mt-0 md:ml-4">
                    <canvas id="chart-${key}" class="sparkline-container w-full h-full"></canvas>
                    </div>
                </div>
                </div>
            </div>
            `;
            
            kpiGrid.append(cardHtml);
            
            setTimeout(() => createSparkline(`chart-${key}`, kpi.trendData, chartColor), 100);
            
            // Click handlers for KPIs (only output and lineStopMins are clickable)
            if (key === 'lineStopMins') {
                $(`[data-type="lineStopMins"]`).on('click', function () {
                    window.location.href = '/socks_linestop_analysis';
                });
            } else if (key === 'output') {
                $(`[data-type="output"]`).on('click', function () {
                    const table = $('table');
                    if (table.length) {
                        $('html, body').animate({
                            scrollTop: table.offset().top - 80 // Offset for header
                        }, 500);
                        table.closest('.bg-card').addClass('highlight-row');
                        setTimeout(() => table.closest('.bg-card').removeClass('highlight-row'), 2000);
                    }
                });
            }
        
    });
    
    // Icons now use Font Awesome; no initialization required
    }

    // Drilldown table functionality
    let expandedRows = {};
    let filteredData = [];
    let originalData = []; // Store original unfiltered data
    let selectedBuyers = [];
    let selectedOcns = [];
    let buyerOcnData = [];
    let selectedBuyer = null;
    let selectedOcn = null;
    let selectedRangeLabel = 'Last Week';
    let isDataLoaded = false; // Track data loading status

    function buildTableRows() {
        const rows = [];
        
        function addRows(items, level = 0, parentUnit = null, parentNode = null) {
            items.forEach(item => {
                rows.push({ item, level, parentUnit, parentNode });
                const itemId = item.unitId || item.dateId || item.shiftId || item.lineId;
                
                if (expandedRows[itemId]) {
                    if (item.dates) {
                        addRows(item.dates, level + 1, item);
                    } else if (item.shifts) {
                        addRows(item.shifts, level + 1, parentUnit, item);
                    } else if (item.machines) {
                        addRows(item.machines, level + 1, parentUnit, item);
                    } else if (item.lines) {
                        addRows(item.lines, level + 1, parentUnit, item);
                    }
                }
            });
        }
        
        addRows(filteredData);
        return rows;
    }

    function fetchDailySnapshot() {
        $("#loadSpinner").show();
        const dateRangePicker = $('#dateRangePicker').data('daterangepicker');
        const dateType = dateRangePicker.chosenLabel || 'Last Week';
        const fromDate = dateRangePicker.startDate.format('YYYY-MM-DD');
        const toDate = dateRangePicker.endDate.format('YYYY-MM-DD');

        $.ajax({
            url: '/socksdailysnapshot',
            type: 'GET',
            data: {
                type: dateType,
                fromdate: fromDate,
                enddate: toDate
            },
            dataType: 'json',
            success: function (response) {
                if (response.success) {
                    // Map overall_totals to mockKpiData
                    response.overall_totals.forEach(item => {
                        const keyMap = {
                            "Back Feed Avail": "backFeedAvail",
                            "Capacity Utilization": "capacityUtilization",
                            "Cutting Output": "cuttingOutput",
                            "Direct Labour": "directLabourStrength",
                            "Efficiency": "efficiency",
                            "Knitting prod": "output",
                            "Line Stop Mins": "lineStopMins",
                            "Orders on Track": "ordersOnTrack",
                            "Packing Output": "packingOutput",
                            "WIP": "wip"
                        };
                        const kpiKey = keyMap[item.Type];
                        if (kpiKey && mockKpiData[kpiKey]) {
                            mockKpiData[kpiKey].value = parseFloat(item.Cutting) ;
                            mockKpiData[kpiKey].yesterday = parseFloat(item.Prv_Cut) ;
                        }
                    });

                    initializeKpiCards();
                    originalData = transformApiData(response.units);
                    filteredData = JSON.parse(JSON.stringify(originalData));
                    populateFilters(originalData); // Use transformed data
                    applyFilters();
                    isDataLoaded = true;
                    
                    // Update segmented control after data is loaded
                    const rows = buildTableRows();
                    updateSegmentedControl(rows);
                    updateLabourSegmentedControl();
                } else {
                    isDataLoaded = false;
                    $('#drilldownTableBody').html(`
                        <tr>
                            <td colspan="12" class="h-24 text-center text-muted-foreground">
                                No data available for the current selection.
                            </td>
                        </tr>
                    `);
                }
                $("#loadSpinner").hide();
            },
            error: function (xhr, status, error) {
                isDataLoaded = false;
                $("#loadSpinner").hide();
                console.error('Error fetching daily snapshot:', error);
                $('#drilldownTableBody').html(`
                    <tr>
                        <td colspan="12" class="h-24 text-center text-muted-foreground">
                            Failed to load data. Please try again.
                        </td>
                    </tr>
                `);
            }
        });
    }

	function naturalCompare(a, b) {
        return a.localeCompare(b, undefined, { numeric: true, sensitivity: 'base' });
    }

    function transformApiData(units) {
        if (!Array.isArray(units)) return [];
        return units.map(unit => ({
            unitId: unit.Unit,
            name: ` ${unit.Unit}`,
            type: 'unit',
            runningDays: parseInt(unit.Total_Ern_Rundays) || null,
            sam: parseFloat(unit.Total_Ern_Sam) || null,
            tailorStrength: parseInt(unit.Total_Tailors) || null,
            tailorStrengthTarget: null,
            labourStrength: (parseInt(unit.Total_Dir_Lab ?? unit.Total_Dir_Cnt ?? unit.Direct_Labour ?? unit.Direct) || 0),
            labourStrengthTarget: null,
            wip: unit.Total_Line_WIP,
            output: unit.Total_Output_Qty,
            target: null,
            earnedMinutes: unit.Total_Ern_Mins,
            efficiency: parseFloat(unit.Total_Ern_EfficiencyPerc) || 0,
            outputTrend: [],
            efficiencyTrend: [],
            totalInputBlocks: Math.floor(Math.random() * 10) || 0,
            totalLineStops: Math.floor(Math.random() * 10) || 0,
            newLinesCount: Math.floor(Math.random() * 10) || 0,
            dates: (Array.isArray(unit.dates) ? unit.dates : []).map(date => ({
                dateId: `${unit.Unit}-${date.EntryDate || 'NoDate'}`,
                name: date.EntryDate || 'Date',
                type: 'date',
                runningDays: parseInt(date.Total_Ern_Rundays) || null,
                sam: parseFloat(date.Total_Ern_Sam) || null,
                tailorStrength: parseInt(date.Total_Tailors) || null,
                tailorStrengthTarget: null,
                labourStrength: (parseInt(date.Total_Dir_Lab ?? date.Total_Dir_Cnt ?? date.Direct_Labour ?? date.Direct) || 0),
                labourStrengthTarget: null,
                // wip: date.Total_Line_WIP,
                wip:"-",
                output: date.Total_Output_Qty,
                target: null,
                earnedMinutes: date.Total_Ern_Mins,
                efficiency: parseFloat(date.Total_Ern_EfficiencyPerc) || 0,
                outputTrend: [],
                efficiencyTrend: [],
                totalInputBlocks: Math.floor(Math.random() * 10) || 0,
                totalLineStops: Math.floor(Math.random() * 10) || 0,
                shifts: (Array.isArray(date.shifts) ? date.shifts : []).map(shift => ({
                    shiftId: `${unit.Unit}-${date.EntryDate || 'NoDate'}-S${shift.Shift || 'NA'}`,
                    name: `Shift ${shift.Shift}`,
                    type: 'shift',
                    runningDays: parseInt(shift.Total_Ern_Rundays) || null,
                    sam: parseFloat(shift.Total_Ern_Sam) || null,
                    tailorStrength: parseInt(shift.Total_Tailors) || null,
                    tailorStrengthTarget: null,
                    labourStrength: (parseInt(shift.Total_Dir_Lab ?? shift.Total_Dir_Cnt ?? shift.Direct_Labour ?? shift.Direct) || 0),
                    labourStrengthTarget: null,
                    // wip: shift.Total_Line_WIP,
                    wip:"-",
                    output: shift.Total_Output_Qty,
                    target: null,
                    earnedMinutes: shift.Total_Ern_Mins,
                    efficiency: parseFloat(shift.Total_Ern_EfficiencyPerc) || 0,
                    outputTrend: [],
                    efficiencyTrend: [],
                    totalInputBlocks: Math.floor(Math.random() * 10) || 0,
                    totalLineStops: Math.floor(Math.random() * 10) || 0,
                    machines: (Array.isArray(shift.data) ? shift.data : []).map(m => ({
                        machineId: `${unit.Unit}-${date.EntryDate || 'NoDate'}-S${shift.Shift || 'NA'}-${m.Machine || m.Line_No || 'M'}`,
                        name: `${m.Machine || m.Line_No || ''}`.trim(),
                        type: 'machine',
                        ocn: m.Order_No,
                        buyer: m.Buyer,
                       // model: (m.MODEL || '').trim(),
                        runningDays: parseInt(m.Run_Days) || null,
                        sam: parseFloat(m.Sam || m.SAM) || null,
                        //tailorStrength: parseInt(m.Tailors || m.Tailor) || 0,
                        tailorStrengthTarget: null,
                        labourStrength: (parseInt(m.Dir_Lab ?? m.Dir_Cnt ?? m.Direct_Labour ?? m.Direct) || 0),
                        labourStrengthTarget: null,
                        // wip: m.Line_WIP !== null && m.Line_WIP !== undefined ? parseInt(m.Line_WIP) : 0,
                        wip:"-",
                        output: m.Total_Output_Qty !== undefined ? parseInt(m.Total_Output_Qty) : (parseInt(m.Output_Qty) || 0),
                        target: null,
                        earnedMinutes: parseInt(m.Ern_Mins) || 0,
                        efficiency: parseFloat(m.Eff_Perc || m.Total_Ern_EfficiencyPerc) || 0,
                        outputTrend: [],
                        efficiencyTrend: [],
                        isInputBlocked: (m.Line_WIP || 0) > 1000,
                        hasLineStopToday: false
                    })).sort((a, b) => naturalCompare(a.name, b.name))
                }))
            }))
        }));
    }

    function renderTable() {
        const tbody = $('#drilldownTableBody');
        tbody.empty();
        
        const rows = buildTableRows();
        
        if (rows.length === 0) {
            tbody.append(`
                <tr>
                    <td colspan="12" class="h-24 text-center text-muted-foreground">
                        No data available for the current selection.
                    </td>
                </tr>
            `);
            return;
        }
        
        rows.forEach(({ item, level, parentUnit, parentNode }) => {
            const rowId = item.unitId || item.dateId || item.shiftId || item.machineId || item.lineId;
            const isExpandable = item.dates || item.shifts || item.machines || item.lines;
            const isExpanded = expandedRows[rowId];
            const isLine = !isExpandable;
            
            // Use placeholders for missing data
            const tailorDiff = item.tailorStrengthTarget ? item.tailorStrength - item.tailorStrengthTarget : 0;
            const isTailorGood = tailorDiff >= 0;
            const labourDiff = item.labourStrengthTarget ? item.labourStrength - item.labourStrengthTarget : 0;
            const isLabourGood = labourDiff >= 0;
            const outputDiff = item.target ? item.output - item.target : 0;
            const isOutputGood = outputDiff >= 0;
            const earnedMinutesDiff = item.target && item.sam ? item.earnedMinutes - (item.target * item.sam) : 0;
            const isEarnedMinutesGood = earnedMinutesDiff >= 0;
            // Build Buyer / order / model for leaf rows
            let buyerOrderModel = '';
            if (isLine) {
                const buyerTxt = item.buyer ? item.buyer.replace(/[^a-zA-Z]/g, '') : '';
                const ocnTxt = item.ocn ? item.ocn.split("\\").pop() : '';
                const modelTxt = item.model || '';
                const left = [buyerTxt, ocnTxt].filter(Boolean).join('-');
                //buyerOrderModel = left ? (modelTxt ? `${left}- ${modelTxt}` : left) : (modelTxt || '');
				buyerOrderModel = left;
            }
            const rowClass = level === 0 ? 'bg-primary-50' : (level === 1 ? 'bg-muted/50' : '');
            
            const rowHtml = `
                <tr class="border-b border-border ${rowClass}" data-row-id="${rowId}">
                    <td class="h-12 align-middle" style="padding-left: ${level * 20 + 16}px">
                        <div class="flex items-center gap-2">
                            ${isExpandable ? 
                                `<button class="inline-flex items-center justify-center h-6 w-6 rounded hover:bg-muted transition-colors expand-btn" data-row-id="${rowId}">
                                    <i class="fa-solid ${isExpanded ? 'fa-chevron-down' : 'fa-chevron-right'} h-4 w-4"></i>
                                </button>` :
                                `<div class="w-6 h-6 flex items-center justify-center">
                                    ${item.isInputBlocked ? '<div class="status-dot-red cursor-pointer" title="Input Blocked - High WIP"></div>' : ''}
                                    ${item.hasLineStopToday && !item.isInputBlocked ? '<div class="status-dot-yellow cursor-pointer" title="Line Stop Today"></div>' : ''}
                                </div>`
                            }
                            <span class="font-medium">${item.name}</span>
                            ${item.totalInputBlocks > 0 ? 
                                `<div class="flex items-center gap-1.5 bg-red-100 text-red-700 px-2 py-0.5 rounded-full text-xs font-semibold cursor-pointer" title="Lines blocked - High WIP">
                                    <div class="status-dot-red"></div>
                                    ${item.totalInputBlocks} WIP
                                </div>` : ''
                            }
                            ${item.totalLineStops > 0 ? 
                                `<div class="flex items-center gap-1.5 bg-yellow-100 text-yellow-700 px-2 py-0.5 rounded-full text-xs font-semibold cursor-pointer" title="Total Line Stops">
                                    <div class="status-dot-yellow"></div>
                                    ${item.totalLineStops} ${item.totalLineStops === 1 ? 'Stop' : 'Stops'}
                                </div>` : ''
                            }
                        </div>
                    </td>
                    <td class="h-12 text-xs align-middle text-left">${isLine ? buyerOrderModel : ''}</td>
                    <td class="h-12  align-middle text-right tabular-nums">${item.runningDays ? item.runningDays  : '-'}</td>
                    <td class="h-12  align-middle text-right tabular-nums">${item.sam ? item.sam.toFixed(2) : '-'}</td>
                    <td class="h-12  align-middle text-right">
                        <div class="flex items-center justify-end gap-1.5 tabular-nums">
                            <span class="font-semibold">${item.tailorStrength ? item.tailorStrength.toLocaleString() : '-'}</span>
                            ${item.tailorStrengthTarget ? `
                                <div class="flex items-center text-xs ${isTailorGood ? 'text-green-600' : 'text-red-600'}">
                                    <i class="fa-solid ${isTailorGood ? 'fa-arrow-up' : 'fa-arrow-down'} h-3 w-3"></i>
                                    <span>${Math.abs(tailorDiff)}</span>
                                </div>` : ''
                            }
                        </div>
                    </td>
                    <td class="h-12  align-middle text-center">
                        <div class="flex items-center justify-center gap-1.5 tabular-nums ">
                            <span class="font-semibold ">${item.labourStrength ? item.labourStrength.toLocaleString() : '-'}</span>
                            ${item.labourStrengthTarget ? `
                                <div class="flex items-center text-xs ${isLabourGood ? 'text-green-600' : 'text-red-600'}">
                                    <i class="fa-solid ${isLabourGood ? 'fa-arrow-up' : 'fa-arrow-down'} h-3 w-3"></i>
                                    <span>${Math.abs(labourDiff)}</span>
                                </div>` : ''
                            }
                        </div>
                    </td>
                    <td class="h-12  align-middle text-right tabular-nums">${item.wip ? item.wip.toLocaleString() : '-'}</td>
                    <td class="h-12  align-middle text-right">
                        <div class="flex items-center justify-end gap-1.5 tabular-nums">
                            <span class="font-semibold">${item.output ? item.output.toLocaleString() : '-'}</span>
                            ${item.target ? `
                                <div class="flex items-center text-xs ${isOutputGood ? 'text-green-600' : 'text-red-600'}">
                                    <i class="fa-solid ${isOutputGood ? 'fa-arrow-up' : 'fa-arrow-down'} h-3 w-3"></i>
                                    <span>${Math.abs(outputDiff)}</span>
                                </div>` : ''
                            }
                        </div>
                    </td>
                    <td class="h-12  align-middle text-center">
                        ${item.outputTrend.length > 0 ? `<canvas id="output-trend-${rowId}" class="sparkline-container mx-auto"></canvas>` : '-'}
                    </td>
                    <td class="h-12  align-middle text-right">
                        <div class="flex items-center justify-end gap-1.5 tabular-nums">
                            <span class="font-semibold">${item.earnedMinutes ? item.earnedMinutes.toLocaleString() : '-'}</span>
                            ${item.target && item.sam ? `
                                <div class="flex items-center text-xs ${isEarnedMinutesGood ? 'text-green-600' : 'text-red-600'}">
                                    <i class="fa-solid ${isEarnedMinutesGood ? 'fa-arrow-up' : 'fa-arrow-down'} h-3 w-3"></i>
                                    <span>${Math.abs(Math.round(earnedMinutesDiff))}</span>
                                </div>` : ''
                            }
                        </div>
                    </td>
                    <td class="h-12 align-middle text-right tabular-nums">
                        <span class="font-semibold">${item.efficiency ? item.efficiency.toFixed(1) + '%' : '-'}</span>
                    </td>
                    <td class="h-12 align-middle text-center">
                        ${item.efficiencyTrend.length > 0 ? `<canvas id="efficiency-trend-${rowId}" class="sparkline-container mx-auto"></canvas>` : '-'}
                    </td>
                </tr>
            `;
            
            tbody.append(rowHtml);
            
            // Create sparkline charts if data exists
            setTimeout(() => {
                if (item.outputTrend.length > 0) {
                    createSparkline(`output-trend-${rowId}`, item.outputTrend, '#3b82f6');
                }
                if (item.efficiencyTrend.length > 0) {
                    createSparkline(`efficiency-trend-${rowId}`, item.efficiencyTrend, '#10b981');
                }
            }, 100);
        });
        
        // Icons now use Font Awesome; no reinitialization required
        
        // Update segmented control with sums
        updateSegmentedControl(rows);
        updateLabourSegmentedControl();
    }
    
    function updateSegmentedControl(rows) {
        let totalOutput = 0;
        let totalEarnedMinutes = 0;
        
        // Traverse units > dates > shifts > lines
        originalData.forEach(unit => {
            (unit.dates || []).forEach(date => {
                (date.shifts || []).forEach(shift => {
                    (shift.machines || shift.lines || []).forEach(machine => {
                        if (machine.output !== undefined && machine.output !== null) {
                            totalOutput += parseFloat(machine.output) || 0;
                        }
                        if (machine.earnedMinutes !== undefined && machine.earnedMinutes !== null) {
                            totalEarnedMinutes += parseFloat(machine.earnedMinutes) || 0;
                        }
                    });
                });
            });
        });
        
        $('#output-sum').text(totalOutput.toLocaleString());
        $('#earned-minutes-sum').text(totalEarnedMinutes.toLocaleString());
    }

    function updateLabourSegmentedControl() {
        let machTailor = 0, machDirect = 0;
        let shiftTailor = 0, shiftDirect = 0;
        let dateTailor = 0, dateDirect = 0;
        let unitTailor = 0, unitDirect = 0;

        (originalData || []).forEach(unit => {
            if (unit.tailorStrength != null) unitTailor += Number(unit.tailorStrength) || 0;
            if (unit.labourStrength != null) unitDirect += Number(unit.labourStrength) || 0;

            (unit.dates || []).forEach(date => {
                if (date.tailorStrength != null) dateTailor += Number(date.tailorStrength) || 0;
                if (date.labourStrength != null) dateDirect += Number(date.labourStrength) || 0;

                (date.shifts || []).forEach(shift => {
                    if (shift.tailorStrength != null) shiftTailor += Number(shift.tailorStrength) || 0;
                    if (shift.labourStrength != null) shiftDirect += Number(shift.labourStrength) || 0;

                    (shift.machines || shift.lines || []).forEach(m => {
                        if (m.tailorStrength != null) machTailor += Number(m.tailorStrength) || 0;
                        if (m.labourStrength != null) machDirect += Number(m.labourStrength) || 0;
                    });
                });
            });
        });

        const totalTailor = machTailor > 0 ? machTailor : (shiftTailor > 0 ? shiftTailor : (dateTailor > 0 ? dateTailor : unitTailor));
        const totalDirect = machDirect > 0 ? machDirect : (shiftDirect > 0 ? shiftDirect : (dateDirect > 0 ? dateDirect : unitDirect));

        const unalloc = 0; // fixed as 00
        const tailorEl = document.getElementById('labour-tailor-sum');
        const directEl = document.getElementById('labour-direct-sum');
        const unallocEl = document.getElementById('labour-unalloc-sum');
        if (tailorEl) tailorEl.textContent = (totalTailor || 0).toLocaleString();
        if (directEl) directEl.textContent = (totalDirect || 0).toLocaleString();
        if (unallocEl) unallocEl.textContent = unalloc.toLocaleString(undefined, { minimumIntegerDigits: 2 });
    }

    function populateFilters(units) {   
        if (!units || !Array.isArray(units)) {
            console.warn('populateFilters: units is undefined or not an array');
            $('#buyerDropdownMenu').empty().append('<li class="px-3 py-2">No buyers available</li>');
            $('#ocnDropdownMenu').empty().append('<li class="px-3 py-2">Select at least one buyer</li>');
            return;
        }

        const buyers = new Set();
        const ocnsByBuyer = {};

        units.forEach(unit => {
            (unit.dates || []).forEach(date => {
                (date.shifts || []).forEach(shift => {
                    (shift.machines || shift.lines || []).forEach(machine => {
                        const buyer = machine.buyer;
                        const ocn = machine.ocn;
                        if (!buyer || !ocn) return;
                        buyers.add(buyer);
                        if (!ocnsByBuyer[buyer]) ocnsByBuyer[buyer] = new Set();
                        ocnsByBuyer[buyer].add(ocn);
                    });
                });
            });
        });

        const buyerMenu = $('#buyerDropdownMenu');
        buyerMenu.empty();
        if (buyers.size === 0) {
            buyerMenu.append('<li class="px-3 py-2">No buyers available</li>');
        } else {
            [...buyers].sort().forEach(buyer => {
                const isChecked = selectedBuyers.includes(buyer) ? 'checked' : '';
                buyerMenu.append(`
                    <li class="px-3 py-2 hover:bg-muted cursor-pointer flex items-center" style="font-size: 10px;">
                        <input type="checkbox" class="mr-2" data-value="${buyer}" ${isChecked}>
                        ${buyer}
                    </li>
                `);
            });
        }

        const ocnMenu = $('#ocnDropdownMenu');
        ocnMenu.empty();
        if (selectedBuyers.length > 0) {
            const ocns = new Set();
            selectedBuyers.forEach(buyer => {
                if (ocnsByBuyer[buyer]) [...ocnsByBuyer[buyer]].forEach(ocn => ocns.add(ocn));
            });
            if (ocns.size === 0) {
                ocnMenu.append('<li class="px-3 py-2">No OCNs available</li>');
            } else {
                [...ocns].sort().forEach(ocn => {
                    const isChecked = selectedOcns.includes(ocn) ? 'checked' : '';
                    ocnMenu.append(`
                        <li class="px-3 py-2 hover:bg-muted cursor-pointer flex items-center" style="font-size: 11px;">
                            <input type="checkbox" class="mr-2" data-value="${ocn}" ${isChecked}>
                            ${ocn}
                        </li>
                    `);
                });
            }
        } else {
            ocnMenu.append('<li class="px-3 py-2">Select at least one buyer</li>');
        }

        buyerOcnData = ocnsByBuyer;
    }

    function applyFilters() {
        let tempData = JSON.parse(JSON.stringify(originalData));

        const filterUnit = (unit) => ({
            ...unit,
            dates: (unit.dates || []).map(date => ({
                ...date,
                shifts: (date.shifts || []).map(shift => ({
                    ...shift,
                    machines: (shift.machines || shift.lines || [])
                        .filter(machine => selectedBuyers.length === 0 || selectedBuyers.includes(machine.buyer))
                        .filter(machine => selectedOcns.length === 0 || selectedOcns.includes(machine.ocn))
                })).filter(shift => ((shift.machines || []).length > 0))
            })).filter(date => ((date.shifts || []).filter(s => (s.machines || []).length > 0).length > 0))
        });

        tempData = tempData.map(filterUnit).filter(unit => (unit.dates || []).some(d => (d.shifts || []).some(s => (s.machines || []).length > 0)));

        filteredData = tempData;
        updateFilterBadge();
        renderTable();
    }

    function updateFilterBadge() {
        const filterBadge = $('#filterBadge');
        const activeFilters = selectedBuyers.length + selectedOcns.length;
        if (activeFilters > 0) filterBadge.text(activeFilters).removeClass('hidden');
        else filterBadge.addClass('hidden');
    }

    // Event handlers
    function initializeEventHandlers() {
        $(document).on('click', '.expand-btn', function() {
            const rowId = $(this).data('row-id');
            expandedRows[rowId] = !expandedRows[rowId];
            renderTable();
        });
        
        $('#filterBtn').click(function(e) {
            if (!isDataLoaded) return;
            e.stopPropagation();
            $('#filterDropdown').toggleClass('show');
        });

        $(document).click(function(e) {
            if (!$(e.target).closest('#filterBtn, #filterDropdown').length) {
                $('#filterDropdown').removeClass('show');
            }
        });

        $('#buyerDropdownBtn').click(function(e) {
            if (!isDataLoaded) return;
            e.stopPropagation();
            $('#buyerDropdownMenu').toggleClass('hidden');
            $('#ocnDropdownMenu').addClass('hidden');
        });

        $(document).on('change', '#buyerDropdownMenu input[type="checkbox"]', function() {
            if (!isDataLoaded) return;
            const buyer = $(this).data('value');
            if ($(this).is(':checked')) {
                if (!selectedBuyers.includes(buyer)) selectedBuyers.push(buyer);
            } else {
                selectedBuyers = selectedBuyers.filter(b => b !== buyer);
                if (buyerOcnData[buyer]) {
                    selectedOcns = selectedOcns.filter(ocn => ![...buyerOcnData[buyer]].includes(ocn));
                }
            }
            $('#buyerSelected').text(selectedBuyers.length > 0 ? `${selectedBuyers.length} Buyer${selectedBuyers.length > 1 ? 's' : ''} Selected` : 'Select Buyers');
            $('#ocnSelected').text(selectedOcns.length > 0 ? `${selectedOcns.length} OCN${selectedOcns.length > 1 ? 's' : ''} Selected` : 'Select OCNs');
            populateFilters(originalData);
            applyFilters();
        });

        $('#ocnDropdownBtn').click(function(e) {
            if (!isDataLoaded) return;
            e.stopPropagation();
            $('#ocnDropdownMenu').toggleClass('hidden');
            $('#buyerDropdownMenu').addClass('hidden');
        });

        $(document).on('change', '#ocnDropdownMenu input[type="checkbox"]', function() {
            if (!isDataLoaded) return;
            const ocn = $(this).data('value');
            if ($(this).is(':checked')) {
                if (!selectedOcns.includes(ocn)) selectedOcns.push(ocn);
            } else {
                selectedOcns = selectedOcns.filter(o => o !== ocn);
            }
            $('#ocnSelected').text(selectedOcns.length > 0 ? `${selectedOcns.length} OCN${selectedOcns.length > 1 ? 's' : ''} Selected` : 'Select OCNs');
            applyFilters();
        });

        $(document).click(function() {
            $('#buyerDropdownMenu, #ocnDropdownMenu').addClass('hidden');
        });

        $('#resetFiltersBtn').click(function() {
            if (!isDataLoaded) return;
            selectedBuyers = [];
            selectedOcns = [];
            $('#buyerSelected').text('Select Buyers');
            $('#ocnSelected').text('Select OCNs');
            $('#ocnDropdownMenu').empty().append('<li class="px-3 py-2">Select at least one buyer</li>');
            filteredData = JSON.parse(JSON.stringify(originalData));
            populateFilters(originalData);
            applyFilters();
        });

        // Notifications panel
        $('#notificationsBtn').click(function() {
            $('#notificationOverlay').removeClass('hidden');
            setTimeout(() => {
                $('#notificationPanel').removeClass('translate-x-full');
            }, 10);
        });

        $(document).on('click', '#notificationOverlay, #closeNotifications', function(e) {
            if ($(e.target).is('#notificationOverlay') || $(e.target).closest('#closeNotifications').length) {
                $('#notificationPanel').addClass('translate-x-full');
                setTimeout(() => {
                    $('#notificationOverlay').addClass('hidden');
                }, 300);
            }
        });

        // Tooltip functionality
        $(document).on('mouseenter', '[data-tooltip]', function () {
            const tooltip = $('#tooltip');
            const title = $(this).data('tooltip');

            if (title) {
                // Empty tooltip first, then set new text
                tooltip.text('').text(title).addClass('show');

                // Position tooltip above the element
                const offset = $(this).offset();
                const tooltipHeight = tooltip.outerHeight();
                const tooltipWidth = tooltip.outerWidth();
                const elementWidth = $(this).outerWidth();

                tooltip.css({
                    left: offset.left + (elementWidth / 2) - (tooltipWidth / 2) + 'px',
                    top: offset.top - tooltipHeight - 8 + 'px' // 8px gap above the element
                });
            }
        });

        $(document).on('mouseleave', '[data-tooltip]', function () {
            $('#tooltip').removeClass('show').text('');
        });

    }

    $(document).ready(function () {
        initializeKpiCards();
        renderTable();
        initializeEventHandlers();
        // Icons now use Font Awesome; no initialization required

        // Initialize date range picker
        $('#dateRangePicker').daterangepicker({
            opens: 'left',
            startDate: moment(),
            endDate: moment(),
            maxDate: moment(), // Restrict dates beyond today
            ranges: {
                'Today': [moment(), moment()],
                'Last Week': [moment().subtract(6, 'days'), moment()], // Changed from This Week
                'This Month': [moment().startOf('month'), moment().endOf('month')]
            }
        }, function(start, end, label) {
            // Format dates as dd-mm-yy
            const startFormatted = start.format('DD-MM-YY');
            const endFormatted = end.format('DD-MM-YY');
            // Store the selected range label
            selectedRangeLabel = label;
            // Check if start and end are the same day
            const isSameDay = start.isSame(end, 'day');
            const titleText = `
             Production_dashboard_Overview- 
            <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
                SOCKS
            </span>
            <span class="ml-4" style="color: #fff; font-weight: 500;">
                ${startFormatted} - ${endFormatted}
            </span>
            `;

            $('.dashboard_title').html(titleText); // ✅ Use .html() instead of .text()
            
            // Update input value
            $('#dateRangePicker').val(`${startFormatted} - ${endFormatted}`);
            
            // Fetch data
            fetchDailySnapshot();
        });

        $(`#HistoricalButton`).on('click', function () {
            window.location.href = '/history_dashboard/Production';
        });

        // Initial title setup
        const initialStart = moment();
        const initialEnd = moment();
        const initialStartFormatted = initialStart.format('DD-MM-YY');
        const initialEndFormatted = initialEnd.format('DD-MM-YY');
        const isSameDay = initialStart.isSame(initialEnd, 'day');
        $('.dashboard_title').html(`
        Production Overview dashboard -   
        <span style="color: #000; font-weight: bold;  background-color: #FFFF00; padding: 2px 4px; border-radius: 10px;">SOCKS</span>
        <span class="ml-4">
            ${initialStartFormatted} - ${initialEndFormatted}
        </span>
        `);
        
        // Initial fetch
        $('#dateRangePicker').val(`${initialStartFormatted} - ${initialEndFormatted}`);
        fetchDailySnapshot();
    });

</script>
</body>
</html>
