<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Line Stop Analysis</title>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <script src="https://cdn.tailwindcss.com"></script>
  <!-- Font Awesome CSS -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/jquery.dataTables.min.css" />
  <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            'background': 'rgb(245, 246, 247)',
            'foreground': 'rgb(31, 41, 55)',
            'card': 'rgb(255, 255, 255)',
            'card-foreground': 'rgb(31, 41, 55)',
            'primary': 'rgb(15, 61, 126)',
            'primary-foreground': 'rgb(248, 250, 252)',
            'primary-100': 'rgb(219, 234, 254)',
            'primary-50': 'rgb(239, 246, 255)',
            'primary-25': 'rgb(248, 250, 252)',
            'secondary': 'rgb(241, 245, 249)',
            'muted': 'rgb(241, 245, 249)',
            'muted-foreground': 'rgb(100, 116, 139)',
            'border': 'rgb(226, 232, 240)',
            'input': 'rgb(226, 232, 240)',
            'ring': 'rgb(15, 61, 126)',
          },
          fontFamily: {
            'body': ['Inter', 'system-ui', 'sans-serif'],
          }
        }
      }
    }
  </script>
  <style>
    .date-picker input {
      background: rgba(248, 250, 252, 0.1);
      color: white;
    }
    .date-picker input::placeholder {
      color: rgba(248, 250, 252, 0.7);
    }
    .spinner {
      display: inline-block;
      width: 1.5rem;
      height: 1.5rem;
      border: 3px solid rgba(255, 255, 255, 0.3);
      border-top-color: #fff;
      border-radius: 50%;
      animation: spin 1s ease-in-out infinite;
    }
    @keyframes spin {
      to { transform: rotate(360deg); }
    }
    .no-data-overlay {
      position: relative;
    }
    .no-data-message {
      position: absolute;
      inset: 0;
      display: flex;
      align-items: center;
      justify-content: center;
      background: white;
      color: #6b7280;
      font-size: 1rem;
      font-weight: 500;
      z-index: 10;
    }
    .chart-canvas {
      display: block;
    }
    .chart-canvas.hidden {
      display: none !important;
    }
    /* Compact DataTables controls to match other pages */
    .dataTables_wrapper .dataTables_length label { font-size: 12px; color:#6b7280; }
    .dataTables_wrapper .dataTables_length select { font-size: 12px; padding: 2px 6px; height: 26px; line-height: 22px; border:1px solid #e5e7eb; border-radius:6px; width:auto; min-width:56px; }
    .dataTables_wrapper .dataTables_filter input { font-size: 12px; padding: 2px 6px; height: 26px; line-height: 22px; border:1px solid #D3D3D3; border-radius:6px; width:120px; background-color:#ffffff; }
    .dataTables_wrapper .dataTables_info { font-size: 12px; color:#6b7280; }
    .dataTables_wrapper .dataTables_paginate { font-size: 12px; }
    .dataTables_wrapper .dataTables_paginate .paginate_button { padding: 2px 6px; margin: 0 1px; font-size: 12px; }
    @keyframes gradient {
      0% { background-position: 0% 50%; }
      100% { background-position: 200% 50%; }
    }
    .animate-gradient {
      background-size: 200% 200%;
      animation: gradient 3s ease infinite;
    }
  </style>
</head>
<body class="bg-gray-50 text-gray-800">
  
  <?php echo $__env->make('LifeAtGainup.production.header', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

  <div class="px-6 py-3 text-sm text-gray-600">
    <a id="productionBreadcrumb" href="/production_gar_dashboard" class="text-blue-700 hover:underline">Production Dashboard</a> 
    <span> &gt; </span>
    <span class="text-gray-800 font-medium">Line Stop Analysis</span>
  </div>
  <div class="mx-auto px-6 space-y-6">
    <div class="grid grid-cols-4 gap-4">
      <div class="kpi-card bg-white rounded-2xl shadow p-4 cursor-pointer" data-type="lostMinutes">
        <p class="text-gray-600 text-sm">Total Lost Minutes</p>
        <h2 class="text-2xl font-bold">0 mins</h2>
        <p class="text-red-600 text-sm" style="display: block;">▼ 0% vs last week</p>
        <p class="text-gray-400 text-xs">Target: 2,500 mins</p>
      </div>
      <div class="kpi-card bg-white rounded-2xl shadow p-4 cursor-pointer" data-type="financialLoss">
        <p class="text-gray-600 text-sm">Financial Loss</p>
        <h2 class="text-2xl font-bold">$0</h2>
        <p class="text-green-600 text-sm" style="display: block;">▼ 0% vs last week</p>
        <p class="text-gray-400 text-xs">Target: 3,750</p>
      </div>
      <div class="kpi-card bg-white rounded-2xl shadow p-4 cursor-pointer" data-type="stops">
        <p class="text-gray-600 text-sm">Number of Stops</p>
        <h2 class="text-2xl font-bold">0</h2>
        <p class="text-red-600 text-sm" style="display: block;">▼ 0% vs last week</p>
        <p class="text-gray-400 text-xs">Target: 100</p>
      </div>
      <div class="kpi-card bg-white rounded-2xl shadow p-4 cursor-pointer" data-type="mtbs">
        <p class="text-gray-600 text-sm">MTBS</p>
        <h2 class="text-2xl font-bold">0 mins</h2>
        <p class="text-red-600 text-sm" style="display: block;">▼ 0% vs last week</p>
        <p class="text-gray-400 text-xs">Target: 120 mins</p>
      </div>
    </div>
    <div class="grid grid-cols-2 gap-4">
      <div class="bg-white rounded-2xl shadow p-4 relative no-data-overlay">
        <div class="flex flex-row items-center gap-2">
            <h3 id="unitChartTitle" class="font-semibold text-gray-700 mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Unit-wise Lost Minutes</h3>
            <p class="font-semibold text-gray-500 text-xs">Drill-down of total minutes lost due to line stops.</p>
        </div>    
        <p id="unitBreadcrumb" class="my-2">
          <span id="allUnits" class="cursor-pointer hover:underline">All units</span>
        </p>
        <canvas id="unitChart" class="chart-canvas" height="100"></canvas>
        <div id="unitNoData" class="no-data-message" style="display: none;">
          No data available
        </div>
      </div>
      <div class="bg-white rounded-2xl shadow p-4 relative no-data-overlay">
        <div class="flex flex-row items-center gap-2">
            <h3 id="reasonChartTitle" class="font-semibold text-gray-700 mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Reasons for Lost Minutes</h3>
            <p class="font-semibold text-gray-500 text-xs">Primary reasons for production stoppages (Pareto). Click an OCN-dependent bar to drill down.</p>
        </div>    
        <p id="reasonBreadcrumb" class="my-2">
          <span id="allReasons" class="cursor-pointer hover:underline">All Reasons</span>
        </p>
        <canvas id="reasonChart" class="chart-canvas" height="100"></canvas>
        <div id="reasonNoData" class="no-data-message" style="display: none;">
          No data available
        </div>
      </div>
    </div>
    <div class="bg-white rounded-2xl shadow p-4 relative no-data-overlay">
      <div class="flex flex-row items-center gap-2">
          <h3 id="trendChartTitle" class="font-semibold text-gray-700 mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Trend for Overall - Lost Minutes</h3>
          <p class="font-semibold text-gray-500 text-xs my-2">Overall trend for minutes lost due to line stops.</p>
      </div>    
      <canvas id="trendChart" class="chart-canvas h-72" height="100"></canvas>
      <div id="trendNoData" class="no-data-message" style="display: none;">
        No data available
      </div>
    </div>
    <div class="bg-white rounded-2xl shadow p-4">
      <div class="flex flex-row items-center gap-2">
          <h3 id="logTableTitle" class="font-semibold text-gray-700 mb-1 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Line Stop Log</h3>
          <p class="font-semibold text-gray-500 text-xs my-2">A detailed log of all line stop events based on the current filters.</p>
      </div>    
      <div id="logTableContainer">
      <table class="w-full text-left border-collapse text-xs" id="logTable">
        <thead>
          <tr class="text-gray-500 text-sm border-b">
            <th class="py-2">Date</th>
            <th class="py-2">Unit / Zone / Line</th>
            <th class="py-2">Reason</th>
            <th class="py-2">OCN</th>
            <th class="py-2">Start</th>
            <th class="py-2">End</th>
            <th class="py-2">Minutes</th>
            <th class="py-2">Loss</th>
            <th class="py-2">Remarks</th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
      </div>
    </div>
<script>

    // Icons now use Font Awesome; no initialization required
let unitChart, reasonChart, trendChart;
let currentLevel = "units";
let selectedPath = [];
let selectedfromdate = null;
let selectedenddate = null;
let reasonLevel = "reasons";
let reasonPath = [];
let apiData = null;
let currentPage = 1;
const rowsPerPage = 10;
let selectedRangeLabel = "Last Week";
let hasData = false;
let logTableInstance = null;

function resetLogTableInstance() {
  if (logTableInstance) {
    logTableInstance.destroy();
    logTableInstance = null;
  }
}

function showNoData() {
  hasData = false;
  $("#unitChart, #reasonChart, #trendChart").addClass("hidden");
  $("#unitNoData, #reasonNoData, #trendNoData").show();
  $(".kpi-card h2").text("No data available");
  $(".kpi-card p.text-red-600, .kpi-card p.text-green-600").hide();
  const tbody = $("#logTable tbody");
  tbody.html('<tr><td colspan="7" class="text-center py-8 text-gray-500">No data available</td></tr>');
  $("#tableInfo").text("No data available");
  $("#prevPage, #nextPage").prop("disabled", true);
  currentPage = 1;
  selectedPath = [];
  reasonPath = [];
  currentLevel = "units";
  reasonLevel = "reasons";
  updateBreadcrumb();
  updateReasonBreadcrumb();
  if (unitChart) unitChart.options.onClick = () => {};
  if (reasonChart) reasonChart.options.onClick = () => {};
  if (trendChart) reasonChart.options.onClick = () => {};
}

function showData() {
  hasData = true;
  $("#unitChart, #reasonChart, #trendChart").removeClass("hidden");
  $("#unitNoData, #reasonNoData, #trendNoData").hide();
  $(".kpi-card p.text-red-600, .kpi-card p.text-green-600").show();
}

function fetchData(type = "lostMinutes", fromDate = "", toDate = "") {
  $("#loadSpinner").show();
  $.ajax({
    url: '/linestopminis',
    method: 'GET',
    data: {
      type: type,
      fromdate: fromDate,
      enddate: toDate
    },
    success: function(response) {
      $("#loadSpinner").hide();
      console.log("API Response:", response);
      if (response && response.units && Array.isArray(response.units) && response.units.length > 0) {
        apiData = response.units;
        showData();
        initializeCharts();
        updateKPICards();
        updateLogTable();
        updateBreadcrumb();
        updateReasonBreadcrumb();
      } else {
        console.warn("No valid data in response:", response);
        apiData = [];
        showNoData();
        initializeCharts();
      }
    },
    error: function(jqXHR, textStatus, errorThrown) {
      console.error("Failed to fetch data:", textStatus, errorThrown, jqXHR.responseText);
      $("#loadSpinner").hide();
      apiData = [];
      showNoData();
      initializeCharts();
    }
  });
}

function getUnitChartData(level, unit = null, zone = null, filterReason = null, filterOcn = null) {
  if (!apiData || !Array.isArray(apiData) || apiData.length === 0) {
    return { labels: [], values: [] };
  }
  if (level === "units") {
    const units = {};
    apiData.forEach(u => {
      let sum = 0;
      if (u.zones && Array.isArray(u.zones)) {
        u.zones.forEach(z => {
          if (z.lines && Array.isArray(z.lines)) {
            z.lines.forEach(l => {
              if (l.data && Array.isArray(l.data)) {
                l.data.forEach(row => {
                  if ((!filterReason || row.Reason_Lkup === filterReason) && (!filterOcn || row.Order_No === filterOcn)) {
                    sum += row.Tot_DTM || 0;
                  }
                });
              }
            });
          }
        });
      }
      if (sum > 0 && u.Unit) units[u.Unit] = sum;
    });
    return { labels: Object.keys(units), values: Object.values(units) };
  } else if (level === "zones") {
    const unitData = apiData.find(u => u.Unit === unit);
    if (!unitData || !unitData.zones || !Array.isArray(unitData.zones)) {
      return { labels: [], values: [] };
    }
    const zones = {};
    unitData.zones.forEach(z => {
      let sum = 0;
      if (z.lines && Array.isArray(z.lines)) {
        z.lines.forEach(l => {
          if (l.data && Array.isArray(l.data)) {
            l.data.forEach(row => {
              if ((!filterReason || row.Reason_Lkup === filterReason) && (!filterOcn || row.Order_No === filterOcn)) {
                sum += row.Tot_DTM || 0;
              }
            });
          }
        });
      }
      if (sum > 0 && z.Zone) zones[z.Zone] = sum;
    });
    return { labels: Object.keys(zones), values: Object.values(zones) };
  } else if (level === "lines") {
    const unitData = apiData.find(u => u.Unit === unit);
    if (!unitData || !unitData.zones) {
      return { labels: [], values: [] };
    }
    const zoneData = unitData.zones.find(z => z.Zone === zone);
    if (!zoneData || !zoneData.lines || !Array.isArray(zoneData.lines)) {
      return { labels: [], values: [] };
    }
    const lines = {};
    zoneData.lines.forEach(l => {
      let sum = 0;
      if (l.data && Array.isArray(l.data)) {
        l.data.forEach(row => {
          if ((!filterReason || row.Reason_Lkup === filterReason) && (!filterOcn || row.Order_No === filterOcn)) {
            sum += row.Tot_DTM || 0;
          }
        });
      }
      if (sum > 0 && l.Line_No) lines[`Line ${l.Line_No}`] = sum;
    });
    return { labels: Object.keys(lines), values: Object.values(lines) };
  }
  return { labels: [], values: [] };
}

function getReasonChartData(level, reason = null, filterUnit = null, filterZone = null, filterLine = null) {
  if (!apiData || !Array.isArray(apiData) || apiData.length === 0) {
    return { labels: [], values: [] };
  }
  if (level === "reasons") {
    const reasons = {};
    apiData.forEach(unit => {
      if (!filterUnit || unit.Unit === filterUnit) {
        if (unit.zones && Array.isArray(unit.zones)) {
          unit.zones.forEach(zone => {
            if (!filterZone || zone.Zone === filterZone) {
              if (zone.lines && Array.isArray(zone.lines)) {
                zone.lines.forEach(line => {
                  if (!filterLine || `Line ${line.Line_No}` === filterLine) {
                    if (line.data && Array.isArray(line.data)) {
                      line.data.forEach(row => {
                        if (row.Reason_Lkup) {
                          reasons[row.Reason_Lkup] = (reasons[row.Reason_Lkup] || 0) + (row.Tot_DTM || 0);
                        }
                      });
                    }
                  }
                });
              }
            }
          });
        }
      }
    });
    const sortedReasons = Object.entries(reasons)
      .sort(([,a], [,b]) => b - a)
      .reduce((acc, [key, value]) => ({...acc, [key]: value}), {});
    return { labels: Object.keys(sortedReasons), values: Object.values(sortedReasons) };
  } else if (level === "ocns") {
    const ocns = {};
    apiData.forEach(unit => {
      if (!filterUnit || unit.Unit === filterUnit) {
        if (unit.zones && Array.isArray(unit.zones)) {
          unit.zones.forEach(zone => {
            if (!filterZone || zone.Zone === filterZone) {
              if (zone.lines && Array.isArray(zone.lines)) {
                zone.lines.forEach(line => {
                  if (!filterLine || `Line ${line.Line_No}` === filterLine) {
                    if (line.data && Array.isArray(line.data)) {
                      line.data.forEach(row => {
                        if (row.Reason_Lkup === reason && row.Order_No) {
                          ocns[row.Order_No] = (ocns[row.Order_No] || 0) + (row.Tot_DTM || 0);
                        }
                      });
                    }
                  }
                });
              }
            }
          });
        }
      }
    });
    const sortedOCNs = Object.entries(ocns)
      .sort(([,a], [,b]) => b - a)
      .reduce((acc, [key, value]) => ({...acc, [key]: value}), {});
    return { labels: Object.keys(sortedOCNs), values: Object.values(sortedOCNs) };
  }
  return { labels: [], values: [] };
}

function getTrendData(filterUnit = null, filterZone = null, filterLine = null, filterReason = null, filterOcn = null) {
  if (!apiData || !Array.isArray(apiData) || apiData.length === 0) {
    console.warn("getTrendData: No valid apiData");
    return { labels: [], values: [] };
  }
  const dateSums = {};
  apiData.forEach(unit => {
    if (!filterUnit || unit.Unit === filterUnit) {
      if (unit.zones && Array.isArray(unit.zones)) {
        unit.zones.forEach(zone => {
          if (!filterZone || zone.Zone === filterZone) {
            if (zone.lines && Array.isArray(zone.lines)) {
              zone.lines.forEach(line => {
                if (!filterLine || `Line ${line.Line_No}` === filterLine) {
                  if (line.data && Array.isArray(line.data)) {
                    line.data.forEach(row => {
                      if (row.Date && (!filterReason || row.Reason_Lkup === filterReason) && (!filterOcn || row.Order_No === filterOcn)) {
                        const rawDate = row.Date.trim().split(' ')[0];
                        dateSums[rawDate] = (dateSums[rawDate] || 0) + (row.Tot_DTM || 0);
                      }
                    });
                  }
                }
              });
            }
          }
        });
      }
    }
  });
  if (Object.keys(dateSums).length === 0) {
    console.warn("getTrendData: No dates with data after filtering");
    return { labels: [], values: [] };
  }
  const parsedDates = Object.keys(dateSums).map(rawDate => {
    const parsed = moment(rawDate, 'DD-MM-YY', true);
    if (!parsed.isValid()) {
      console.warn(`getTrendData: Skipping invalid date (expected DD-MM-YY): ${rawDate}`);
      return null;
    }
    return { rawDate, parsed };
  }).filter(item => item !== null);
  if (parsedDates.length === 0) {
    console.warn("getTrendData: No valid dates after parsing");
    return { labels: [], values: [] };
  }
  parsedDates.sort((a, b) => a.parsed - b.parsed);
  const sortedDates = parsedDates.map(item => item.rawDate);
  const values = sortedDates.map(date => dateSums[date]);
  const labels = parsedDates.map(item => item.parsed.format('MMM D'));
  console.log("Trend Data (sorted):", { labels, values });
  return { labels, values };
}

function getFilteredLogs(unit = null, zone = null, line = null, reason = null, ocn = null) {
  if (!apiData || !Array.isArray(apiData) || apiData.length === 0) {
    return [];
  }
  let logs = [];
  apiData.forEach(u => {
    if (!unit || u.Unit === unit) {
      if (u.zones && Array.isArray(u.zones)) {
        u.zones.forEach(z => {
          if (!zone || z.Zone === zone) {
            if (z.lines && Array.isArray(z.lines)) {
              z.lines.forEach(l => {
                if (!line || `Line ${l.Line_No}` === line) {
                  if (l.data && Array.isArray(l.data)) {
                    l.data.forEach(row => {
                      if ((!reason || row.Reason_Lkup === reason) && (!ocn || row.Order_No === ocn)) {
                        const ocnLastFour = row.Order_No ? row.Order_No.slice(-4) : " ";
                        const buyer = row.Buyer ? row.Buyer.replace(/[^a-zA-Z]/g, '').slice(0, 6) : " ";
                        logs.push({
                          date: row.Date || " ",
                          path: `${u.Unit} / ${z.Zone || ' '} / Line ${l.Line_No}`,
                          reason: row.Reason || " ",
                          ocn: `${buyer}${ocnLastFour}`,
                          duration: row.Tot_DTM || 0,
                          total_duration: row.Tot_DTM || 0,
                          loss: row.Loss_Amnt || 0,
                          remarks: row.Remarks || "-",
                          start_time: row.Start_Time || " ",
                          end_time: row.End_Time || " "
                        });
                      }
                    });
                  }
                }
              });
            }
          }
        });
      }
    }
  });
  return logs;
}

function updateLogTable(page = 1) {
  const logs = getFilteredLogs(selectedPath[0], selectedPath[1], selectedPath[2], reasonPath[0], reasonPath[1]);
  const tbody = $("#logTable tbody");
  if (!logTableInstance) {
    logTableInstance = $('#logTable').DataTable({
      language: { lengthMenu: 'Show _MENU_ rows', search: 'Search:' },
      pageLength: 10,
      lengthMenu: [10, 25, 50, 100],
      searching: true,
      ordering: true,
      info: true,
      paging: true,
      data: [],
      columns: [
        { data: 'date', defaultContent: '--' },
        { data: 'path', defaultContent: '--' },
        { data: 'reason', defaultContent: '--' },
        { data: 'ocn', defaultContent: '--' },
        { data: 'start_time', defaultContent: '--' },
        { data: 'end_time', defaultContent: '--' },
        { data: 'duration', defaultContent: 0 },
        { data: 'loss', defaultContent: 0, render: value => (value || 0).toLocaleString() },
        { data: 'remarks', defaultContent: '-' }
      ]
    });
  }
  logTableInstance.clear();
  if (logs.length === 0) {
    logTableInstance.rows.add([]);
  } else {
    const formattedLogs = logs.map(row => ({
      ...row,
      start_time: row.start_time || '--',
      end_time: row.end_time || '--',
      loss: row.loss || 0,
      ocn: `${row.ocn}<br>`
    }));
    logTableInstance.rows.add(formattedLogs);
  }
  logTableInstance.draw();
}

function updateKPICards() {
  const logs = getFilteredLogs(selectedPath[0], selectedPath[1], selectedPath[2], reasonPath[0], reasonPath[1]);
  if (logs.length === 0) {
    $(".kpi-card h2").text("No data available");
    $(".kpi-card p.text-red-600, .kpi-card p.text-green-600").hide();
    return;
  }
  const totalLostMinutes = logs.reduce((sum, log) => sum + (log.total_duration || 0), 0);
  const totalFinancialLoss = logs.reduce((sum, log) => sum + (log.loss || 0), 0);
  const numberOfStops = logs.length;
  const mtbs = numberOfStops ? (totalLostMinutes / numberOfStops).toFixed(1) : 0;
  $(`.kpi-card[data-type="lostMinutes"] h2`).text(`${totalLostMinutes.toLocaleString()} mins`);
  $(`.kpi-card[data-type="financialLoss"] h2`).text(`${totalFinancialLoss.toLocaleString()}`);
  $(`.kpi-card[data-type="stops"] h2`).text(numberOfStops.toLocaleString());
  $(`.kpi-card[data-type="mtbs"] h2`).text(`${mtbs} mins`);
  $(".kpi-card p.text-red-600, .kpi-card p.text-green-600").show();
}

function updateBreadcrumb() {
  let breadcrumbHtml = `<span id="allUnits" class="cursor-pointer hover:underline">All units</span>`;
  if (selectedPath.length > 0) {
    breadcrumbHtml += ` > <span id="unitName" class="cursor-pointer hover:underline" data-unit="${selectedPath[0]}">${selectedPath[0]}</span>`;
    if (selectedPath.length > 1) {
      breadcrumbHtml += ` > <span id="zoneName" class="cursor-pointer hover:underline" data-unit="${selectedPath[0]}" data-zone="${selectedPath[1]}">${selectedPath[1]}</span>`;
      if (selectedPath.length > 2) {
        breadcrumbHtml += ` > <span id="lineName" class="cursor-pointer hover:underline" data-unit="${selectedPath[0]}" data-zone="${selectedPath[1]}" data-line="${selectedPath[2]}">${selectedPath[2]}</span>`;
      }
    }
  }
  $("#unitBreadcrumb").html(breadcrumbHtml);

  // Update Unit Chart title
  let unitChartTitle = "";
  if (currentLevel === "units") {
    unitChartTitle = "Unit-wise Lost Minutes";
  } else if (currentLevel === "zones") {
    unitChartTitle = `Zone-wise Lost Minutes: ${selectedPath[0]}`;
  } else if (currentLevel === "lines") {
    unitChartTitle = `Line-wise Lost Minutes: ${selectedPath[0]} / ${selectedPath[1]}`;
  }
  if (selectedPath.length === 0 && reasonPath.length === 0) {
    unitChartTitle += " (Overall)";
  }
  $("#unitChartTitle").text(unitChartTitle);

  // Update Trend and Log Table titles
  let pathTitle = "Overall";
  if (selectedPath.length > 0 || reasonPath.length > 0) {
    const pathParts = [];
    if (selectedPath[0]) pathParts.push(selectedPath[0]);
    if (selectedPath[1]) pathParts.push(selectedPath[1]);
    if (selectedPath[2]) pathParts.push(selectedPath[2]);
    if (reasonPath[0]) {
      const fullReason = apiData?.find(u => u.zones?.some(z => z.lines?.some(l => l.data?.some(row => row.Reason_Lkup === reasonPath[0]))))?.zones
        ?.flatMap(z => z.lines?.flatMap(l => l.data?.filter(row => row.Reason_Lkup === reasonPath[0])) || [])[0]?.Reason || reasonPath[0];
      pathParts.push(fullReason);
    }
    if (reasonPath[1]) pathParts.push(reasonPath[1].slice(-4));
    pathTitle = pathParts.join(" / ");
  }
  $("#trendChartTitle").text(`Trend for ${pathTitle} - Lost Minutes`);
  //$("#logTableTitle").text(`Line Stop Log: ${pathTitle}`);
}

function updateReasonBreadcrumb() {
  let breadcrumbHtml = `<span id="allReasons" class="cursor-pointer hover:underline">All Reasons</span>`;
  if (reasonPath.length > 0) {
    breadcrumbHtml += ` > <span id="reasonName" class="cursor-pointer hover:underline" data-reason="${reasonPath[0]}">${reasonPath[0]}</span>`;
    if (reasonPath.length > 1) {
      breadcrumbHtml += ` > <span id="ocnName" class="cursor-pointer hover:underline" data-reason="${reasonPath[0]}" data-ocn="${reasonPath[1]}">${reasonPath[1].slice(-4)}</span>`;
    }
  }
  $("#reasonBreadcrumb").html(breadcrumbHtml);

  // Update Reason Chart title
  let reasonChartTitle = "";
  if (reasonLevel === "reasons") {
    reasonChartTitle = "Reasons for Lost Minutes";
  } else if (reasonLevel === "ocns") {
    const fullReason = apiData?.find(u => u.zones?.some(z => z.lines?.some(l => l.data?.some(row => row.Reason_Lkup === reasonPath[0]))))?.zones
      ?.flatMap(z => z.lines?.flatMap(l => l.data?.filter(row => row.Reason_Lkup === reasonPath[0])) || [])[0]?.Reason || reasonPath[0];
    reasonChartTitle = `OCNs for Lost Minutes: ${fullReason}`;
  }
  if (selectedPath.length === 0 && reasonPath.length === 0) {
    reasonChartTitle += " (Overall)";
  }
  $("#reasonChartTitle").text(reasonChartTitle);
}

function updateTrendChart() {
  const trendData = getTrendData(selectedPath[0], selectedPath[1], selectedPath[2], reasonPath[0], reasonPath[1]);
  if (trendData.labels.length > 0) {
    $("#trendChart").removeClass("hidden");
    $("#trendNoData").hide();
    if (trendChart) {
      trendChart.data.labels = trendData.labels;
      trendChart.data.datasets[0].data = trendData.values;
      trendChart.update();
    } else {
      trendChart = new Chart(document.getElementById("trendChart"), {
        type: "line",
        data: {
          labels: trendData.labels,
          datasets: [{ data: trendData.values, borderColor: "#2563eb", fill: false, tension: 0.4 }]
        },
        options: {
          plugins: { legend: { display: false } },
          scales: { y: { beginAtZero: true } },
          onClick: () => {}
        }
      });
    }
  } else {
    $("#trendChart").addClass("hidden");
    $("#trendNoData").show();
    if (trendChart) {
      trendChart.destroy();
      trendChart = null;
    }
  }
}

function initializeCharts() {
  const isApiDataValid = apiData && Array.isArray(apiData) && apiData.length > 0;
  if (!isApiDataValid) {
    showNoData();
    return;
  }
  showData();
  const filterReason = reasonPath[0];
  const filterOcn = reasonPath[1];
  const unitData = getUnitChartData(currentLevel, selectedPath[0], selectedPath[1], filterReason, filterOcn);
  if (unitData.labels.length === 0) {
    $("#unitChart").addClass("hidden");
    $("#unitNoData").show();
    unitChart = null;
  } else {
    $("#unitChart").removeClass("hidden");
    $("#unitNoData").hide();
    if (unitChart) unitChart.destroy();
    const colors = currentLevel === "lines" && selectedPath[2] ? unitData.labels.map(l => l === selectedPath[2] ? "#1e3a8a" : "#cbd5e1") : "#cbd5e1";
    unitChart = new Chart(document.getElementById("unitChart"), {
      type: "bar",
      data: {
        labels: unitData.labels,
        datasets: [{ data: unitData.values, backgroundColor: colors }]
      },
      options: {
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } },
        onClick: (e, elements) => {
          if (!elements.length) return;
          const index = elements[0].index;
          const label = unitChart.data.labels[index];
          if (currentLevel === "units") {
            selectedPath = [label];
            currentLevel = "zones";
            const zoneData = getUnitChartData("zones", label, null, reasonPath[0], reasonPath[1]);
            if (zoneData.labels.length > 0) {
              unitChart.data.labels = zoneData.labels;
              unitChart.data.datasets[0].data = zoneData.values;
              unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
              unitChart.update();
            } else {
              currentLevel = "units";
              selectedPath = [];
              return;
            }
          } else if (currentLevel === "zones") {
            selectedPath = [selectedPath[0], label];
            currentLevel = "lines";
            const lineData = getUnitChartData("lines", selectedPath[0], label, reasonPath[0], reasonPath[1]);
            if (lineData.labels.length > 0) {
              unitChart.data.labels = lineData.labels;
              unitChart.data.datasets[0].data = lineData.values;
              unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
              unitChart.update();
            } else {
              currentLevel = "zones";
              selectedPath = [selectedPath[0]];
              return;
            }
          } else if (currentLevel === "lines") {
            selectedPath = [selectedPath[0], selectedPath[1], label];
            unitChart.data.datasets[0].backgroundColor = unitChart.data.labels.map(l => l === label ? "#1e3a8a" : "#cbd5e1");
            unitChart.update();
          }
          const filterUnit = selectedPath[0];
          const filterZone = selectedPath[1];
          const filterLine = selectedPath[2];
          let reasonData = getReasonChartData(reasonLevel, reasonPath[0], filterUnit, filterZone, filterLine);
          if (reasonData.labels.length === 0 && reasonLevel !== "reasons") {
            reasonLevel = "reasons";
            reasonPath = [];
            reasonData = getReasonChartData("reasons", null, filterUnit, filterZone, filterLine);
          }
          if (reasonData.labels.length > 0) {
            $("#reasonChart").removeClass("hidden");
            $("#reasonNoData").hide();
            const highlight = reasonLevel === "ocns" ? reasonPath[1] : null;
            renderReasonChart(reasonData.labels, reasonData.values, highlight);
          } else {
            $("#reasonChart").addClass("hidden");
            $("#reasonNoData").show();
          }
          updateTrendChart();
          updateKPICards();
          updateLogTable();
          resetLogTableInstance();
          // Rebuild the table to reflect updated DataTable configuration
          updateLogTable();
          updateBreadcrumb();
          updateReasonBreadcrumb();
        }
      }
    });
  }
  const filterUnit = selectedPath[0];
  const filterZone = selectedPath[1];
  const filterLine = selectedPath[2];
  const reasonData = getReasonChartData(reasonLevel, reasonPath[0], filterUnit, filterZone, filterLine);
  if (reasonData.labels.length === 0) {
    $("#reasonChart").addClass("hidden");
    $("#reasonNoData").show();
  } else {
    $("#reasonChart").removeClass("hidden");
    $("#reasonNoData").hide();
    const highlight = reasonLevel === "ocns" ? reasonPath[1] : null;
    renderReasonChart(reasonData.labels, reasonData.values, highlight);
  }
  updateTrendChart();
}

function renderReasonChart(labels, values, highlightLabel = null) {
  const displayLabels = reasonLevel === 'ocns' ? labels.map(l => l.match(/\d{4}$/)?.[0] || l) : labels;
  const total = values.reduce((a, b) => a + (b || 0), 0) || 1;
  let running = 0;
  const sortedGroups = displayLabels;
  const barCount = sortedGroups.length;
  const fontSize = barCount > 10 ? 6 : 10;
  const cumulativePct = values.map(v => {
    running += (v || 0);
    return +(running / total * 100).toFixed(1);
  });
  const barColors = highlightLabel
    ? labels.map(l => l === highlightLabel ? '#1e3a8a' : '#1d4ed8')
    : '#1d4ed8';
  const tooltipData = labels.map((label, index) => {
    const value = values[index];
    let stops = 0;
    let financialLoss = 0;
	 let buyer = '-';
    apiData.forEach(unit => {
      if (unit.zones && Array.isArray(unit.zones)) {
        unit.zones.forEach(zone => {
          if (zone.lines && Array.isArray(zone.lines)) {
            zone.lines.forEach(line => {
              if (line.data && Array.isArray(line.data)) {
                line.data.forEach(row => {
                  if (reasonLevel === 'reasons' && row.Reason_Lkup === label) {
                    stops++;
                    financialLoss += row.Loss_Amnt || 0;
                    if (buyer === '-' && row.Buyer) buyer = row.Buyer;
                  } else if (reasonLevel === 'ocns' && row.Order_No === label && 
                           reasonPath[0] && row.Reason_Lkup === reasonPath[0]) {
                    stops++;
                    financialLoss += row.Loss_Amnt || 0;
                    if (buyer === '-' && row.Buyer) buyer = row.Buyer;
                  }
                });
              }
            });
          }
        });
      }
    });
    return {
      lossMinutes: value,
      stops: stops,
      financialLoss: financialLoss,
      cumulative: cumulativePct[index],
	  buyer: buyer
    };
  });
  const data = {
    labels: displayLabels,
    datasets: [
      {
        type: 'bar',
        data: values,
        backgroundColor: barColors,
        borderWidth: 0,
      },
      {
        type: 'line',
        data: cumulativePct,
        xAxisID: 'x2',
        borderColor: '#47D138',
        backgroundColor: '#DC2626',
        tension: 0,
        pointRadius: 5,
        pointBackgroundColor: '#DC2626',
        pointBorderColor: '#ffffff',
        pointBorderWidth: 2,
        fill: false,
        yAxisID: undefined,
        pointHoverRadius: 7,
        pointHoverBorderWidth: 3,
        hoverBackgroundColor: 'transparent', // Disable hover effect for line
        hoverBorderColor: 'transparent'     // Disable hover effect for line
      }
    ]
  };
  const options = {
    plugins: { 
      legend: { display: false }, 
      tooltip: { 
        mode: 'index', 
        intersect: true, // Only trigger tooltip when directly over a bar
        backgroundColor: 'rgba(255, 255, 255, 0.95)',
        titleColor: '#1f2937',
        bodyColor: '#374151',
        borderColor: '#e5e7eb',
        borderWidth: 1,
        cornerRadius: 8,
        displayColors: false,
        filter: function(context) {
          return context.datasetIndex === 0; // Only show tooltip for bar dataset
        },
        callbacks: {
          title: function(context) {
            const shortReason = context[0].label;
            let fullReasonName = shortReason;
            if (apiData && Array.isArray(apiData)) {
              apiData.forEach(unit => {
                if (unit.zones && Array.isArray(unit.zones)) {
                  unit.zones.forEach(zone => {
                    if (zone.lines && Array.isArray(zone.lines)) {
                      zone.lines.forEach(line => {
                        if (line.data && Array.isArray(line.data)) {
                          line.data.forEach(row => {
                            if (row.Reason_Lkup === shortReason && row.Reason) {
                              fullReasonName = row.Reason;
                              return;
                            }
                          });
                        }
                      });
                    }
                  });
                }
              });
            }
            return fullReasonName;
          },
          afterTitle: function(context) {
            return '';
          },
          label: function(context) {
            if (context.datasetIndex !== 0) return ''; // Skip line dataset
            const index = context.dataIndex;
            const data = tooltipData[index];
            return [
              `Lost Minutes: ${data.lossMinutes.toLocaleString()}`,
              `Stops: ${data.stops}`,
              `Financial Loss: ${data.financialLoss.toLocaleString()}`,
              `Cumulative: ${data.cumulative}%`,
			  `Buyer : ${data.buyer}`
            ];
          }
        }
      }
    },
    indexAxis: 'y',
    scales: {
      y: {
        ticks: {
          font: {
            size: fontSize
          }
        }
      },
      x: {
        beginAtZero: true,
        position: 'bottom',
        grid: { display: true },
        ticks: { callback: val => Number(val).toLocaleString() }
      },
      x2: {
        beginAtZero: true,
        position: 'top',
        min: 0,
        max: 100,
        grid: { display: false },
        ticks: { callback: val => `${val}%` }
      }
    },
    onClick: (e, elements) => {
      if (!elements.length) return;
      const index = elements[0].index;
      const label = labels[index];
      if (reasonLevel === "reasons") {
        reasonPath = [label];
        reasonLevel = "ocns";
        const ocnData = getReasonChartData("ocns", label, selectedPath[0], selectedPath[1], selectedPath[2]);
        if (ocnData.labels.length === 0) {
          reasonLevel = "reasons";
          reasonPath = [];
          return;
        }
        renderReasonChart(ocnData.labels, ocnData.values);
      } else if (reasonLevel === "ocns") {
        reasonPath = [reasonPath[0], label];
        reasonChart.data.datasets[0].backgroundColor = labels.map(l => l === label ? '#1e3a8a' : '#1d4ed8');
        reasonChart.update();
      }
      const filterReason = reasonPath[0];
      const filterOcn = reasonPath[1];
      let unitData = getUnitChartData(currentLevel, selectedPath[0], selectedPath[1], filterReason, filterOcn);
      if (unitData.labels.length === 0 && currentLevel !== "units") {
        currentLevel = "units";
        selectedPath = [];
        unitData = getUnitChartData("units", null, null, filterReason, filterOcn);
      }
      if (unitData.labels.length > 0) {
        $("#unitChart").removeClass("hidden");
        $("#unitNoData").hide();
        const colors = currentLevel === "lines" && selectedPath[2] ? unitData.labels.map(l => l === selectedPath[2] ? "#1e3a8a" : "#cbd5e1") : "#cbd5e1";
        unitChart.data.labels = unitData.labels;
        unitChart.data.datasets[0].data = unitData.values;
        unitChart.data.datasets[0].backgroundColor = colors;
        unitChart.update();
      } else {
        $("#unitChart").addClass("hidden");
        $("#unitNoData").show();
      }
      updateTrendChart();
      updateKPICards();
      updateLogTable();
      updateBreadcrumb();
      updateReasonBreadcrumb();
    }
  };
  if (reasonChart) {
    reasonChart.data = data;
    reasonChart.options = options;
    reasonChart.update();
  } else {
    reasonChart = new Chart(document.getElementById('reasonChart'), { data, options });
  }
}

$(document).ready(function () {

  // ========== NEW CODE: Extract URL parameters ==========
  const urlParams = new URLSearchParams(window.location.search);
  const fromDateParam = urlParams.get('fromdate');
  const endDateParam = urlParams.get('enddate');

  // Initialize date range variables
  let initialStart, initialEnd, initialStartFormatted, initialEndFormatted;
  let initialDisplayStartFormatted, initialDisplayEndFormatted;
  let selectedRangeLabel = 'Last Week'; // Default range

  if (fromDateParam && endDateParam && moment(fromDateParam, 'YYYY-MM-DD', true).isValid() && moment(endDateParam, 'YYYY-MM-DD', true).isValid()) {
    // Use URL parameters if valid
    initialStart = moment(fromDateParam, 'YYYY-MM-DD');
    initialEnd = moment(endDateParam, 'YYYY-MM-DD');
    initialStartFormatted = initialStart.format('YYYY-MM-DD');
    initialEndFormatted = initialEnd.format('YYYY-MM-DD');
	selectedfromdate = initialStartFormatted;
    selectedenddate = initialEndFormatted;
    initialDisplayStartFormatted = initialStart.format('DD-MM-YY');
    initialDisplayEndFormatted = initialEnd.format('DD-MM-YY');

    // Determine the selected range label
    if (initialStart.isSame(initialEnd, 'day')) {
      selectedRangeLabel = 'Today';
    } else if (initialStart.isSame(moment().subtract(6, 'days'), 'day') && initialEnd.isSame(moment(), 'day')) {
      selectedRangeLabel = 'Last Week';
    } else if (initialStart.isSame(moment().startOf('month'), 'day') && initialEnd.isSame(moment().endOf('month'), 'day')) {
      selectedRangeLabel = 'This Month';
    } else {
      selectedRangeLabel = 'Custom Range';
    }
  } else {
    // Fallback to default (Last Week)
    initialStart = moment().subtract(6, 'days');
    initialEnd = moment();
    initialStartFormatted = initialStart.format('YYYY-MM-DD');
    initialEndFormatted = initialEnd.format('YYYY-MM-DD');
	selectedfromdate = initialStartFormatted;
    selectedenddate = initialEndFormatted;
    initialDisplayStartFormatted = initialStart.format('DD-MM-YY');
    initialDisplayEndFormatted = initialEnd.format('DD-MM-YY');
  }
  // ========== END NEW CODE ==========

  $('#dateRangePicker').daterangepicker({
    opens: 'left',
    startDate: initialStart,  // 
    endDate: initialEnd,      // 
    maxDate: moment(), // Restrict dates beyond today
    ranges: {
        'Today': [moment(), moment()],
        'Last Week': [moment().subtract(6, 'days'), moment()],
        'This Month': [moment().startOf('month'), moment().endOf('month')]
    },
    locale: {
      format: 'DD-MM-YY'
    }
  }, function(start, end, label) {
    const startFormatted = start.format('YYYY-MM-DD');
    const endFormatted = end.format('YYYY-MM-DD');
	selectedfromdate = startFormatted;
    selectedenddate = endFormatted;
    const displayStartFormatted = start.format('DD-MM-YY');
    const displayEndFormatted = end.format('DD-MM-YY');
    selectedRangeLabel = label;
    const isSameDay = start.isSame(end, 'day');

    const titleText = `
    Line Stop Analysis for - 
    <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
        GARMENTS
    </span>
    <span class="ml-4" style="color: #fff; font-weight: 500;">
        ${displayStartFormatted} - ${displayEndFormatted}
    </span>
    `;

    $('.dashboard_title').html(titleText);

    $('#dateRangePicker').val(`${displayStartFormatted} - ${displayEndFormatted}`);
    currentLevel = "units";
    selectedPath = [];
    reasonLevel = "reasons";
    reasonPath = [];
    currentPage = 1;
    // Update breadcrumb link with selected dates
    const breadcrumbUrl = `/production_gar_dashboard?fromdate=${startFormatted}&enddate=${endFormatted}`;
    $('#productionBreadcrumb').attr('href', breadcrumbUrl);
    fetchData(selectedRangeLabel, startFormatted, endFormatted);
  });

  // ========== UPDATED INITIAL SETUP ==========
  $('.dashboard_title').html(`
  Line Stop Analysis for 
    <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
      GARMENTS
    </span>
    <span class="ml-4" style="color: #fff; font-weight: 500;">
      ${initialDisplayStartFormatted} - ${initialDisplayEndFormatted}
    </span>
  `);

  $('#dateRangePicker').val(`${initialDisplayStartFormatted} - ${initialDisplayEndFormatted}`);
  // Set initial breadcrumb link with initial dates
  const initialBreadcrumbUrl = `/production_gar_dashboard?fromdate=${initialStartFormatted}&enddate=${initialEndFormatted}`;
  $('#productionBreadcrumb').attr('href', initialBreadcrumbUrl);
  fetchData('Last Week', initialStartFormatted, initialEndFormatted);

  $("#unitBreadcrumb").on("click", "#allUnits", () => {
    currentLevel = "units";
    selectedPath = [];
    reasonLevel = "reasons";
    reasonPath = [];
    const unitData = getUnitChartData("units", null, null, null, null);
    if (unitData.labels.length > 0) {
      $("#unitChart").removeClass("hidden");
      $("#unitNoData").hide();
      unitChart.data.labels = unitData.labels;
      unitChart.data.datasets[0].data = unitData.values;
      unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, null, null, null);
    if (reasonData.labels.length > 0) {
      $("#reasonChart").removeClass("hidden");
      $("#reasonNoData").hide();
      renderReasonChart(reasonData.labels, reasonData.values);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#unitBreadcrumb").on("click", "#unitName", function() {
    const unit = $(this).data("unit");
    currentLevel = "zones";
    selectedPath = [unit];
    const zoneData = getUnitChartData("zones", unit, null, reasonPath[0], reasonPath[1]);
    if (zoneData.labels.length > 0) {
      unitChart.data.labels = zoneData.labels;
      unitChart.data.datasets[0].data = zoneData.values;
      unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, unit, null, null);
    if (reasonData.labels.length > 0) {
      const highlight = reasonLevel === "ocns" ? reasonPath[1] : null;
      renderReasonChart(reasonData.labels, reasonData.values, highlight);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#unitBreadcrumb").on("click", "#zoneName", function() {
    const unit = $(this).data("unit");
    const zone = $(this).data("zone");
    currentLevel = "lines";
    selectedPath = [unit, zone];
    const lineData = getUnitChartData("lines", unit, zone, reasonPath[0], reasonPath[1]);
    if (lineData.labels.length > 0) {
      unitChart.data.labels = lineData.labels;
      unitChart.data.datasets[0].data = lineData.values;
      unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, unit, zone, null);
    if (reasonData.labels.length > 0) {
      const highlight = reasonLevel === "ocns" ? reasonPath[1] : null;
      renderReasonChart(reasonData.labels, reasonData.values, highlight);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#unitBreadcrumb").on("click", "#lineName", function() {
    const unit = $(this).data("unit");
    const zone = $(this).data("zone");
    const line = $(this).data("line");
    currentLevel = "lines";
    selectedPath = [unit, zone, line];
    const lineData = getUnitChartData("lines", unit, zone, reasonPath[0], reasonPath[1]);
    if (lineData.labels.length > 0) {
      unitChart.data.labels = lineData.labels;
      unitChart.data.datasets[0].data = lineData.values;
      unitChart.data.datasets[0].backgroundColor = lineData.labels.map(l => l === line ? "#1e3a8a" : "#cbd5e1");
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, unit, zone, line);
    if (reasonData.labels.length > 0) {
      const highlight = reasonLevel === "ocns" ? reasonPath[1] : null;
      renderReasonChart(reasonData.labels, reasonData.values, highlight);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#reasonBreadcrumb").on("click", "#allReasons", () => {
    currentLevel = "units";
    selectedPath = [];
    reasonLevel = "reasons";
    reasonPath = [];
    const unitData = getUnitChartData("units", null, null, null, null);
    if (unitData.labels.length > 0) {
      $("#unitChart").removeClass("hidden");
      $("#unitNoData").hide();
      unitChart.data.labels = unitData.labels;
      unitChart.data.datasets[0].data = unitData.values;
      unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, null, null, null);
    if (reasonData.labels.length > 0) {
      $("#reasonChart").removeClass("hidden");
      $("#reasonNoData").hide();
      renderReasonChart(reasonData.labels, reasonData.values);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#reasonBreadcrumb").on("click", "#reasonName", function() {
    const reason = $(this).data("reason");
    reasonLevel = "ocns";
    reasonPath = [reason];
    const ocnData = getReasonChartData("ocns", reason, selectedPath[0], selectedPath[1], selectedPath[2]);
    if (ocnData.labels.length > 0) {
      renderReasonChart(ocnData.labels, ocnData.values);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    const unitData = getUnitChartData(currentLevel, selectedPath[0], selectedPath[1], reason, null);
    if (unitData.labels.length > 0) {
      const colors = currentLevel === "lines" && selectedPath[2] ? unitData.labels.map(l => l === selectedPath[2] ? "#1e3a8a" : "#cbd5e1") : "#cbd5e1";
      unitChart.data.labels = unitData.labels;
      unitChart.data.datasets[0].data = unitData.values;
      unitChart.data.datasets[0].backgroundColor = colors;
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#reasonBreadcrumb").on("click", "#ocnName", function() {
    const reason = $(this).data("reason");
    const ocn = $(this).data("ocn");
    reasonLevel = "ocns";
    reasonPath = [reason, ocn];
    const ocnData = getReasonChartData("ocns", reason, selectedPath[0], selectedPath[1], selectedPath[2]);
    if (ocnData.labels.length > 0) {
      renderReasonChart(ocnData.labels, ocnData.values, ocn);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    const unitData = getUnitChartData(currentLevel, selectedPath[0], selectedPath[1], reason, ocn);
    if (unitData.labels.length > 0) {
      const colors = currentLevel === "lines" && selectedPath[2] ? unitData.labels.map(l => l === selectedPath[2] ? "#1e3a8a" : "#cbd5e1") : "#cbd5e1";
      unitChart.data.labels = unitData.labels;
      unitChart.data.datasets[0].data = unitData.values;
      unitChart.data.datasets[0].backgroundColor = colors;
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });

  $("#prevPage").click(() => {
    if (currentPage > 1) {
      updateLogTable(currentPage - 1);
    }
  });

  $("#nextPage").click(() => {
    updateLogTable(currentPage + 1);
  });

  $(`#HistoricalButton`).on('click', function () {
      window.location.href = `/history_dashboard/Line Stop?fromdate=${selectedfromdate}&enddate=${selectedenddate}`;
  });

  $(".kpi-card").click(function () {
    currentLevel = "units";
    selectedPath = [];
    reasonLevel = "reasons";
    reasonPath = [];
    const unitData = getUnitChartData("units", null, null, null, null);
    if (unitData.labels.length > 0 && unitChart) {
      unitChart.data.labels = unitData.labels;
      unitChart.data.datasets[0].data = unitData.values;
      unitChart.data.datasets[0].backgroundColor = "#cbd5e1";
      unitChart.update();
    } else {
      $("#unitChart").addClass("hidden");
      $("#unitNoData").show();
    }
    const reasonData = getReasonChartData("reasons", null, null, null, null);
    if (reasonData.labels.length > 0 && reasonChart) {
      renderReasonChart(reasonData.labels, reasonData.values);
    } else {
      $("#reasonChart").addClass("hidden");
      $("#reasonNoData").show();
    }
    updateTrendChart();
    updateKPICards();
    updateLogTable();
    updateBreadcrumb();
    updateReasonBreadcrumb();
  });
});
</script>
</body>
</html><?php /**PATH C:\Apache24\htdocs\erpportal\resources\views/LifeAtGainup/Production/Linestop_analysis_dashboard.blade.php ENDPATH**/ ?>