<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gainup - Yearly Budget Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.datatables.net/2.1.8/css/dataTables.dataTables.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/buttons/3.0.1/css/buttons.dataTables.min.css">


	<script src="https://code.jquery.com/jquery-3.7.1.js"></script>
<script src="https://cdn.datatables.net/2.1.8/js/dataTables.min.js"></script>

<script src="https://cdn.datatables.net/buttons/3.0.1/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.1/js/buttons.html5.min.js"></script>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>

    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/moment.js/2.29.4/moment.min.js"></script>

    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        html, body { height: 100%; }
        body { background: #f8f9fa; font-family: 'Segoe UI', Roboto, -apple-system, BlinkMacSystemFont, Arial; font-size: 14px; color: #12263a; }
        
        .viewport { 
            display: flex; 
            flex-direction: column; 
            height: 100vh; 
            overflow: hidden;
            padding: 4px;
            gap: 4px;
        }
        
        .budget-header {
			display: flex;
			justify-content: space-between;
			align-items: center;   /* aligns header-left and filters in one line */
			padding: 12px 16px;
			border-radius: 6px;
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			color: #fff;
		}

		/* Stack heading + subtext correctly */
		.header-left {
			display: flex;
			flex-direction: column;
			justify-content: center; /* vertical alignment fix */
			gap: 4px;
		}

		/* Heading style */
		.header-left h4 {
			margin: 0;
			font-size: 1.3rem;
			font-weight: 700;
			line-height: 1.2;
		}

		/* Subtitle style */
		.header-left .subtext {
			font-size: 0.85rem;
			opacity: 0.9;
			margin-top: -2px;
		}

		/* Filter bar stays single row */
		.header-controls {
			display: flex;
			align-items: center;
			gap: 8px;
			flex-wrap: nowrap;
		}

		/* Select styling */
		.header-select {
			flex: 1;
			min-width: 120px;
			max-width: 160px;
			padding: 6px 10px;
			font-size: 0.8rem;
			border: none;
			border-radius: 4px;
			color: #333;
			background: #fff;
			white-space: nowrap;
			overflow: hidden;
			text-overflow: ellipsis;
		}

		/* Reset button */
		.btn-header {
			padding: 6px 12px;
			background: rgba(255,255,255,0.25);
			border: none;
			border-radius: 4px;
			color: #fff;
			cursor: pointer;
			flex-shrink: 0;
		}
		.btn-header:hover {
			background: rgba(255,255,255,0.35);
		}

        
        .content-grid {
            display: grid;
            grid-template-columns: 1.8fr 0.9fr 0.8fr;
            gap: 4px;
            flex-shrink: 0;
            height: auto;
        }
        
        .card-section { 
            background: white; 
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 10px;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
            box-shadow: 0 1px 3px rgba(16,24,40,0.04);
        }
        
        .card-section h6 { 
            font-size: 0.9rem; 
            font-weight: 600; 
            margin: 0 0 8px 0; 
            color: #222;
            flex-shrink: 0;
        }
        
        .chart-box { 
            position: relative; 
            height: 120px;
            width: 100%;
            flex-shrink: 0;
        }
        
        .grid-table {
            width: 100%;
            font-size: 0.8rem;
            border-collapse: collapse;
        }
        
        .grid-table thead {
            background: #f8f9fa;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .grid-table th {
            padding: 10px 8px;
            font-weight: 700;
            text-align: left;
            border-bottom: 2px solid #dce6ea;
            font-size: 0.85rem;
            color: #0f1724;
            background: #fafcfe;
        }
        
        .grid-table td {
            padding: 10px 8px;
            border-bottom: 1px solid #f3f6f7;
            color: #12263a;
            font-size: 0.9rem;
        }
        
        .kpi-summary {
            display: flex;
            flex-direction: column;
            gap: 6px;
            flex: 1;
            min-height: 0;
            overflow-y: auto;
        }
        
        .kpi-row {
            display: flex;
            justify-content: space-between;
            padding: 4px 4px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 0.8rem;
        }
        
        .kpi-row:last-child {
            border-bottom: none;
        }
        
        .kpi-label { 
            color: #666; 
            font-weight: 500;
        }
        
        .kpi-value { 
            color: #333;
            font-weight: 600;
        }
        
        .controls-section {
            display: flex;
            flex-direction: column;
            gap: 6px;
        }
        
        .control-group {
            display: flex;
            flex-direction: column;
        }
        
        .control-group label {
            font-size: 0.75rem;
            color: #666;
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .control-group select {
            font-size: 0.8rem;
            padding: 4px 6px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .btn-control {
            padding: 5px 10px;
            font-size: 0.75rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
        }
        
        .btn-primary-sm {
            background: #667eea;
            color: white;
        }
        
        .btn-primary-sm:hover {
            background: #5568d3;
        }
        
        .btn-secondary-sm {
            background: #e0e0e0;
            color: #333;
        }
        
        .btn-secondary-sm:hover {
            background: #d0d0d0;
        }
        
        .left-charts {
            display: contents;
        }
        
        .chart-card {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 6px 8px;
            display: flex;
            flex-direction: column;
            height: fit-content;
            box-shadow: 0 1px 2px rgba(16,24,40,0.04);
        }
        
        .chart-card h6 {
            font-size: 0.9rem;
            font-weight: 700;
            margin: 0 0 4px 0;
            color: #0f1724;
            flex-shrink: 0;
        }
        
        .right-controls {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 6px 8px;
            display: flex;
            flex-direction: column;
            box-shadow: 0 1px 3px rgba(16,24,40,0.03);
        }
        
        .full-grid {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 8px;
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            box-shadow: 0 1px 4px rgba(16,24,40,0.04);
        }
        
        .full-grid h6 {
            font-size: 0.9rem;
            font-weight: 600;
            margin: 0 0 8px 0;
            color: #222;
            flex-shrink: 0;
        }
        
        .grid-scroll {
            overflow-y: auto;
            flex: 1;
            min-height: 0;
        }
        
        @media (max-width: 1400px) {
            .content-grid {
                grid-template-columns: 1fr 1fr;
            }
            .right-controls {
                grid-column: 1 / -1;
            }
        }
        
        @media (max-width: 1024px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
        }
		
		/* Make toolbar row flex */
.dataTables_wrapper .dt-layout-row {
    display: flex !important;
    align-items: center !important;
}

/* Push everything to the right */
.dataTables_wrapper .dt-layout-cell {
    display: flex;
    justify-content: flex-end !important;
    gap: 10px;
}

/* Smaller button */
.dt-button {
    padding: 4px 10px !important;
    font-size: 12px !important;
    border-radius: 4px !important;
}

/* Smaller Search input */
.dataTables_filter input,
.dt-search input {
    height: 28px !important;
    font-size: 12px !important;
}
/* Make all table cells stay in one line */
#budgetgrid td, 
#budgetgrid th {
    white-space: nowrap !important;
    overflow: hidden;
    text-overflow: ellipsis;   /* show ... when too long */
    vertical-align: middle !important;
    padding-top: 6px !important;
    padding-bottom: 6px !important;
}

/* Make table more compact */
#budgetgrid {
    font-size: 13px;
    line-height: 1.2;
}

/* Fix header alignment */
#budgetgrid thead th {
    font-weight: 600 !important;
}
#budgetgrid td {
    padding: 4px 8px !important;
}

#budgetgrid th {
    white-space: nowrap !important;
}


/* ON SMALL SCREENS → wrap into 2 rows automatically */
@media (max-width: 900px) {
    .header-controls {
        flex-wrap: wrap;
    }
    .header-select {
        flex: 1 1 45%;       /* 2 columns */
        max-width: none;
    }
}

/* Right-align number columns */
#budgetgrid td:nth-child(7),
#budgetgrid td:nth-child(8),
#budgetgrid td:nth-child(9),
#budgetgrid td:nth-child(10),
#budgetgrid td:nth-child(11) {
    text-align: right !important;
}

#budgetgrid th:nth-child(7),
#budgetgrid th:nth-child(8),
#budgetgrid th:nth-child(9),
#budgetgrid th:nth-child(10),
#budgetgrid th:nth-child(11) {
    text-align: right !important;
}
.grid-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
    padding: 0 4px;
}

.grid-title {
    margin: 0;
    font-size: 0.9rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 4px;
}

/* Make Excel button look like DataTables export button */
.excel-btn {
    padding: 4px 12px;
    font-size: 12px;
    border-radius: 4px;
    background: #e0e0e0;
    border: 1px solid #d0d0d0;
    cursor: pointer;
}

.excel-btn:hover {
    background: #d5d5d5;
}
#budgetgrid td {
    max-width: 120px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}
    </style>
</head>
<body>
    <div class="viewport">
        <!-- Header -->
        <div class="budget-header">
			<div class="header-left">
				<h4>Budget Dashboard</h4>
				<span class="subtext">Annual Budget Allocation & Usage Tracking</span>
			</div>

			<div class="header-controls">
				<select id="companyFilter" class="header-select"><option value="">Company</option></select>
				<select id="divisionFilter" class="header-select"><option value="">Division</option></select>
				<select id="departmentFilter" class="header-select"><option value="">Department</option></select>
				<select id="projectFilter" class="header-select"><option value="">Project</option></select>
				<select id="activityFilter" class="header-select"><option value="">Activity</option></select>
				<select id="employeeFilter" class="header-select"><option value="">Employee</option></select>

				<button class="btn-header" onclick="resetFilters()">🔄 Reset</button>
			</div>
		</div>


        <!-- Main Content Grid (3 columns: Chart1, Chart2, Summary) -->
        <div class="content-grid">
            <div class="left-charts">
                <!-- Chart 1 -->
                <div class="chart-card">
                    <h6>📊 As On Month vs Used Budget</h6>
                    <div class="chart-box">
                        <canvas id="asonBarChart"></canvas>
                    </div>
                </div>

                <!-- Chart 2 -->
                <div class="chart-card">
                    <h6>🎯 Total Budget Allocation</h6>
                    <div class="chart-box">
                        <canvas id="totalDoughnutChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Summary -->
            <div class="right-controls">
                <h6>📈 Summary <!--<span style="color:red;font-size: 10px;">(Value in thousands) </span>--></h6>
                <div class="kpi-summary" id="quickSummary">
                    <!-- Populated by JS -->
                </div>
            </div>
        </div>

        <!-- Full Width Grid -->
        <div class="full-grid">
            <div class="grid-header">
				<h6 class="grid-title">📄 Department Budget Details </h6>

				<button id="excelBtn" class="excel-btn dt-button">Excel</button>
			</div>
            <div class="grid-scroll">
                <table class="grid-table" id="budgetgrid">
                    <thead>
                        <tr>
							<th>Company</th>
							<th>Project ID</th>
                            <th>Division</th>
                            <th>Project</th>
                            <th>Activity</th>
                            <!--<th>Description</th>-->
                            <th>Budget<br> Incharge</th>  
							<th>Total <br>Budget (A)</th>                          
                            <th>Approved <br>budget(B)</th>
							<th>Unapproved <br>budget(A-B)</th>
                            <th>As On <br>Month (C)</th>
                            <th>Used <br>Budget (D)</th>
                            <th>Pending <br>Budget (C-D)</th>
							<th>Budget <br>Overrun</th>
                        </tr>
                    </thead>
                    <tbody id="budgetGrid">
                        <!-- Populated by JS -->
                    </tbody>
					
                </table>
            </div>
        </div>
    </div>
    <!-- Modal -->
    <div class="modal fade" id="orderDetailsModal" tabindex="-1">
      <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title" id="projecttitle">Order No Details</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
          </div>
          <div class="modal-body" id="modalContent">
            Loading...
            
          </div>
          	  
        </div>
		
      </div>
    </div>

   <script>
let budgetData = [];
let currentData = [];
let asonChart, doughnutChart;
let gridTable;  // GLOBAL


/* ------------------ Helpers ------------------ */
function formatMoney(value) {
    const num = Number(value) || 0;

    // Convert to thousands
    let thousands = num;// / 1000;

    // Remove .0
    if (Number.isInteger(thousands)) {
        return thousands.toLocaleString("en-IN");
    }

    return thousands.toLocaleString("en-IN", { minimumFractionDigits: 1, maximumFractionDigits: 1 });
}

function safeNumber(v) { return Number(v) || 0; }

/* ------------------ Transform API ------------------ */
function transformApiData(apiResponse) {
    const grouped = {};
    if (apiResponse && Array.isArray(apiResponse.data)) {
        apiResponse.data.forEach(item => {
            const key = item.Project || 'Unknown';
            if (!grouped[key]) {
                grouped[key] = {
                    dept: key,
                    division: item.Division || 'General',
                    yearBudget: 0,
                    ason: 0,
                    used: 0,
                    pending: 0,
                    items: []
                };
            }
            grouped[key].items.push(item);
            grouped[key].yearBudget += safeNumber(item.Approved_Budget);
            grouped[key].ason += safeNumber(item.Budget_As_On_Month);
            grouped[key].used += safeNumber(item.Used_Budget);
            grouped[key].pending += safeNumber(item.Pending_Budget);
        });
    }
    return Object.values(grouped);
}

/* ------------------ Fill Select ------------------ */
function fillSelect(id, set) {
    const select = document.getElementById(id);
    if (!select) return;
    const title = select.options[0] ? select.options[0].text : '';
    select.innerHTML = `<option value="">${title}</option>`;
    [...set].filter(s=>s!==undefined && s!==null).sort().forEach(val => {
        const v = String(val);
        select.innerHTML += `<option value="${escapeHtml(v)}">${escapeHtml(v)}</option>`;
    });
}

function escapeHtml(str) {
    return String(str).replace(/&/g, "&amp;").replace(/"/g, "&quot;")
      .replace(/'/g, "&#39;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
}

/* ------------------ Populate Filters ------------------ */
function populateAllFilters() {
    const companySet = new Set();
    const divisionSet = new Set();
    const departmentSet = new Set();
    const projectSet = new Set();
    const activitySet = new Set();
    const employeeSet = new Set();

    // walk original raw items (budgetData groups -> items)
    budgetData.forEach(group => {
        group.items.forEach(item => {
            companySet.add(item.Company || '-');
            divisionSet.add(item.Division || '-');
            departmentSet.add(item.Project || '-');
            projectSet.add(item.Order_No || '-');
            activitySet.add(item.Proj_Activity_Name || '-');
            employeeSet.add(item.Employee || '-');
        });
    });

    fillSelect('companyFilter', companySet);
    fillSelect('divisionFilter', divisionSet);
    fillSelect('departmentFilter', departmentSet);
    fillSelect('projectFilter', projectSet);
    fillSelect('activityFilter', activitySet);
    fillSelect('employeeFilter', employeeSet);
}

/* ------------------ Update Dependent Dropdowns ------------------ */
function updateDependentDropdowns(items) {
    const divisions = new Set();
    const departments = new Set();
    const projects = new Set();
    const activities = new Set();
    const employees = new Set();

    items.forEach(i => {
        divisions.add(i.Division || '-');
        departments.add(i.Project || '-');
        projects.add(i.Order_No || '-');
        activities.add(i.Proj_Activity_Name || '-');
        employees.add(i.Employee || '-');
    });

    // Only replace dependent selects (keep company as-is)
    fillSelect('divisionFilter', divisions);
    fillSelect('departmentFilter', departments);
    fillSelect('projectFilter', projects);
    fillSelect('activityFilter', activities);
    fillSelect('employeeFilter', employees);
}

/* ------------------ Apply Filters ------------------ */
function applyFilters() {
    const fCompany = (document.getElementById('companyFilter') || {}).value;
    const fDivision = (document.getElementById('divisionFilter') || {}).value;
    const fDept = (document.getElementById('departmentFilter') || {}).value;
    const fProject = (document.getElementById('projectFilter') || {}).value;
    const fActivity = (document.getElementById('activityFilter') || {}).value;
    const fEmployee = (document.getElementById('employeeFilter') || {}).value;

    // collect filtered raw items
    const filteredItems = [];
    budgetData.forEach(g => {
        g.items.forEach(item => {
            if (
                (!fCompany  || String(item.Company) === String(fCompany)) &&
                (!fDivision || String(item.Division) === String(fDivision)) &&
                (!fDept     || String(item.Project) === String(fDept)) &&
                (!fProject  || String(item.Order_No) === String(fProject)) &&
                (!fActivity || String(item.Proj_Activity_Name) === String(fActivity)) &&
                (!fEmployee || String(item.Employee) === String(fEmployee))
            ) {
                filteredItems.push(item);
            }
        });
    });

    // update cascaded dropdown options based on currently filtered items
    updateDependentDropdowns(filteredItems);

    // regroup into currentData (group by Project)
    const map = {};
    filteredItems.forEach(item => {
        const key = item.Project || 'Unknown';
        if (!map[key]) {
            map[key] = { dept: key, division: item.Division || '-', yearBudget: 0, ason: 0, used: 0, pending: 0, items: [] };
        }
        map[key].items.push(item);
        map[key].yearBudget += safeNumber(item.Approved_Budget);
        map[key].ason += safeNumber(item.Budget_As_On_Month);
        map[key].used += safeNumber(item.Used_Budget);
        map[key].pending += safeNumber(item.Pending_Budget);
    });

    currentData = Object.values(map);

    populateGrid();
    updateSummary();
    initCharts();
}

/* ------------------ Reset Filters ------------------ */
function resetFilters() {
    ['companyFilter','divisionFilter','departmentFilter','projectFilter','activityFilter','employeeFilter']
        .forEach(id => {
            const el = document.getElementById(id);
            if (el) el.value = '';
        });

    // restore
    currentData = JSON.parse(JSON.stringify(budgetData));
    populateAllFilters();
    populateGrid();
    updateSummary();
    initCharts();
}

/* ------------------ Populate Table ------------------ */
function populateGrid() {
    const tbody = document.getElementById('budgetGrid');
    if (!tbody) return;

    const grouped = groupGridByOrderNo();
    const rows = [];

    grouped.forEach(item => {
        rows.push(`
            <tr>
                <td>${escapeHtml(item.company)}</td>
                <td>
                   <a href="#" class="order-link" data-order="${escapeHtml(item.order)}">
                      ${escapeHtml(item.order)}
                   </a>
                </td>
                <td>${escapeHtml(item.division)}</td>
                <td>${escapeHtml(item.project)}</td>
                <td>${escapeHtml(item.activity)}</td>
                <td>${escapeHtml(item.employee)}</td>
				<td>${formatMoney(item.fulltotal)}</td>
                <td><strong>${formatMoney(item.total)}</strong></td>
				<td>${formatMoney(item.unaptotal)}</td>
                <td>${formatMoney(item.ason)}</td>
                <td><strong>${formatMoney(item.used)}</strong></td>
                <td>${formatMoney(item.pending)}</td>
                <td></td>
            </tr>
        `);
    });

    tbody.innerHTML = rows.join("") || 
        '<tr><td colspan="11" style="text-align:center;color:#666">No records</td></tr>';
    document.querySelectorAll(".order-link").forEach(link => {
        link.addEventListener("click", function(e) {
            e.preventDefault();
            const orderNo = this.dataset.order;
            showOrderDetails(orderNo);
        });
    });
}

function showOrderDetails(orderNo) {
    $('#projecttitle').html(orderNo+' Details');
    let html = `
        <table class="table table-bordered table-sm">
            <thead>
                <tr>
                    <th>Activity</th>
					<th>Description</th>
                    <th>Employee</th>
					<th>Period</th>
                    <th>ACC Div</th>
                    <th>Total Budget</th>
                    <th>As On</th>
                    <th>Used</th>
                    <th>Pending</th>
                </tr>
            </thead>
            <tbody>
    `;

    // Collect split-up rows
    currentData.forEach(g => {
        g.items.forEach(item => {
            if (String(item.Order_No) === String(orderNo)) {
                html += `
                    <tr>
                        <td>${escapeHtml(item.Proj_Activity_Name || '-')}</td>
						<td>${escapeHtml(item.Description || '-')}</td>
                        <td>${escapeHtml(item.Employee || '-')}</td>
						<td>-</td>
                        <td>-</td>
                        <td>${formatMoney(item.Approved_Budget)}</td>
                        <td>${formatMoney(item.Budget_As_On_Month)}</td>
                        <td><a href="#" class="usedClick" data-code="${item.Company_Code}" data-order="${item.Item_ID}">${formatMoney(item.Used_Budget)}</a></td>
                        <td>${formatMoney(item.Pending_Budget)}</td>
                    </tr>
                `;
            }
        });
    });

    html += `</tbody></table><div id="ajaxDetailsContainer" class="mt-3"></div>`;

    document.getElementById("modalContent").innerHTML = html;

    // Show modal
    new bootstrap.Modal(document.getElementById('orderDetailsModal')).show();
	// 🔥 Register click event for Used column
    $(".usedClick").click(function () {
        let company = $(this).data("code");
        let order = $(this).data("order");
        loadBudgetSplitUp(company, order);
    });
}

function loadBudgetSplitUp(companyCode, orderNo) {
    $("#ajaxDetailsContainer").html('<div class="text-center p-2">Loading...</div>');

    $.ajax({
        url: "/budget_splitup",
        type: "GET",
        data: { companyid: companyCode, itemid: orderNo },
        success: function (res) {
            let rows = res.data; // array

			if (!rows || rows.length === 0) {
				$("#ajaxDetailsContainer").html('<div class="text-danger p-2">No records found.</div>');
				return;
			}

			let html = `<h6 class="fw-bold text-primary mt-3 mb-3">
                    <i class="bi bi-cash-stack me-1"></i> Used Amount Details
                </h6>
				<table class="table table-bordered table-striped table-sm mt-3">
					<thead class="bg-light">
						<tr>
							<th>Entry ID</th>
                            <th>Date</th>
							<th>Description</th>
                            <th>Supp Inv No</th>
                            <th>Supp Inv Date</th>
                            <th>Amount</th>
							<th>Approved</th>							
							<th>Approved1</th>
							
						</tr>
					</thead>
					<tbody>
			`;

			rows.forEach(r => {
				html += `
					<tr>
						<td>${r.Eno}</td>
						<td>${r.EDate ? moment(r.EDate).format("DD-MMM-YYYY") : '-'}</td>
						<td>${r.Description || '-'}</td>
                        <td>${r.supp_inv_no || r.Eno}</td>
						<td>${r.supp_inv_date ? moment(r.supp_inv_date).format("DD-MMM-YYYY") : '-'}</td>
                        <td>${r.Amount ? Number(r.Amount).toLocaleString() : '-'}</td>
						<td>${r.Approved || '-'}</td>
						<td>${r.Approved1 || '-'}</td>
						
					</tr>
				`;
			});

			html += `</tbody></table>`;

			$("#ajaxDetailsContainer").html(html);   // 🔥 insert below main table
        },
        error: function () {
            $("#ajaxDetailsContainer").html(`<div class="text-danger p-2">Something went wrong</div>`);
        }
    });
}


/* ------------------ Update Summary ------------------ */
function updateSummary() {
    const totals = { total:0, ason:0, used:0, pending:0 };
    currentData.forEach(g => {
        totals.total += safeNumber(g.yearBudget);
        totals.ason += safeNumber(g.ason);
        totals.used += safeNumber(g.used);
        totals.pending += safeNumber(g.pending);
    });

    const container = document.getElementById('quickSummary');
    if (!container) return;
    container.innerHTML = `
        <div class="kpi-row"><span class="kpi-label">Total Budget (FY)</span><span class="kpi-value">${formatMoney(totals.total)}</span></div>
        <div class="kpi-row"><span class="kpi-label">As On Month</span><span class="kpi-value">${formatMoney(totals.ason)}</span></div>
        <div class="kpi-row"><span class="kpi-label">Used Budget</span><span class="kpi-value">${formatMoney(totals.used)}</span></div>
        <div class="kpi-row"><span class="kpi-label">Pending Budget</span><span class="kpi-value">${formatMoney(totals.pending)}</span></div>
    `;
}

/* ------------------ Charts ------------------ */
function initCharts() {
    const barCanvas = document.getElementById('asonBarChart');
    const donutCanvas = document.getElementById('totalDoughnutChart');
    if (!barCanvas || !donutCanvas) return;

    // destroy previous
    if (asonChart) { try{ asonChart.destroy(); }catch(e){} }
    if (doughnutChart) { try{ doughnutChart.destroy(); }catch(e){} }

    // Build arrays aggregated by department/project (currentData already grouped)
    const labels = currentData.map(g => (g.dept || '-').substring(0,12));
    const asonValues = currentData.map(g => safeNumber(g.ason));
    const usedValues = currentData.map(g => safeNumber(g.used));
    const pendingValues = currentData.map(g => safeNumber(g.pending));

    // Guard: if no labels -> show placeholder to avoid Chart error
    const barLabels = labels.length ? labels : ['No Data'];
    const barAson = labels.length ? asonValues : [0];
    const barUsed = labels.length ? usedValues : [0];
    const barPending = labels.length ? pendingValues : [0];

    // Bar chart: AsOn vs Used vs Pending (department-wise)
    asonChart = new Chart(barCanvas.getContext('2d'), {
        type: 'bar',
        data: {
            labels: barLabels,
            datasets: [
                { label: 'As On Month', data: barAson, backgroundColor: '#667eea', borderRadius: 4 },
                { label: 'Used Budget', data: barUsed, backgroundColor: '#ff6b6b', borderRadius: 4 },
                { label: 'Pending Budget', data: barPending, backgroundColor: '#ffd93d', borderRadius: 4 }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'top', labels:{font:{size:10}} },
                tooltip: {
                    callbacks: {
                        label: function(ctx) {
                            const val = safeNumber(ctx.parsed.y);
                            return ctx.dataset.label + ': ₹' + (val/100000).toFixed(1) + 'L';
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: { callback: v => '₹' + (v/100000).toFixed(1) + 'L', font: {size:9} }
                },
                x: { ticks: { font: { size:9 } } }
            }
        }
    });

    // Doughnut: Used vs Remaining (across currentData)
    const totals = currentData.reduce((s,g) => {
        s.budget += safeNumber(g.yearBudget);
        s.used += safeNumber(g.used);
        return s;
    }, { budget: 0, used: 0 });

    let used = totals.used;
    let remaining = Math.max(0, totals.budget - totals.used);

    // Guard for no-data: Chart.js accepts zeros, but keep labels correct
    const donutData = (totals.budget === 0 && totals.used === 0) ? [0,0] : [used, remaining];
    const donutLabels = ['Used','Remaining'];

    doughnutChart = new Chart(donutCanvas.getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: donutLabels,
            datasets: [{
                data: donutData,
                backgroundColor: ['#ff6b6b', '#4ecdc4'],
                borderColor: '#fff',
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'bottom', labels: { font: { size: 10 } } },
                tooltip: {
                    callbacks: {
                        label: function(ctx) {
                            const val = safeNumber(ctx.parsed);
                            return ctx.label + ': ₹' + (val/100000).toFixed(1) + 'L';
                        }
                    }
                }
            }
        }
    });

}

/* ------------------ DataTable ------------------ */
function initDataTable() {
 if (gridTable) {
        gridTable.destroy();
    }
    try {
       gridTable = new DataTable('#budgetgrid', {
            destroy: true,
            paging: false,
            searching: false,
            ordering: true,
			autoWidth: false,
			layout: {
				topStart: null,
				topEnd: null
			},
			buttons: [
				{
					extend: 'excel',
					title: 'Budget_Details'
				}
			]
		});

		document.getElementById("excelBtn").addEventListener("click", () => {
    if (gridTable) {
        gridTable.button(0).trigger();   // DataTables v2 correct API
    }
});
    } catch(e) {
        console.warn('DataTable init failed', e);
    }
}

function groupGridByOrderNo() {
    const map = {};

    currentData.forEach(group => {
        group.items.forEach(item => {
            const key = item.Order_No || "-";
            if (!map[key]) {
                map[key] = {
                    order: key,
					companyid: item.Company_Code || 1,
                    company: item.Company || "-",
                    division: item.Division || "-",
                    project: item.Project || "-",
                    activity: item.Proj_Activity_Name || "-",
                    employee: item.Employee || "-",
					fulltotal:0,
                    total: 0,
					unaptotal:0,
                    ason: 0,
                    used: 0,
                    pending: 0
                };
            }
            map[key].fulltotal += safeNumber(item.Full_Budget);
			map[key].total += safeNumber(item.Approved_Budget);
			map[key].unaptotal += safeNumber(item.UnApproved_Budget);
            map[key].ason += safeNumber(item.Budget_As_On_Month);
            map[key].used += safeNumber(item.Used_Budget);
            map[key].pending += safeNumber(item.Pending_Budget);
        });
    });

    return Object.values(map);
}

/* ------------------ Fetch + DOM Ready ------------------ */
document.addEventListener('DOMContentLoaded', function() {

    // Attach change listeners for filters
    ['companyFilter','divisionFilter','departmentFilter','projectFilter','activityFilter','employeeFilter']
        .forEach(id => {
            const el = document.getElementById(id);
            if (el) el.addEventListener('change', applyFilters);
        });

    function getCurrentDateFormatted() {
        const today = new Date();
        const d = String(today.getDate()).padStart(2,'0');
        const m = ["Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"][today.getMonth()];
        const y = today.getFullYear();
        return `${d}-${m}-${y}`;
    }

    // fetch API
    fetch('budgetdetails?curdate=' + getCurrentDateFormatted())
        .then(r => r.json())
        .then(apiData => {
            // transform and setup
            budgetData = transformApiData(apiData);
            // deep copy for currentData initially
            currentData = JSON.parse(JSON.stringify(budgetData));

            populateAllFilters();
            populateGrid();
            updateSummary();
            initCharts();
            initDataTable();
        })
        .catch(err => {
            console.error('Error fetching budget details:', err);
            // still initialize empty UI
            budgetData = [];
            currentData = [];
            populateAllFilters();
            populateGrid();
            updateSummary();
            initCharts();
            initDataTable();
        });

});
</script>


</body>
</html>
