<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Direct Labour Strength</title>

  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

  <style>
    body { background-color: #f8fafc; font-family: 'Inter', sans-serif; }
    .card { background: white; border-radius: 0.75rem; box-shadow: 0 1px 3px rgba(0,0,0,0.05); padding: 1.25rem; }
    .chart-container { position: relative; height: 380px; width: 100%; }
    .chart-container > canvas { width: 100% !important; height: 100% !important; display: block; }
    .crumb { cursor: pointer; color: #2563eb; font-weight: 500; }
    .crumb:hover { text-decoration: underline; }
    .table-row:hover { background-color: #f9fafb; }
    .inactive-card { background-color: #ddd8d8ff; filter: grayscale(50%); }
    .spinner {
      display: inline-block;
      width: 1.5rem;
      height: 1.5rem;
      border: 3px solid rgba(255, 255, 255, 0.3);
      border-top-color: #fff;
      border-radius: 50%;
      animation: spin 1s ease-in-out infinite;
    }
    @keyframes spin {
      to { transform: rotate(360deg); }
    }
  </style>
</head>
<body class="min-h-screen flex flex-col">

  @php
    $empl_no = Session::get('empl_id');
    $empl_name = Session::get('emp_name');
  @endphp

  @include('LifeAtGainup.GUTProduction.header')

  <div class="px-6 py-3 flex justify-between items-center text-sm">
    <div>
      <a href="/production_gut_dashboard/{{$portal_type}}" class="text-blue-700 hover:underline">Production Dashboard</a> > 
      <span class="font-bold">Direct Labour Strength</span>
    </div>
    <div>
     
    </div>
  </div>

  <!-- KPI Cards -->
  <div class="px-6 grid grid-cols-5 gap-4 mb-6">
    <div class="card"><div class="text-sm text-gray-600">Total Strength</div><div class="text-3xl font-bold" id="strengthVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Present</div><div class="text-3xl font-bold text-blue-600" id="presentVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Allocated</div><div class="text-3xl font-bold text-green-600" id="allocatedVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Unallocated</div><div class="text-3xl font-bold text-orange-600" id="unallocatedVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Absenteeism</div><div class="text-3xl font-bold text-red-600" id="absenteeismVal">0%</div></div>
    
  </div>

  <!-- Charts -->
  <div class="px-6 grid grid-cols-2 gap-6 mb-6">
    <div class="card">
      <div class="flex justify-between items-center mb-3">
        <div class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
         Labour On Roll vs Unallocated
        </div>
        <div id="barCrumb" class="text-xs font-medium text-gray-600">
          <span class="crumb" data-level="0">All Units</span>
        </div>
      </div>
      <div class="chart-container"><canvas id="barChart"></canvas></div>
    </div>

    <div class="card inactive-card ">
      <div class="font-semibold mb-3 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Recruitment Trend
      </div>
      <div class="chart-container"><canvas id="lineChart"></canvas></div>
    </div>
  </div>

  <!-- Summary Table WITH RECRUITMENT & ATTRITION -->
  <div class="px-6 mb-10">
    <div class="card">
      <div class="font-semibold mb-4 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Summary Table
      </div>
      <div class="overflow-x-auto">
        <table class="w-full text-gray-700" id="summaryTable">
          <thead class="text-xs text-gray-500 uppercase tracking-wider border-b">
            <tr>
              <th class="text-left py-3 pl-6">Name</th>
              <th class="text-center py-3">On Roll</th>
              <th class="text-center py-3">Present</th>
              <th class="text-center py-3">Allocated</th>
              <th class="text-center py-3">Unallocated</th>
              <th class="text-center py-3">Absent</th>
              <th class="text-center py-3 text-green-600 font-bold">Recruitment</th>
              <th class="text-center py-3 text-red-600 font-bold">Attrition</th>
            </tr>
          </thead>
          <tbody class="divide-y text-sm"></tbody>
        </table>
      </div>
    </div>
  </div>

  <script>
    let rawData = null;
    let drillPath = [];
    let barChart = null;
    let portaltype = '{{$portal_type}}';

    function processData(resp) {
      if (!resp?.units?.length) return null;

      const units = {};
      const total = { onroll:0, present:0, allocated:0, unallocated:0, absent:0, recruitment:0, attrition:0 };

      resp.units.forEach(u => {
        const unitName = u.Unit || "Unknown";
        const t = u.Totals || {};

        units[unitName] = {
          totals: {
            onroll: +t.Total_Onroll || 0,
            present: +t.Total_Present || 0,
            allocated: +t.Total_Allocated || 0,
            unallocated: +t.Total_Unallocated || 0,
            absent: +t.Total_Absent || 0,
            recruitment: +t.Total_Recruitment || 0,
            attrition: +t.Total_Attrition || 0
          },
          depts: {},
          lines: {}
        };

        // Global total
        total.onroll += units[unitName].totals.onroll;
        total.present += units[unitName].totals.present;
        total.allocated += units[unitName].totals.allocated;
        total.unallocated += units[unitName].totals.unallocated;
        total.absent += units[unitName].totals.absent;
        total.recruitment += units[unitName].totals.recruitment;
        total.attrition += units[unitName].totals.attrition;

        // Department aggregation (use Emp_Cnt as Allocated and Diff as Unallocated from API)
        (u.data || []).forEach(item => {
          const dept = item.Department || "Others";
          const line = item.Line_No || "No Line";

          if (!units[unitName].depts[dept]) {
            units[unitName].depts[dept] = { 
              allocated: 0, 
              unallocated: 0, 
              onroll: 0, 
              present: 0,
              absent: 0,
              recruitment: 0, 
              attrition: 0 
            };
          }
          if (!units[unitName].lines[`${dept}|${line}`]) {
            units[unitName].lines[`${dept}|${line}`] = { 
              name: line, 
              allocated: 0, 
              unallocated: 0,
              present: 0,
              absent: 0
            };
          }

          var empCnt = +item.Emp_Cnt || 0;   // treated as Allocated
          var diff   = +item.Diff || 0;       // treated as Unallocated (can be negative)

          units[unitName].depts[dept].allocated   += empCnt;
          units[unitName].depts[dept].unallocated += diff;
          units[unitName].depts[dept].onroll      += +item.Onroll || 0;
          units[unitName].depts[dept].present     += +item.Present || 0;
          units[unitName].depts[dept].absent      += +item.Absent || 0;
          units[unitName].depts[dept].recruitment += +item.Recruitment || 0;
          units[unitName].depts[dept].attrition   += +item.Attrition || 0;

          units[unitName].lines[`${dept}|${line}`].allocated   += empCnt;
          units[unitName].lines[`${dept}|${line}`].unallocated += diff;
          units[unitName].lines[`${dept}|${line}`].present     += +item.Present || 0;
          units[unitName].lines[`${dept}|${line}`].absent      += +item.Absent || 0;
          units[unitName].lines[`${dept}|${line}`].recruitment = (units[unitName].lines[`${dept}|${line}`].recruitment || 0) + (+item.Recruitment || 0);
          units[unitName].lines[`${dept}|${line}`].attrition   = (units[unitName].lines[`${dept}|${line}`].attrition || 0) + (+item.Attrition || 0);
        });
      });

      return { units, total };
    }

    function fetchData(portaltype = "",from, to) {
      $("#loadSpinner").show();
      $.get('/gut_directlabour_details', {
        fromdate: from,
        enddate: to,
        portaltype: portaltype,
        empid: "{{ Session::get('empl_id') }}"
      })
      .done(res => {
        if (res?.units) {
          rawData = processData(res);
          drillPath = [];
          updateAll();
        }
      })
      .fail(error => {
        console.error('Error fetching data:', error);
      })
      .always(() => {
        $("#loadSpinner").hide();
      });
    }

    function updateAll() {
      updateKPIs();
      updateBarChart();
      updateTable();
      updateCrumb();
    }

    function updateKPIs() {
      let kpi = rawData?.total || { onroll:0, present:0, allocated:0, unallocated:0, absent:0, recruitment:0, attrition:0 };

      if (drillPath.length === 1) {
        kpi = rawData.units[drillPath[0]]?.totals || kpi;
      } else if (drillPath.length === 2) {
        const [unit, dept] = drillPath;
        const d = rawData.units[unit]?.depts[dept];
        if (d) {
          kpi = { ...kpi, allocated: d.allocated, unallocated: d.unallocated, onroll: d.onroll };
        }
      }

      $('#strengthVal').text(kpi.onroll.toLocaleString());
      $('#allocatedVal').text(kpi.allocated.toLocaleString());
      $('#unallocatedVal').text(kpi.unallocated.toLocaleString());
      $('#presentVal').text(kpi.present.toLocaleString());
      $('#absenteeismVal').text(kpi.onroll > 0 ? ((kpi.absent / kpi.onroll) * 100).toFixed(1) + '%' : '0%');
    }

    function updateBarChart() {
      if (!barChart || !rawData) return;

      let labels = [], allocated = [], unalloc = [];

      if (drillPath.length === 0) {
        // Top level - show units
        labels = Object.keys(rawData.units);
        labels.forEach(unit => {
          const data = rawData.units[unit].totals;
          allocated.push(data.onroll - data.unallocated);
          unalloc.push(data.unallocated);
        });
      } else if (drillPath.length === 1) {
        // Unit level - show departments
        const unit = drillPath[0];
        const depts = rawData.units[unit]?.depts || {};
        labels = Object.keys(depts);
        labels.forEach(dept => {
          const data = depts[dept];
          allocated.push(data.onroll - data.unallocated);
          unalloc.push(data.unallocated);
        });
      } else if (drillPath.length === 2) {
        // Department level - show lines
        const [unit, dept] = drillPath;
        const lines = Object.keys(rawData.units[unit].lines)
          .filter(k => k.startsWith(dept + '|'))
          .map(k => rawData.units[unit].lines[k]);
        
        lines.forEach(line => {
          labels.push(line.name === "No Line" ? "Unassigned" : line.name);
          allocated.push(line.allocated);
          unalloc.push(line.unallocated);
        });
      }
      
      // Update chart data
      barChart.data.labels = labels;
      barChart.data.datasets[0].data = allocated;
      barChart.data.datasets[1].data = unalloc;
      barChart.update();
    }

    function updateTable() {
      const tbody = $('#summaryTable tbody').empty();
      if (!rawData) return;

      if (drillPath.length === 0) {
        // All Units Level
        Object.entries(rawData.units).forEach(([u, d]) => {
          const t = d.totals;
          tbody.append(`
            <tr class="table-row">
              <td class="py-3 pl-6 font-medium">${u}</td>
              <td class="text-center">${t.onroll}</td>
              <td class="text-center">${t.present}</td>
              <td class="text-center text-green-600 font-bold">${t.allocated}</td>
              <td class="text-center text-orange-600 font-bold">${t.unallocated}</td>
              <td class="text-center text-red-600">${t.absent}</td>
              <td class="text-center text-green-600 font-bold">${t.recruitment}</td>
              <td class="text-center text-red-600 font-bold">${t.attrition}</td>
            </tr>
          `);
        });
      } else if (drillPath.length === 1) {
        // Unit → Department Level
        const unit = drillPath[0];
        Object.entries(rawData.units[unit].depts).forEach(([dept, d]) => {
          tbody.append(`
            <tr class="table-row">
              <td class="py-3 pl-6">${dept}</td>
              <td class="text-center">${d.onroll}</td>
              <td class="text-center text-blue-600">${d.present}</td>
              <td class="text-center text-green-600">${d.allocated}</td>
              <td class="text-center text-orange-600">${d.unallocated}</td>
              <td class="text-center text-red-600">${d.absent}</td>
              <td class="text-center text-green-600 font-semibold">${d.recruitment || 0}</td>
              <td class="text-center text-red-600 font-semibold">${d.attrition || 0}</td>
            </tr>
          `);
        });
      } else if (drillPath.length === 2) {
        // Department → Line Level (no recruitment/attrition here)
        const [unit, dept] = drillPath;
        Object.keys(rawData.units[unit].lines)
          .filter(k => k.startsWith(dept + '|'))
          .forEach(k => {
            const l = rawData.units[unit].lines[k];
            const name = l.name === "No Line" ? "Unassigned" : l.name;
            tbody.append(`
              <tr class="table-row">
                <td class="py-3 pl-6">Line ${name}</td>
                <td class="text-center">-</td>
                <td class="text-center text-blue-600">${l.present || '-'}</td>
                <td class="text-center text-green-600">${l.allocated}</td>
                <td class="text-center text-orange-600">${l.unallocated}</td>
                <td class="text-center text-red-600">${l.absent || '-'}</td>
                <td class="text-center text-green-600">${l.recruitment || '0'}</td>
                <td class="text-center text-red-600">${l.attrition || '0'}</td>
              </tr>
            `);
          });
      }
    }

    function updateCrumb() {
      let html = '<span class="crumb" data-level="0">All Units</span>';
      if (drillPath[0]) html += ` → <span class="crumb" data-level="1">${drillPath[0]}</span>`;
      if (drillPath[1]) html += ` → <span class="crumb" data-level="2">${drillPath[1]}</span>`;
      $('#barCrumb').html(html);

      $('#barCrumb').off('click', '.crumb').on('click', '.crumb', function() {
        const level = parseInt($(this).data('level'));
        drillPath = drillPath.slice(0, level);
        updateAll();
      });
    }

    $(document).ready(function() {
      // Bar Chart
      barChart = new Chart(document.getElementById('barChart').getContext('2d'), {
        type: 'bar',
        data: { 
          labels: [], 
          datasets: [
            { 
              label: 'Allocated',
              backgroundColor: 'navy',
              borderColor: 'navy',
              borderWidth: 1,
              data: []
            },
            { 
              label: 'Unallocated',
              backgroundColor: 'grey',
              borderColor: 'grey',
              borderWidth: 1,
              data: []
            }
          ]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: { 
            legend: { 
              display: false
            },
            tooltip: {
              callbacks: {
                label: function(context) {
                  const allocated = context.chart.data.datasets[0].data[context.dataIndex] || 0;
                  const unallocated = context.chart.data.datasets[1].data[context.dataIndex] || 0;
                  const total = allocated + unallocated;
                  
                  return [
                    `On Roll: ${total}`,
                    `Unallocated: ${unallocated}`
                  ];
                }
              }
            }
          },
          scales: { 
            x: { 
              stacked: true,
              grid: {
                display: false
              }
            }, 
            y: { 
              beginAtZero: true,
              stacked: true,
              ticks: {
                stepSize: 50 // Adjust this value to control the step size of Y-axis
              },
              grid: {
                display: true
              }
            } 
          },
          onClick: (e, el) => {
            if (!el.length || !rawData) return;
            const idx = el[0].index;
            const labels = barChart.data.labels;

            if (drillPath.length === 0) {
              const unit = labels[idx];
              if (rawData.units[unit]?.depts && Object.keys(rawData.units[unit].depts).length > 0) {
                drillPath = [unit];
                updateAll();
              }
            } else if (drillPath.length === 1) {
              const dept = labels[idx];
              const unit = drillPath[0];
              if (Object.keys(rawData.units[unit].lines).some(k => k.startsWith(dept + '|'))) {
                drillPath = [unit, dept];
                updateAll();
              }
            }
          }
        }
      });

      // Line Chart
      new Chart(document.getElementById('lineChart').getContext('2d'), {
        type: 'line',
        data: {
          labels: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'],
          datasets: [{ data: [15,22,18,30,25,35,32,40,38,45,42,50], borderColor: '#2563eb', backgroundColor: 'rgba(37,99,235,0.1)', fill: true, tension: 0.4 }]
        },
        options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
      });

      // DateRangePicker + URL Support + Title Update
      const urlParams = new URLSearchParams(window.location.search);
      const fromParam = urlParams.get('fromdate');
      const endParam = urlParams.get('enddate');
      const initialStart = (fromParam && moment(fromParam).isValid()) ? moment(fromParam) : moment();
      const initialEnd = (endParam && moment(endParam).isValid()) ? moment(endParam) : moment();

      $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: initialStart,
        endDate: initialEnd,
        maxDate: moment(),
        ranges: {
          'Today': [moment(), moment()],
          'Last Week': [moment().subtract(6, 'days'), moment()],
          'This Month': [moment().startOf('month'), moment().endOf('month')]
        },
        locale: { format: 'DD-MM-YY' }
      }, function(start, end) {
        const startFmt = start.format('YYYY-MM-DD');
        const endFmt = end.format('YYYY-MM-DD');
        const display = start.format('DD-MM-YY') + ' - ' + end.format('DD-MM-YY');

        $('#dateRangePicker').val(display);
        $('.dashboard_title').html(`
          Direct Labour Strength for - 
          <span style="color:#000;background:#FFFF00;padding:2px 8px;border-radius:10px;font-weight:bold;"> {{$portal_type}}</span>
          <span class="ml-4 text-white font-medium">${display}</span>
        `);

        const gutLink = document.getElementById('DashLink') || document.getElementById('gutDashLink');
        if (gutLink) gutLink.href = `/production_gut_dashboard/{{$portal_type}}?fromdate=${startFmt}&enddate=${endFmt}`;

        fetchData(portaltype,startFmt, endFmt);
      });

      // Initial Load
      const initStart = initialStart.format('YYYY-MM-DD');
      const initEnd = initialEnd.format('YYYY-MM-DD');
      const initDisplay = initialStart.format('DD-MM-YY') + ' - ' + initialEnd.format('DD-MM-YY');

      $('#dateRangePicker').val(initDisplay);
      $('.dashboard_title').html(`
        Direct Labour Strength for - 
        <span style="color:#000;background:#FFFF00;padding:2px 8px;border-radius:10px;font-weight:bold;"> {{$portal_type}}</span>
        <span class="ml-4 text-white font-medium">${initDisplay}</span>
      `);

      fetchData(portaltype,initStart, initEnd);

      // Historical Button
      $('#HistoricalButton').on('click', function () {
        const drp = $('#dateRangePicker').data('daterangepicker');
        const from = drp.startDate.format('YYYY-MM-DD');
        const to = drp.endDate.format('YYYY-MM-DD');
        window.location.href = `/gut_history_dashboard/{{$portal_type}}?fromdate=${from}&enddate=${to}&type=Labour`;
      });
      $('#HistoricalButton').hide();
    });
  </script>
</body>
</html>