<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>WIP Analysis</title>

  <!-- Tailwind CDN -->
  <script src="https://cdn.tailwindcss.com"></script>

  <!-- jQuery -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.4/jquery.min.js"></script>

  <!-- Datepicker -->
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

  <!-- Chart.js -->
  <script src="https://cdn.jsdelivr.net/npm/chart.js@4.3.0/dist/chart.umd.min.js"></script>

  <!-- Font Awesome CSS -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

  <!-- DataTables -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css" />
  <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
  <script>
    tailwind.config = {
      theme: {
        extend: {
          colors: {
            'background': 'rgb(245, 246, 247)',
            'foreground': 'rgb(31, 41, 55)',
            'card': 'rgb(255, 255, 255)',
            'card-foreground': 'rgb(31, 41, 55)',
            'primary': 'rgb(15, 61, 126)',
            'primary-foreground': 'rgb(248, 250, 252)',
            'primary-100': 'rgb(219, 234, 254)',
            'primary-50': 'rgb(239, 246, 255)',
            'primary-25': 'rgb(248, 250, 252)',
            'secondary': 'rgb(241, 245, 249)',
            'muted': 'rgb(241, 245, 249)',
            'muted-foreground': 'rgb(100, 116, 139)',
            'border': 'rgb(226, 232, 240)',
            'input': 'rgb(226, 232, 240)',
            'ring': 'rgb(15, 61, 126)',
          },
          fontFamily: {
            'body': ['Inter', 'system-ui', 'sans-serif'],
          }
        }
      }
    }
  </script>
  <style>
    :root {
      --navy: #0f3b66;
      --muted: #6b7280;
      --green: #16a34a;
      --red: #ef4444;
      --card-bg: #ffffff;
    }
    body { font-family: Inter, ui-sans-serif, system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial; }
    .kpi-value { font-weight: 700; font-size: 1.45rem; letter-spacing: -0.2px; }
    .kpi-sub { font-size: 0.85rem; color: var(--muted); }
    .small-pill { display: inline-block; padding: .25rem .5rem; border-radius: 999px; background: #f3f4f6; font-weight: 600; color: #111827; font-size: .75rem; }
    canvas { -moz-user-select: none; -webkit-user-select: none; -ms-user-select: none; }
    .card { background: var(--card-bg); border-radius: 10px; box-shadow: 0 2px 8px rgba(15,59,102,0.06); border: 1px solid rgba(15,59,102,0.04); }
    .sparkline { width: 110px; height: 36px; }
    .row-spark { width: 72px; height: 19px; }
	#dateRangePicker { display: none; }
    .inactive-card { background-color: #ddd8d8ff; filter: grayscale(50%); }
    .spinner {
      display: inline-block;
      width: 1.5rem;
      height: 1.5rem;
      border: 3px solid rgba(255, 255, 255, 0.3);
      border-top-color: #fff;
      border-radius: 50%;
      animation: spin 1s ease-in-out infinite;
    }
    @keyframes spin {
      to { transform: rotate(360deg); }
    }
    table.dataTable tbody th, table.dataTable tbody td {
      padding: 4px 5px;
    }
    /* Compact DataTables controls */
    .dataTables_wrapper .dataTables_length label { font-size: 12px; color:#6b7280; }
    .dataTables_wrapper .dataTables_length select { font-size: 12px; padding: 2px 6px; height: 26px; line-height: 22px; border:1px solid #e5e7eb; border-radius:6px; width:auto; min-width:56px; }
    .dataTables_wrapper .dataTables_filter input { font-size: 12px; padding: 6px 10px; height: 28px; line-height: 22px; border:1px solid #D3D3D3; border-radius:6px; width:220px; background-color: #ffffff; }
    .dataTables_wrapper .dataTables_info { font-size: 12px; color:#6b7280; }
    .dataTables_wrapper .dataTables_paginate { font-size: 12px; }
    .dataTables_wrapper .dataTables_paginate .paginate_button { padding: 2px 6px; margin: 0 1px; font-size: 12px; }
  </style>
</head>
<body class="bg-slate-50 text-slate-800">
        @php
        $empl_no= Session::get('empl_id');
        $empl_name = Session::get('emp_name');
        @endphp
  <!-- NAV -->
  @include('LifeAtGainup.production.header')
  
  <div class="px-6 py-3 text-sm text-gray-600">
    <a id="productionBreadcrumb" href="/production_gar_dashboard" class="text-blue-700 hover:underline">Production Dashboard</a> 
    <span> &gt; </span>
    <span class="text-gray-800 font-medium">WIP Analysis</span>
  </div>

  <main class="mx-auto px-6 space-y-6">
    <!-- TOP ROW: KPI cards -->
    <div class="grid grid-cols-4 gap-4">
      <!-- KPI 1 -->
      <div class="card p-4 cursor-pointer hover:shadow-md" id="kpi1">
        <div class="flex items-start justify-between">
          <div>
            <div class="kpi-sub">WIP Quantity</div>
            <div class="kpi-value text-2xl">0</div>
            <div class="text-sm mt-1"><span class="text-green-600 font-semibold">▲ 0%</span> vs last week</div>
          </div>
          <div class="w-36">
            <canvas id="spark1" class="sparkline"></canvas>
          </div>
        </div>
        <div class="text-xs text-slate-400 mt-3">Target: 8,500</div>
      </div>

      <!-- KPI 2 -->
      <div class="card p-4 cursor-pointer hover:shadow-md" id="kpi2">
        <div class="flex items-start justify-between">
          <div>
            <div class="kpi-sub">WIP Value</div>
            <div class="kpi-value text-2xl">0</div>
            <div class="text-sm mt-1"><span class="text-green-600 font-semibold">▲ 0%</span> vs last week</div>
          </div>
          <div class="w-36">
            <canvas id="spark2" class="sparkline"></canvas>
          </div>
        </div>
        <div class="text-xs text-slate-400 mt-3">Target: 100,000</div>
      </div>

      <!-- KPI 3 -->
      <div class="card p-4 cursor-pointer hover:shadow-md" id="kpi3">
        <div class="flex items-start justify-between">
          <div>
            <div class="kpi-sub">Avg. WIP Age</div>
            <div class="kpi-value text-2xl">0 days</div>
            <div class="text-sm mt-1"><span class="text-green-600 font-semibold">▲ 0%</span> vs last week</div>
          </div>
          <div class="w-36">
            <canvas id="spark3" class="sparkline"></canvas>
          </div>
        </div>
        <div class="text-xs text-slate-400 mt-3">Target: 4 days</div>
      </div>

      <!-- KPI 4 -->
      <div class="card p-4 cursor-pointer hover:shadow-md inactive-card" id="kpi4">
        <div class="flex items-start justify-between">
          <div>
            <div class="kpi-sub">Critical WIP Quantity</div>
            <div class="kpi-value text-2xl">0</div>
            <div class="text-sm mt-1"><span class="text-red-600 font-semibold">▼ 0%</span> vs last week</div>
          </div>
          <div class="w-36">
            <canvas id="spark4" class="sparkline"></canvas>
          </div>
        </div>
        <div class="text-xs text-slate-400 mt-3">Target: 0</div>
      </div>
    </div>

    <!-- Second row: Overall Trend and Buyers -->
    <div class="grid grid-cols-12 gap-6">
      <div class="col-span-8 card p-4">
        <div class="flex items-center justify-between mb-3">
          <div>
            <div class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Overall Trend</div>
          </div>
          <div class="hidden text-xs text-slate-500">Showing last 7 days</div>
        </div>
        <canvas id="overallTrend" class="w-full" height="300"></canvas>
      </div>

      <div class="col-span-4 card p-4">
        <div class="flex items-center justify-between mb-2">
          <div class="text-sm font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Buyer</div>
          <button class="hidden text-xs px-2 py-1 rounded border">All Buyer</button>
        </div>
        <div id="buyersCrumb" class="text-sm text-slate-600">All Buyer</div>
        <div id="buyersContainer"  style="max-height: 300px;">
            <canvas id="buyersHorizontal" class="w-full cursor-pointer"></canvas>
        </div>
      </div>
    </div>

    <!-- Third row: Unit-wise and Stage-wise -->
    <div class="grid grid-cols-12 gap-6">
      <div class="col-span-6 card p-4">
        <div id="unitTitle" class="font-semibold mb-2 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Unit-wise WIP Quantity</div>
        <div id="unitCrumb" class="text-sm text-slate-600 mb-2">All Units</div>
        <canvas id="unitWise" class="w-full cursor-pointer" height="250"></canvas>
      </div>

      <div class="col-span-6 card p-4">
        <div class="flex items-center justify-between">
          <div>
            <div id="stageTitle" class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Stage-wise WIP</div>
          </div>
          <div class="hidden text-xs text-slate-500">All Stages</div>
        </div>
        <div id="stageCrumb" class="text-sm text-slate-600">All Stages</div>
        <div class="mt-3">
          <canvas id="stageWise" class="w-full cursor-pointer" height="220"></canvas>
          <div class="mt-2 text-xs text-slate-500 flex items-center gap-3">
            <span class="inline-block w-3 h-2 bg-[#08325b] rounded-sm"></span> Pieces
          </div>
        </div>
      </div>
    </div>

    <!-- Company Table -->
    <div class="card p-4">
      <div class="flex items-center justify-between mb-4">
        <div class="flex flex-row items-center gap-2">
          <div class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Company WIP Details</div>
          <div class="text-xs text-slate-500">Detailed breakdown of all items currently in production across all stages.</div>
        </div>
        <div class="hidden flex items-center gap-3">
          <button class="px-3 py-1 border rounded text-sm">Filters</button>
          <button class="px-3 py-1 border rounded text-sm">Export</button>
        </div>
      </div>
      <div class="overflow-x-auto" id="wipTableContainer">
        <table id="wipTable" class="stripe hover" style="width:100%">
          <thead class="text-xs">
            <tr>
              <th>Stage</th>
              <th>Unit</th>
              <th>OCN / Style / Color</th>
              <th>Size</th>
              <th>Pieces</th>
              <th>Value</th>
              <th>Trend</th>
              <th>Age (Days)</th>
            </tr>
          </thead>
          <tbody class="text-xs"></tbody>
          <tfoot class="hidden">
            <tr>
              <th colspan="2">Total</th>
              <th></th>
              <th></th>
              <th id="totalPieces"></th>
              <th id="totalValue"></th>
              <th></th>
              <th></th>
            </tr>
          </tfoot>
        </table>
      </div>
    </div>
  </main>

  <!-- SCRIPTS -->
  <script>

      // Icons now use Font Awesome; no initialization required
    let selectedRangeLabel = 'Today';
	let selectedfromdate = null;
	let selectedenddate = null;
    let apiData = null;
    let allItems = [];
    let rawTrendData = [];
    let filters = { unit: null, stage: null, buyer: null, item: null, ageRange: null };
    let drillPath = ['root'];
    let stageDrillPath = ['root'];
    let unitDrillPath = ['root'];
    let currentMetric = 'qty'; // default to quantity

    function toIntSafe(v) {
      const n = parseInt(v);
      return Number.isNaN(n) ? 0 : n;
    }
    function toFloatSafe(v) {
      const n = parseFloat(v);
      return Number.isNaN(n) ? 0 : n;
    }

    const sparkCommonOpts = {
      type: 'line',
      options: {
        plugins: { legend: { display: false } },
        elements: { point: { radius: 0 } },
        responsive: true,
        maintainAspectRatio: false,
        scales: { x: { display: false }, y: { display: false } },
      }
    };

    const createSpark = (id, data, color) => {
      const canvas = document.getElementById(id);
      const existing = Chart.getChart(canvas);
      if (existing) existing.destroy();
      const ctx = canvas.getContext('2d');
      // Build a soft vertical gradient from the line color
      const hexToRgb = (hex) => {
        const clean = hex.replace('#','');
        const bigint = parseInt(clean.length === 3 ? clean.split('').map(c=>c+c).join('') : clean, 16);
        return { r: (bigint >> 16) & 255, g: (bigint >> 8) & 255, b: bigint & 255 };
      };
      const { r, g, b } = hexToRgb(color);
      const grad = ctx.createLinearGradient(0, 0, 0, canvas.height || 40);
      grad.addColorStop(0, `rgba(${r},${g},${b},0.18)`);
      grad.addColorStop(1, `rgba(${r},${g},${b},0.03)`);
      return new Chart(ctx, {
        ...sparkCommonOpts,
        data: {
          labels: data.map((_, i) => i),
          datasets: [{ data, borderColor: color, backgroundColor: grad, tension: 0.35, fill: true, borderWidth: 2 }]
        }
      });
    };

    const buyersCanvas = document.getElementById('buyersHorizontal');
    const buyersContainer = document.getElementById('buyersContainer');

    const buyersData = [
    "Buyer 1", "Buyer 2", "Buyer 3", "Buyer 4", "Buyer 5",
    "Buyer 6", "Buyer 7", "Buyer 8", "Buyer 9", "Buyer 10",
    "Buyer 11", "Buyer 12", "Buyer 13", "Buyer 14", "Buyer 15"
    ]; // example labels

    // Dynamically adjust chart height based on number of bars
    const barHeight = 20; // height per bar (adjust as needed)
    const chartHeight = buyersData.length * barHeight;
    buyersCanvas.height = chartHeight;

    // Create chart
    const buyersChart = new Chart(buyersCanvas.getContext('2d'), {
    type: 'bar',
    data: {
        labels: buyersData.map((buyer) => buyer.slice(0, 5)),
        datasets: [{
        data: buyersData.map(() => Math.floor(Math.random() * 1000)),
        backgroundColor: '#143e7a',
        borderRadius: 6,
        barThickness: 12
        }]
    },
    options: {
        indexAxis: 'y',
        plugins: { legend: { display: false } },
        scales: {
        x: {
            grid: { color: '#e5e7eb' },
            border: { color: '#e5e7eb' },
            ticks: {
            color: '#64748b',
            callback: (v) => v.toLocaleString()
            }
        },
        y: {
            ticks: {
            color: '#6b7280',
            fontSize: 12,
            font: { weight: 600 , size: 9}
            },
            grid: { display: false }
        }
        },
        responsive: false,
        maintainAspectRatio: false,
        layout: { padding: { right: 12 } }
    }
    });



    const oCanvas = document.getElementById('overallTrend');
    const otx = oCanvas.getContext('2d');
    const grad = otx.createLinearGradient(0, 0, 0, 200);
    grad.addColorStop(0, 'rgba(14,78,162,0.18)');
    grad.addColorStop(1, 'rgba(14,78,162,0.02)');
    const overallChart = new Chart(otx, {
      type: 'line',
      data: {
        labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
        datasets: [{ 
          label: 'WIP', 
          data: [], 
          borderColor: '#0e4ea2', 
          tension: 0.4, 
          borderWidth: 2, 
          fill: true, 
          backgroundColor: grad, 
          pointRadius: 4,
          pointHoverRadius: 6,
          pointBackgroundColor: '#fff',
          pointBorderColor: '#0e4ea2',
          pointBorderWidth: 2
        }]
      },
      options: {
        plugins: { 
          legend: { display: false },
          tooltip: {
            backgroundColor: 'rgba(30, 41, 59, 0.95)',
            titleColor: '#fff',
            bodyColor: '#e2e8f0',
            titleFont: { weight: 'bold', size: 12 },
            bodyFont: { size: 12 },
            padding: 12,
            displayColors: false,
            borderColor: 'rgba(255, 255, 255, 0.1)',
            borderWidth: 1,
            cornerRadius: 8,
            callbacks: {
              label: function(context) {
                return `WIP: ${context.parsed.y.toLocaleString()}`;
              },
              title: function(context) {
                return context[0].label;
              }
            }
          }
        },
        interaction: {
          intersect: false,
          mode: 'index'
        },
        scales: { 
          x: { 
            grid: { display: false }, 
            ticks: { 
              color: '#6b7280',
              padding: 8
            } 
          }, 
          y: { 
            ticks: { 
              color: '#6b7280',
              padding: 8,
              callback: function(value) {
                return value.toLocaleString();
              }
            }, 
            grid: { 
              color: 'rgba(15,59,102,0.06)',
              drawBorder: false
            },
            border: {
              display: false
            }
          } 
        },
        maintainAspectRatio: false,
        responsive: false
      }
    });

    const unitCanvas = document.getElementById('unitWise');
    const unitCrumbEl = $('#unitCrumb');
    const unitChart = new Chart(unitCanvas.getContext('2d'), {
      type: 'bar',
      data: {
        labels: [],
        datasets: [{ label: 'WIP Qty', data: [], backgroundColor: [], borderRadius: 8, barPercentage: 0.7 }]
      },
      options: {
        plugins: { legend: { display: false } },
        scales: { x: { ticks: { color: '#475569' }, grid: { display: false } }, y: { ticks: { color: '#475569' }, grid: { color: 'rgba(15,59,102,0.06)' } } },
        maintainAspectRatio: false,
        responsive: false
      }
    });

    const stageCanvas = document.getElementById('stageWise');
    const stageChart = new Chart(stageCanvas.getContext('2d'), {
      type: 'bar',
      data: {
        labels: [],
        datasets: [
          { label: 'Pieces', data: [], backgroundColor: '#08325b', borderRadius: 6 }
        ]
      },
      options: {
        plugins: { legend: { display: false } },
        scales: { x: { stacked: false, ticks: { color: '#475569' } }, y: { stacked: false, ticks: { color: '#475569' }, grid: { color: 'rgba(15,59,102,0.06)' } } },
        maintainAspectRatio: false,
        responsive: false
      }
    });

    function renderRowSparklines() {
      $('.row-spark').each(function (i, el) {
        const existing = Chart.getChart(el);
        if (existing) existing.destroy();
        const valuesAttr = $(el).data('values');
        let dataArr = valuesAttr.split(',').map(s => parseFloat(s));
        const ctx = el.getContext('2d');
        new Chart(ctx, {
          type: 'line',
          data: { labels: dataArr.map((_, i) => i), datasets: [{ data: dataArr, borderColor: '#10b981', fill: false, tension: 0.35, borderWidth: 1.6 }] },
          options: {
            plugins: { legend: { display: false } },
            elements: { point: { radius: 0 } },
            scales: { x: { display: false }, y: { display: false } },
            maintainAspectRatio: false,
            responsive: false
          }
        });
      });
    }

    const dataTable = $('#wipTable').DataTable({
      dom: 'lfrtip',
      pageLength: 10,
      lengthChange: true,
      lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
      info: true,
      language: { lengthMenu: 'Show _MENU_ rows', search: 'Search:', searchPlaceholder: '' },
      columnDefs: [
        { targets: 6, orderable: false },
        { targets: 0, width: '8%' },
        { targets: 7, width: '8%' }
      ]
    });
    // Make header sticky when page length > 10
    dataTable.on('length.dt', function(e, settings, len) {
      const container = document.getElementById('wipTableContainer');
      const thead = document.querySelector('#wipTable thead');
      if (!container || !thead) return;
      if (len > 10) {
        container.style.maxHeight = '520px';
        container.style.overflowY = 'auto';
        thead.style.position = 'sticky';
        thead.style.top = '0';
        thead.style.zIndex = '10';
        thead.style.background = '#ffffff';
      } else {
        container.style.maxHeight = '';
        container.style.overflowY = '';
        thead.style.position = '';
        thead.style.top = '';
        thead.style.zIndex = '';
        thead.style.background = '';
      }
    });
    $('.dataTables_filter input').addClass('px-3 py-2 border rounded').css('width', '220px');

    function buildTableRows(rows) {
      return rows.map(r => {
        const ocnLastFour = r.order ? r.order.slice(-4) : "N/A";
        const buyername = r.buyer || "N/A";
        return [
          `<span class="small-pill">${r.stage}</span>`,
          r.unit,
          `<div class="flex items-center space-x-1">
              <strong>${buyername} ${ocnLastFour}</strong>
              <div class="text-xs text-slate-400">${r.item} / ${r.color}</div>
            </div>
            `,
          r.size,
          r.pieces,
          parseInt(r.value).toLocaleString(),
          `<canvas class="row-spark" data-values="${r.trend.join(',')}"></canvas>`,
          r.age
        ];
      });
    }

    function getFilteredItems() {
      return allItems.filter(i => {
        if (filters.unit && i.unit !== filters.unit) return false;
        if (filters.stage && i.stage !== filters.stage) return false;
        if (filters.buyer && i.buyer !== filters.buyer) return false;
        if (filters.item && i.item !== filters.item) return false;
        if (filters.ageRange) {
          const a = i.absAge;
          if (filters.ageRange === '0-1' && !(a >= 0 && a <= 1)) return false;
          if (filters.ageRange === '2-3' && !(a >= 2 && a <= 3)) return false;
          if (filters.ageRange === '4-7' && !(a >= 4 && a <= 7)) return false;
          if (filters.ageRange === '8-14' && !(a >= 8 && a <= 14)) return false;
          if (filters.ageRange === '15-30' && !(a >= 15 && a <= 30)) return false;
          if (filters.ageRange === '>30' && a <= 30) return false;
        }
        return true;
      });
    }

    function calculateMetricValue(item) {
      if (currentMetric === 'qty') return item.pieces;
      if (currentMetric === 'value') return item.value;
      if (currentMetric === 'age') return item.absAge;
      if (currentMetric === 'critical') return item.age <= -14 ? item.pieces : 0;
      if (currentMetric === 'inward') return item.inward;
      if (currentMetric === 'outward') return item.outward;
      return 0;
    }

    function calculateCategoryValue(items, metric) {
      if (items.length === 0) return 0;
      if (metric === 'age') return items.reduce((sum, i) => sum + i.absAge, 0) / items.length;
      return items.reduce((sum, i) => sum + calculateMetricValue(i), 0);
    }

    function updateTrendChart() {
      if (!rawTrendData || rawTrendData.length === 0) return;
      
      // Filter trend data based on active filters
      let filteredTrend = [...rawTrendData];
      
      // Apply unit filter
      if (filters.unit) {
        filteredTrend = filteredTrend.filter(item => item.Unit === filters.unit);
      }
      
      // Apply buyer filter
      if (filters.buyer) {
        filteredTrend = filteredTrend.filter(item => item.Buyer === filters.buyer);
      }
      
      // Apply stage type filter
      if (filters.stage) {
        filteredTrend = filteredTrend.filter(item => item.Type === filters.stage);
      }
      
      // Group filtered trend data by date and calculate total stock quantity per date
      const trendByDate = {};
      filteredTrend.forEach(item => {
        const date = item.Edate;
        const qty = parseInt(item.Stock_Qty) || 0;
        if (!trendByDate[date]) {
          trendByDate[date] = 0;
        }
        trendByDate[date] += qty;
      });
      
      // Sort dates and get the last 7 days of data
      const sortedDates = Object.keys(trendByDate).sort();
      const last7Dates = sortedDates.slice(-7);
      const trendValues = last7Dates.map(date => trendByDate[date]);
      
      // Update the chart with filtered trend data
      if (overallChart) {
        overallChart.data.labels = last7Dates.map(date => new Date(date).toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
        overallChart.data.datasets[0].data = trendValues;
        overallChart.update();
      }
    }

    function updateDashboard() {
      const filtered = getFilteredItems();
      if (filtered.length === 0) return;
      updateTrendChart();

      let totalQty = 0, totalValue = 0, totalAbsAge = 0, criticalQty = 0;
      filtered.forEach(i => {
        totalQty += i.pieces;
        totalValue += i.value;
        totalAbsAge += i.absAge;
        if (i.age <= -14) criticalQty += i.pieces;
      });
      const rawAvgAge = filtered.length > 0 ? (totalAbsAge / filtered.length) : 0;
      const avgAgeNum = Number.isFinite(rawAvgAge) ? rawAvgAge : 0;
      const avgAge = avgAgeNum.toFixed(1);

      $('#kpi1 .kpi-value').text(totalQty.toLocaleString());
      $('#kpi2 .kpi-value').text(totalValue.toLocaleString());
      $('#kpi3 .kpi-value').text(`${avgAge} days`);
      $('#kpi4 .kpi-value').text(criticalQty.toLocaleString());

      let totalMetric, sparkColor, target;
      if (currentMetric === 'qty') {
        totalMetric = totalQty;
        sparkColor = '#0f62a8';
        target = 8500;
      } else if (currentMetric === 'value') {
        totalMetric = totalValue;
        sparkColor = '#059669';
        target = 100000;
      } else if (currentMetric === 'age') {
        totalMetric = avgAgeNum;
        sparkColor = '#059669';
        target = 4;
      } else if (currentMetric === 'critical') {
        totalMetric = criticalQty;
        sparkColor = '#ef4444';
        target = 0;
      } else if (currentMetric === 'inward') {
        totalMetric = filtered.reduce((sum, i) => sum + i.inward, 0);
        sparkColor = '#08325b';
        target = 8500;
      } else if (currentMetric === 'outward') {
        totalMetric = filtered.reduce((sum, i) => sum + i.outward, 0);
        sparkColor = '#2b67a3';
        target = 8500;
      }

      createSpark('spark1', [totalQty * 0.8, totalQty * 0.9, totalQty, totalQty * 1.05, totalQty * 1.1], '#0f62a8');
      createSpark('spark2', [totalValue * 0.8, totalValue * 0.9, totalValue, totalValue * 1.05, totalValue * 1.1], '#059669');
      createSpark('spark3', [avgAgeNum * 0.8, avgAgeNum * 0.9, avgAgeNum, avgAgeNum * 1.05, avgAgeNum * 1.1], '#059669');
      createSpark('spark4', [criticalQty * 1.1, criticalQty * 1.05, criticalQty, criticalQty * 0.95, criticalQty * 0.9], '#ef4444');

      const tableRows = filtered.map(i => ({
        stage: i.stage,
        unit: i.unit,
        order: i.order,
        buyer: i.buyer,
        item: i.item,
        color: i.color,
        size: i.size,
        pieces: i.pieces,
        value: i.value,
        trend: [calculateMetricValue(i) * 0.8, calculateMetricValue(i) * 0.9, calculateMetricValue(i), calculateMetricValue(i) * 1.05, calculateMetricValue(i) * 1.1],
        age: i.age
      }));
      dataTable.clear();
      dataTable.rows.add(buildTableRows(tableRows)).draw();
      renderRowSparklines();
      
      // Calculate and display totals in footer
      const tableTotalPieces = tableRows.reduce((sum, row) => sum + row.pieces, 0);
      const tableTotalValue = tableRows.reduce((sum, row) => sum + row.value, 0);
      
      // Format values in lakhs (divide by 100000 and format to 2 decimal places)
      const piecesInLakhs = (tableTotalPieces / 100000).toFixed(2);
      const valueInLakhs = (tableTotalValue / 100000).toFixed(2);
      
      $('#totalPieces').text(`${piecesInLakhs}L`);
      $('#totalValue').text(`${valueInLakhs}L`);

      // Update titles
      $('#unitTitle').text(`Unit-wise ${currentMetric.toUpperCase()}`);
      $('#stageTitle').text(`Stage-wise ${currentMetric.toUpperCase()}`);

      // Buyers chart
      let groupField = 'buyer';
      if (drillPath.length > 1) groupField = 'item';
      const groups = {};
      filtered.forEach(i => {
        const key = i[groupField];
        if (!groups[key]) groups[key] = [];
        groups[key].push(i);
      });
      const sortedGroups = Object.keys(groups).sort((a, b) => calculateCategoryValue(groups[b], currentMetric) - calculateCategoryValue(groups[a], currentMetric));
      buyersChart.data.labels = sortedGroups;
      buyersChart.data.datasets[0].data = sortedGroups.map(g => calculateCategoryValue(groups[g], currentMetric));
      // Dynamic font size based on bar count
      const barCount = sortedGroups.length;
      const fontSize = barCount > 10 ? 9 : 12;
      buyersChart.options.scales.y.ticks.font.size = fontSize;
      buyersChart.options.scales.x.max = (Math.max(...buyersChart.data.datasets[0].data) || 100) * 1.1;
      buyersChart.options.scales.x.ticks.callback = (v) => currentMetric === 'age' ? v.toFixed(1) : v.toLocaleString();
      buyersChart.update();

      // Unit chart (respect current buyer/item drill filters)
      const unitGroups = {};
      filtered.forEach(i => {
        if (!unitGroups[i.unit]) unitGroups[i.unit] = [];
        unitGroups[i.unit].push(i);
      });
      const sortedUnits = Object.keys(unitGroups).sort((a, b) => calculateCategoryValue(unitGroups[b], currentMetric) - calculateCategoryValue(unitGroups[a], currentMetric));
      unitChart.data.labels = sortedUnits;
      unitChart.data.datasets[0].data = sortedUnits.map(u => calculateCategoryValue(unitGroups[u], currentMetric));
      unitChart.data.datasets[0].backgroundColor = sortedUnits.map(u => u === filters.unit ? '#1d4ed8' : '#083b68');
      unitChart.options.scales.y.ticks.callback = (v) => currentMetric === 'age' ? v.toFixed(1) : v.toLocaleString();
      unitChart.update();

      // Stage chart
      const stageGroups = {};
      filtered.forEach(i => {
        if (!stageGroups[i.stage]) stageGroups[i.stage] = [];
        stageGroups[i.stage].push(i);
      });

      if (filters.stage || filters.ageRange) {
        const agingRanges = ['0-1', '2-3', '4-7', '8-14', '15-30', '>30'];
        const agingGroups = {};
        filtered.forEach(i => {
          const a = i.absAge;
          const r = a <= 1 ? '0-1' : a <= 3 ? '2-3' : a <= 7 ? '4-7' : a <= 14 ? '8-14' : a <= 30 ? '15-30' : '>30';
          if (!agingGroups[r]) agingGroups[r] = [];
          agingGroups[r].push(i);
        });
        stageChart.data.labels = agingRanges;
        stageChart.data.datasets = [{
          label: 'Pieces',
          data: agingRanges.map(r => (agingGroups[r] || []).reduce((sum, i) => sum + i.pieces, 0)),
          backgroundColor: agingRanges.map(r => r === filters.ageRange ? '#1d4ed8' : '#94a3b8'),
          borderRadius: 6
        }];
        stageChart.options.scales.x.stacked = false;
        stageChart.options.scales.y.stacked = false;
      } else {
        // Normalize stage names and map to desired order
        const orderMap = {
          'cut': 0,
          'sm_stock': 1, 'smstock': 1, 'sm-stock': 1, 'sm stock': 1, 's-mar': 1, 's-mark': 1,
          'sew': 2,
          'grey_store': 3, 'greystore': 3, 'grey-store': 3, 'grey store': 3, 'g-store': 3, 'g store': 3
        };
        const orderIndex = (s) => {
          const key = String(s).trim().toLowerCase().replace(/\s+/g, '_');
          return (orderMap[key] ?? 999);
        };
        const sortedStages = Object.keys(stageGroups).sort((a, b) => {
          const oa = orderIndex(a);
          const ob = orderIndex(b);
          if (oa !== ob) return oa - ob; // primary: custom order
          // secondary: keep original value-based sort if both are unknown or equal position

          const totalB = stageGroups[b].reduce((sum, i) => sum + i.pieces, 0);
          const totalA = stageGroups[a].reduce((sum, i) => sum + i.pieces, 0);
          return totalB - totalA;
        });
        stageChart.data.labels = sortedStages;
        stageChart.data.datasets = [
          { label: 'Pieces', data: sortedStages.map(s => filtered.filter(i => i.stage === s).reduce((sum, i) => sum + i.pieces, 0)), backgroundColor: '#08325b', borderRadius: 6 }
        ];
        stageChart.options.scales.x.stacked = false;
        stageChart.options.scales.y.stacked = false;
      }
      stageChart.options.scales.y.ticks.callback = (v) => currentMetric === 'age' ? v.toFixed(1) : v.toLocaleString();
      stageChart.update();

      // Overall trend - Data is now handled in processApiData
    }

    function processApiData(data) {
      allItems = [];
      data.units.forEach(unit => {
        unit.types.forEach(type => {
          type.buyers.forEach(buyer => {
            buyer.data.forEach(item => {
              allItems.push({
                unit: unit.Unit,
                stage: type.Type,
                buyer: buyer.Buyer,
                order: item.Order_No,
                item: item.Item,
                color: item.Color,
                size: item.Size,
                pieces: toIntSafe(item.Pieces),
                value: toFloatSafe(item.Value),
                age: toIntSafe(item.No_Of_Days),
                absAge: Math.abs(toIntSafe(item.No_Of_Days)),
                inward: toIntSafe(item.Inward),
                outward: toIntSafe(item.Outward)
              });
            });
          });
        });
      });
      
      // Store raw trend data for filtering
      rawTrendData = data.trend || [];
      updateTrendChart();
      
      // Apply deep link filters if any, otherwise render default view
      try { applyDeepLinkFilterIfAny(); } catch (e) { updateDashboard(); }
    }

    function applyDeepLinkFilterIfAny() {
      const urlParams = new URLSearchParams(window.location.search);
      const unit = (urlParams.get('unit') || '').trim();
      const buyer = (urlParams.get('buyer') || '').trim();
      const item = (urlParams.get('item') || '').trim();
      const stage = (urlParams.get('stage') || '').trim();
      // accept both 'age' and 'ageRange'
      const ageParam = (urlParams.get('ageRange') || urlParams.get('age') || '').trim();

      if (unit) {
        filters.unit = unit;
        unitDrillPath = ['root', unit];
      }
      if (buyer) {
        filters.buyer = buyer;
        drillPath = ['root', buyer];
      }
      if (item) {
        filters.item = item;
        drillPath = ['root', buyer, item];
      }
      if (stage) {
        filters.stage = stage;
        stageDrillPath = ['root', stage];
        const validAges = ['0-1','2-3','4-7','8-14','15-30','>30'];
        if (ageParam && validAges.includes(ageParam)) {
          filters.ageRange = ageParam;
          stageDrillPath = ['root', stage, ageParam];
        }
      }

      setCrumb();
      setUnitCrumb();
      setStageCrumb();
      updateDashboard();
    }

    const crumbEl = $('#buyersCrumb');
    function setCrumb() {
      const names = drillPath.map(k => k === 'root' ? 'All Buyer' : k);
      const html = names.map((n, idx) => `<span class="drill-link ${idx < names.length - 1 ? 'text-blue-600 cursor-pointer' : ''}" data-idx="${idx}">${n}</span>`).join(' <span class="text-slate-400">›</span> ');
      crumbEl.html(html);
    }

    function applyBuyerDrill() {
      filters.buyer = drillPath.length > 1 ? drillPath[1] : null;
      filters.item = drillPath.length > 2 ? drillPath[2] : null;
      updateDashboard();
    }

    function drillTo(label) {
      // Limit drilling to maximum 3 levels: All Buyer > Buyer > Item
      if (drillPath.length >= 3) {
        return; // Stop drilling at 3 levels
      }
      drillPath.push(label);
      setCrumb();
      applyBuyerDrill();
    }

    buyersCanvas.onclick = function (evt) {
      const points = buyersChart.getElementsAtEventForMode(evt, 'nearest', { intersect: true }, false);
      if (points.length) {
        const idx = points[0].index;
        const label = buyersChart.data.labels[idx];
        drillTo(label);
      }
    };

    crumbEl.on('click', '.drill-link', function () {
      const idx = parseInt($(this).data('idx'));
      if (idx === drillPath.length - 1) return;
      drillPath = drillPath.slice(0, idx + 1);
      setCrumb();
      applyBuyerDrill();
      if (idx === 0) {
        resetAllFilters();
      }
    });

    setCrumb();

    function updateUnitSelection(unitName) {
      // Toggle selection and sync drill path
      if (filters.unit === unitName) {
        filters.unit = null;
        unitDrillPath = ['root'];
      } else {
        filters.unit = unitName;
        unitDrillPath = ['root', unitName];
      }
      setUnitCrumb();
      updateDashboard();
    }

    unitCanvas.onclick = function (evt) {
      const points = unitChart.getElementsAtEventForMode(evt, 'nearest', { intersect: true }, false);
      if (points.length) {
        const idx = points[0].index;
        const unitName = unitChart.data.labels[idx];
        updateUnitSelection(unitName);
      }
    };

    function setUnitCrumb() {
      const names = unitDrillPath.map(k => k === 'root' ? 'All Units' : k);
      const html = names.map((n, idx) => `<span class="drill-link ${idx < names.length - 1 ? 'text-blue-600 cursor-pointer' : ''}" data-idx="${idx}">${n}</span>`).join(' <span class="text-slate-400">›</span> ');
      unitCrumbEl.html(html);
    }

    unitCrumbEl.on('click', '.drill-link', function () {
      const idx = parseInt($(this).data('idx'));
      if (idx === unitDrillPath.length - 1) return;
      // Clicking "All Units" should fully reset all filters, charts, KPIs, table and trends
      if (idx === 0) {
        resetAllFilters();
        return;
      }
      // Otherwise, drill out one level and keep unit filter in sync
      unitDrillPath = unitDrillPath.slice(0, idx + 1);
      filters.unit = unitDrillPath.length > 1 ? unitDrillPath[1] : null;
      setUnitCrumb();
      updateDashboard();
    });

    setUnitCrumb();

    const stageCrumbEl = $('#stageCrumb');
    function setStageCrumb() {
      const names = stageDrillPath.map(k => k === 'root' ? 'All Stages' : k);
      const html = names.map((n, idx) => `<span class="drill-link ${idx < names.length - 1 ? 'text-blue-600 cursor-pointer' : ''}" data-idx="${idx}">${n}</span>`).join(' <span class="text-slate-400">›</span> ');
      stageCrumbEl.html(html);
    }

    function applyStageDrill() {
      filters.stage = stageDrillPath.length > 1 ? stageDrillPath[1] : null;
      filters.ageRange = stageDrillPath.length > 2 ? stageDrillPath[2] : null;
      updateDashboard();
    }

    function drillToStage(stageName) {
      // Limit drilling to maximum 3 levels: All Stages > Stage > Age Range
      if (stageDrillPath.length >= 3) {
        return; // Stop drilling at 3 levels
      }
      stageDrillPath.push(stageName);
      setStageCrumb();
      applyStageDrill();
    }

    stageCanvas.onclick = function (evt) {
      const points = stageChart.getElementsAtEventForMode(evt, 'nearest', { intersect: true }, false);
      if (points.length) {
        const idx = points[0].index;
        const stageName = stageChart.data.labels[idx];
        drillToStage(stageName);
      }
    };

    stageCrumbEl.on('click', '.drill-link', function () {
      const idx = parseInt($(this).data('idx'));
      if (idx === stageDrillPath.length - 1) return;
      stageDrillPath = stageDrillPath.slice(0, idx + 1);
      setStageCrumb();
      applyStageDrill();
      if (idx === 0) {
        resetAllFilters();
      }
    });

    setStageCrumb();

    function resetAllFilters() {
      filters = { unit: null, stage: null, buyer: null, item: null, ageRange: null };
      drillPath = ['root'];
      stageDrillPath = ['root'];
      unitDrillPath = ['root'];
      setCrumb();
      setStageCrumb();
      setUnitCrumb();
      updateDashboard();
    }

    function updateMetric(metric) {
      currentMetric = metric;
      updateDashboard();
    }

    $('#kpi1').on('click', () => updateMetric('qty'));
    $('#kpi2').on('click', () => updateMetric('value'));
    $('#kpi3').on('click', () => updateMetric('age'));
    $('#kpi4').on('click', () => updateMetric('critical'));

    function fetchData(type = "Last Week", fromDate = "", toDate = "") {
      $("#loadSpinner").show();
      $.ajax({
        url: '/wipdetail',
        method: 'GET',
        data: { type: type, fromdate: fromDate, enddate: toDate, empid: "{{ Session('empl_id') }}"  },
        success: function (response) {
          $("#loadSpinner").hide();
          if (response && response.success && response.units) {
            processApiData(response);
          } else {
            console.warn("Invalid API response");
          }
        },
        error: function () {
          $("#loadSpinner").hide();
          console.error("Error fetching data");
        }
      });
    }

    // Function to get URL parameters
    function getUrlParameter(name) {
        name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
        const regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
        const results = regex.exec(location.search);
        return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
    }

    // Parse and apply dates from URL on page load
    function applyDatesFromUrl() {
        const fromDateParam = getUrlParameter('fromdate');
        const endDateParam = getUrlParameter('enddate');

        if (fromDateParam && endDateParam) {
          const initialRouteStart = moment(fromDateParam, 'YYYY-MM-DD');
          const initialRouteEnd = moment(endDateParam, 'YYYY-MM-DD');
		  selectedfromdate = initialRouteStart.format('YYYY-MM-DD');
		  selectedenddate = initialRouteEnd.format('YYYY-MM-DD');
          // Set initial breadcrumb link with initial dates
          const initialBreadcrumbUrl = `/production_gar_dashboard?fromdate=${initialRouteStart.format('YYYY-MM-DD')}&enddate=${initialRouteEnd.format('YYYY-MM-DD')}`;
          $('#productionBreadcrumb').attr('href', initialBreadcrumbUrl);
          $(`#HistoricalButton`).on('click', function () {
              window.location.href = `/history_dashboard/WIP?fromdate=${selectedfromdate}&enddate=${selectedenddate}`;
          });
        }
    }

    $(document).ready(function () {
      applyDatesFromUrl();
      
      $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: moment().subtract(6, 'days'),
        endDate: moment(),
        maxDate: moment(), // Restrict dates beyond today
        ranges: {
            'Today': [moment(), moment()],
            'Last Week': [moment().subtract(6, 'days'), moment()], // Changed from This Week
            'This Month': [moment().startOf('month'), moment().endOf('month')]
        }
      }, function (start, end, label) {
        const startFormatted = start.format('DD-MM-YY');
        const endFormatted = end.format('DD-MM-YY');
        const routeStart = start.format('YYYY-MM-DD');
        const routeEnd = end.format('YYYY-MM-DD');
        selectedRangeLabel = label;
        const isSameDay = start.isSame(end, 'day');

            const titleText = `
            WIP Analysis for - 
            <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
                GARMENTS
            </span>
            <span class="ml-4" style="color: #fff; font-weight: 500;">
                ${startFormatted} - ${endFormatted}
            </span>
            `;

            $('.dashboard_title').html(titleText); // Use .html() instead of .text()

        $('#dateRangePicker').val(`${startFormatted} - ${endFormatted}`);
        // Update breadcrumb link to carry range
        const breadcrumbUrl = `/production_gar_dashboard?fromdate=${routeStart}&enddate=${routeEnd}`;
        $('#productionBreadcrumb').attr('href', breadcrumbUrl);
        fetchData(selectedRangeLabel, startFormatted, endFormatted);
      });

      const initialStart = moment();
      const initialEnd = moment();
      const initialStartFormatted = initialStart.format('DD-MM-YY');
      const initialEndFormatted = initialEnd.format('DD-MM-YY');
      const isSameDay = initialStart.isSame(initialEnd, 'day');

      $('.dashboard_title').html(`
        WIP Analysis for - 
          <span style="color: #000; font-weight: bold; background-color: #FFFF00; padding: 2px 6px; border-radius: 10px;">
            GARMENTS
          </span>
          <span class="ml-4" style="color: #fff; font-weight: 500;">
            ${initialStartFormatted} - ${initialEndFormatted}
          </span>
      `);

      $('#dateRangePicker').val(initialStartFormatted);
      fetchData('Today', initialStartFormatted, initialEndFormatted);

	  $('#HistoricalButton').hide();


    });
  </script>
</body>
</html>