<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Direct Labour Strength</title>

  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

  <style>
    body { background-color: #f8fafc; font-family: 'Inter', sans-serif; }
    .card { background: white; border-radius: 0.75rem; box-shadow: 0 1px 3px rgba(0,0,0,0.05); padding: 1.25rem; }
    .chart-container { position: relative; height: 380px; width: 100%; }
    .chart-container > canvas { width: 100% !important; height: 100% !important; display: block; }
    .crumb { cursor: pointer; color: #2563eb; font-weight: 500; }
    .crumb:hover { text-decoration: underline; }
    .table-row:hover { background-color: #f9fafb; }
    .kpi-spark { height: 20px; width: 60px; max-width: 100%; overflow: hidden; }
  </style>
</head>
<body class="min-h-screen flex flex-col">

  @php
    $empl_no = Session::get('empl_id');
    $empl_name = Session::get('emp_name');
  @endphp

  @include('LifeAtGainup.Production.header')

  <div class="px-6 py-3 flex justify-between items-center text-sm">
    <div>
      <a href="/production_gar_dashboard" class="text-blue-700 hover:underline">Production Dashboard</a> > 
      <span class="font-bold">Direct Labour Strength</span>
    </div>
    <div>
     
    </div>
  </div>

  <!-- KPI Cards -->
  <div class="px-6 grid grid-cols-5 gap-4 mb-6">
    <div class="card">
      <div class="flex items-center justify-between">
        <div>
          <div class="text-sm text-gray-600">Total Strength</div>
          <div class="text-3xl font-bold" id="strengthVal">0</div>
        </div>
        <canvas id="sparkStrength" class="kpi-spark"></canvas>
      </div>
    </div>
    <div class="card">
      <div class="flex items-center justify-between">
        <div>
          <div class="text-sm text-gray-600">Present</div>
          <div class="text-3xl font-bold text-blue-600" id="presentVal">0</div>
        </div>
        <canvas id="sparkPresent" class="kpi-spark"></canvas>
      </div>
    </div>
    <div class="card">
      <div class="flex items-center justify-between">
        <div>
          <div class="text-sm text-gray-600">Allocated</div>
          <div class="text-3xl font-bold text-green-600" id="allocatedVal">0</div>
        </div>
        <canvas id="sparkAllocated" class="kpi-spark"></canvas>
      </div>
    </div>
    <div class="card">
      <div class="flex items-center justify-between">
        <div>
          <div class="text-sm text-gray-600">Unallocated</div>
          <div class="text-3xl font-bold text-orange-600" id="unallocatedVal">0</div>
        </div>
        <canvas id="sparkUnallocated" class="kpi-spark"></canvas>
      </div>
    </div>
    <div class="card">
      <div class="flex items-center justify-between">
        <div>
          <div class="text-sm text-gray-600">Absenteeism</div>
          <div class="text-3xl font-bold text-red-600" id="absenteeismVal">0%</div>
        </div>
        <canvas id="sparkAbsentee" class="kpi-spark"></canvas>
      </div>
    </div>
  </div>

  <!-- Charts -->
  <div class="px-6 grid grid-cols-2 gap-6 mb-6">
    <div class="card">
      <div class="flex justify-between items-center mb-3">
        <div class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
         Labour Allocated vs Unallocated
        </div>
        <div id="barCrumb" class="text-xs font-medium text-gray-600">
          <span class="crumb" data-level="0">All Units</span>
        </div>
      </div>
      <div class="chart-container"><canvas id="barChart"></canvas></div>
    </div>

    <div class="card">
      <div class="font-semibold mb-3 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Recruitment Trend
      </div>
      <div class="chart-container"><canvas id="lineChart"></canvas></div>
    </div>
  </div>

  <!-- Summary Table WITH RECRUITMENT & ATTRITION -->
  <div class="px-6 mb-10">
    <div class="card">
      <div class="font-semibold mb-4 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Summary Table
      </div>
      <div class="overflow-x-auto">
        <table class="w-full text-gray-700" id="summaryTable">
          <thead class="text-xs text-gray-500 uppercase tracking-wider border-b">
            <tr>
              <th class="text-left py-3 pl-6">Name</th>
              <th class="text-center py-3">On Roll</th>
              <th class="text-center py-3">Present</th>
              <th class="text-center py-3">Allocated</th>
              <th class="text-center py-3">Unallocated</th>
              <th class="text-center py-3">Absent</th>
              <th class="text-center py-3 text-green-600 font-bold">Recruitment</th>
              <th class="text-center py-3 text-red-600 font-bold">Attrition</th>
            </tr>
          </thead>
          <tbody class="divide-y text-sm"></tbody>
        </table>
      </div>
    </div>
  </div>

  <script>
    let rawData = null;
    let drillPath = [];
    let barChart = null;

    function initKpiSparklines(){
      var configs = [
        { id: 'sparkStrength',   data: [2,3,4,5,6,7],      color: '#22c55e' },
        { id: 'sparkPresent',    data: [1,1.5,2,2.8,3.6],  color: '#3b82f6' },
        { id: 'sparkAllocated',  data: [1,2,3,4,5,6],      color: '#22c55e' },
        { id: 'sparkUnallocated',data: [3,2.8,2.4,2.0,1.6],color: '#f97316' },
        { id: 'sparkAbsentee',   data: [1.2,1.0,1.1,1.3,1.5], color: '#ef4444' }
      ];
      configs.forEach(function(cfg){
        var c = document.getElementById(cfg.id);
        if (!c) return;
        var ctx = c.getContext('2d');
        new Chart(ctx, {
          type: 'line',
          data: {
            labels: cfg.data.map(function(_,i){ return i+1; }),
            datasets: [{
              data: cfg.data,
              borderColor: cfg.color,
              backgroundColor: cfg.color + '33',
              fill: true,
              tension: 0.5,
              pointRadius: 0
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: { legend: { display: false }, tooltip: { enabled: false } },
            scales: { x: { display: false }, y: { display: false } }
          }
        });
      });
    }

    function processData(resp) {
      if (!resp?.units?.length) return null;

      const units = {};
      const total = { onroll:0, present:0, allocated:0, unallocated:0, absent:0, recruitment:0, attrition:0 };

      resp.units.forEach(u => {
        const unitName = u.Unit || "Unknown";
        const t = u.Totals || {};

        units[unitName] = {
          totals: {
            onroll: +t.Total_Onroll || 0,
            present: +t.Total_Present || 0,
            allocated: +t.Total_Allocated || 0,
            unallocated: +t.Total_Unallocated || 0,
            absent: +t.Total_Absent || 0,
            recruitment: +t.Total_Recruitment || 0,
            attrition: +t.Total_Attrition || 0
          },
          depts: {},
          lines: {}
        };

        // Global total
        total.onroll += units[unitName].totals.onroll;
        total.present += units[unitName].totals.present;
        total.allocated += units[unitName].totals.allocated;
        total.unallocated += units[unitName].totals.unallocated;
        total.absent += units[unitName].totals.absent;
        total.recruitment += units[unitName].totals.recruitment;
        total.attrition += units[unitName].totals.attrition;

        // Department aggregation (use Emp_Cnt as Allocated and Diff as Unallocated from API)
        (u.data || []).forEach(item => {
          const dept = item.Department || "Others";
          const line = item.Line_No || "No Line";

          if (!units[unitName].depts[dept]) {
            units[unitName].depts[dept] = { allocated:0, unallocated:0, onroll:0, recruitment:0, attrition:0 };
          }
          if (!units[unitName].lines[`${dept}|${line}`]) {
            units[unitName].lines[`${dept}|${line}`] = { name: line, allocated:0, unallocated:0 };
          }

          var empCnt = +item.Emp_Cnt || 0;   // treated as Allocated
          var diff   = +item.Diff || 0;       // treated as Unallocated (can be negative)

          units[unitName].depts[dept].allocated   += empCnt;
          units[unitName].depts[dept].unallocated += diff;
          units[unitName].depts[dept].onroll      += +item.Onroll || 0;
          units[unitName].depts[dept].recruitment += +item.Recruitment || 0;
          units[unitName].depts[dept].attrition   += +item.Attrition || 0;

          units[unitName].lines[`${dept}|${line}`].allocated   += empCnt;
          units[unitName].lines[`${dept}|${line}`].unallocated += diff;
        });
      });

      return { units, total };
    }

    function fetchData(from, to) {
      $.get('/directlabour_details', {
        fromdate: from,
        enddate: to,
        empid: "{{ Session::get('empl_id') }}"
      }, res => {
        if (res?.units) {
          rawData = processData(res);
          drillPath = [];
          updateAll();
        }
      });
    }

    function updateAll() {
      updateKPIs();
      updateBarChart();
      updateTable();
      updateCrumb();
    }

    function updateKPIs() {
      let kpi = rawData?.total || { onroll:0, present:0, allocated:0, unallocated:0, absent:0, recruitment:0, attrition:0 };

      if (drillPath.length === 1) {
        kpi = rawData.units[drillPath[0]]?.totals || kpi;
      } else if (drillPath.length === 2) {
        const [unit, dept] = drillPath;
        const d = rawData.units[unit]?.depts[dept];
        if (d) {
          kpi = { ...kpi, allocated: d.allocated, unallocated: d.unallocated, onroll: d.onroll };
        }
      }

      $('#strengthVal').text(kpi.onroll.toLocaleString());
      $('#allocatedVal').text(kpi.allocated.toLocaleString());
      $('#unallocatedVal').text(kpi.unallocated.toLocaleString());
      $('#presentVal').text(kpi.present.toLocaleString());
      $('#absenteeismVal').text(kpi.onroll > 0 ? ((kpi.absent / kpi.onroll) * 100).toFixed(1) + '%' : '0%');
    }

    function updateBarChart() {
  if (!barChart || !rawData) return;

  let labels = [];
  let presentPerc = [];
  let diffPerc = [];

  if (drillPath.length === 0) {
    labels = Object.keys(rawData.units);
    labels.forEach(u => {
      const present = rawData.units[u].totals.present;
      const diff = rawData.units[u].totals.unallocated;

      const total = present + diff;

      const p = total > 0 ? (present / total * 100) : 0;
      const d = total > 0 ? (diff / total * 100) : 0;

      presentPerc.push(p.toFixed(2));
      diffPerc.push(d.toFixed(2));
    });
  }

  barChart.data.labels = labels;
  barChart.data.datasets[0].data = presentPerc;
  barChart.data.datasets[1].data = diffPerc;

  barChart.update();
}



    function updateTable() {
      const tbody = $('#summaryTable tbody').empty();
      if (!rawData) return;

      if (drillPath.length === 0) {
        // All Units Level
        Object.entries(rawData.units).forEach(([u, d]) => {
          const t = d.totals;
          tbody.append(`
            <tr class="table-row">
              <td class="py-3 pl-6 font-medium">${u}</td>
              <td class="text-center">${t.onroll}</td>
              <td class="text-center">${t.present}</td>
              <td class="text-center text-green-600 font-bold">${t.allocated}</td>
              <td class="text-center text-orange-600 font-bold">${t.unallocated}</td>
              <td class="text-center text-red-600">${t.absent}</td>
              <td class="text-center text-green-600 font-bold">${t.recruitment}</td>
              <td class="text-center text-red-600 font-bold">${t.attrition}</td>
            </tr>
          `);
        });
      } else if (drillPath.length === 1) {
        // Unit → Department Level
        const unit = drillPath[0];
        Object.entries(rawData.units[unit].depts).forEach(([dept, d]) => {
          tbody.append(`
            <tr class="table-row">
              <td class="py-3 pl-6">${dept}</td>
              <td class="text-center">${d.onroll}</td>
              <td class="text-center">-</td>
              <td class="text-center text-green-600">${d.allocated}</td>
              <td class="text-center text-orange-600">${d.unallocated}</td>
              <td class="text-center">-</td>
              <td class="text-center text-green-600 font-semibold">${d.recruitment || 0}</td>
              <td class="text-center text-red-600 font-semibold">${d.attrition || 0}</td>
            </tr>
          `);
        });
      } else if (drillPath.length === 2) {
        // Department → Line Level (no recruitment/attrition here)
        const [unit, dept] = drillPath;
        Object.keys(rawData.units[unit].lines)
          .filter(k => k.startsWith(dept + '|'))
          .forEach(k => {
            const l = rawData.units[unit].lines[k];
            const name = l.name === "No Line" ? "Unassigned" : l.name;
            tbody.append(`
              <tr class="table-row">
                <td class="py-3 pl-6">Line ${name}</td>
                <td class="text-center">-</td>
                <td class="text-center">-</td>
                <td class="text-center text-green-600">${l.allocated}</td>
                <td class="text-center text-orange-600">${l.unallocated}</td>
                <td class="text-center">-</td>
                <td class="text-center">-</td>
                <td class="text-center">-</td>
              </tr>
            `);
          });
      }
    }

    function updateCrumb() {
      let html = '<span class="crumb" data-level="0">All Units</span>';
      if (drillPath[0]) html += ` → <span class="crumb" data-level="1">${drillPath[0]}</span>`;
      if (drillPath[1]) html += ` → <span class="crumb" data-level="2">${drillPath[1]}</span>`;
      $('#barCrumb').html(html);

      $('#barCrumb').off('click', '.crumb').on('click', '.crumb', function() {
        const level = parseInt($(this).data('level'));
        drillPath = drillPath.slice(0, level);
        updateAll();
      });
    }

    $(document).ready(function() {
      // Render static mini trend charts in KPI cards
      initKpiSparklines();
      // Bar Chart
      barChart = new Chart(document.getElementById('barChart'), {
  type: 'bar',
  data: {
    labels: [],
    datasets: [
      {
        label: 'Present (%)',
        backgroundColor: '#0ea5e9',
        data: []
      },
      {
        label: 'Diff (%)',
        backgroundColor: '#f97316',
        data: []
      }
    ]
  },
  options: {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: { display: true }
    },
    scales: {
      x: { stacked: true },
      y: {
        stacked: true,
        min: 0,
        max: 100
      }
    }
  }
});




      // Line Chart
      new Chart(document.getElementById('lineChart').getContext('2d'), {
        type: 'line',
        data: {
          labels: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'],
          datasets: [{ data: [15,22,18,30,25,35,32,40,38,45,42,50], borderColor: '#2563eb', backgroundColor: 'rgba(37,99,235,0.1)', fill: true, tension: 0.4 }]
        },
        options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
      });

      // DateRangePicker + URL Support + Title Update
      const urlParams = new URLSearchParams(window.location.search);
      const fromParam = urlParams.get('fromdate');
      const endParam = urlParams.get('enddate');
      const initialStart = (fromParam && moment(fromParam).isValid()) ? moment(fromParam) : moment();
      const initialEnd = (endParam && moment(endParam).isValid()) ? moment(endParam) : moment();

      $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: initialStart,
        endDate: initialEnd,
        maxDate: moment(),
        ranges: {
          'Today': [moment(), moment()],
          'Last Week': [moment().subtract(6, 'days'), moment()],
          'This Month': [moment().startOf('month'), moment().endOf('month')]
        },
        locale: { format: 'DD-MM-YY' }
      }, function(start, end) {
        const startFmt = start.format('YYYY-MM-DD');
        const endFmt = end.format('YYYY-MM-DD');
        const display = start.format('DD-MM-YY') + ' - ' + end.format('DD-MM-YY');

        $('#dateRangePicker').val(display);
        $('.dashboard_title').html(`
          Direct Labour Strength for - 
          <span style="color:#000;background:#FFFF00;padding:2px 8px;border-radius:10px;font-weight:bold;">GARMENTS</span>
          <span class="ml-4 text-white font-medium">${display}</span>
        `);

        const gutLink = document.getElementById('DashLink') || document.getElementById('gutDashLink');
        if (gutLink) gutLink.href = `/production_gar_dashboard/?fromdate=${startFmt}&enddate=${endFmt}`;

        fetchData(startFmt, endFmt);
      });

      // Initial Load
      const initStart = initialStart.format('YYYY-MM-DD');
      const initEnd = initialEnd.format('YYYY-MM-DD');
      const initDisplay = initialStart.format('DD-MM-YY') + ' - ' + initialEnd.format('DD-MM-YY');

      $('#dateRangePicker').val(initDisplay);
      $('.dashboard_title').html(`
        Direct Labour Strength for - 
        <span style="color:#000;background:#FFFF00;padding:2px 8px;border-radius:10px;font-weight:bold;">GARMENTS</span>
        <span class="ml-4 text-white font-medium">${initDisplay}</span>
      `);

      fetchData(initStart, initEnd);

      // Historical Button
      $('#HistoricalButton').on('click', function () {
        const drp = $('#dateRangePicker').data('daterangepicker');
        const from = drp.startDate.format('YYYY-MM-DD');
        const to = drp.endDate.format('YYYY-MM-DD');
        window.location.href = `/history_dashboard/?fromdate=${from}&enddate=${to}&type=Labour`;
      });
    });
  </script>
</body>
</html>