<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Direct Labour Strength</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
  <style>
    body { background-color: #f8fafc; font-family: 'Inter', sans-serif; }
    .card { background: white; border-radius: 0.75rem; box-shadow: 0 1px 3px rgba(0,0,0,0.05); padding: 1.25rem; }
    .chart-container { position: relative; height: 380px; width: 100%; }
    .chart-container > canvas { width: 100% !important; height: 100% !important; display: block; }
    .crumb { cursor: pointer; color: #2563eb; font-weight: 500; }
    .crumb:hover { text-decoration: underline; }
    .table-row:hover { background-color: #f9fafb; cursor: pointer; }
    .inactive-card { background-color: #ddd8d8ff; filter: grayscale(50%); }
  </style>
</head>
<body class="min-h-screen flex flex-col">
  @php
    $empl_no = Session::get('empl_id');
    $empl_name = Session::get('emp_name');
  @endphp
  @include('LifeAtGainup.socksproduction.header')

  <div class="px-6 py-3 flex justify-between items-center text-sm">
    <div>
      <a href="/production_socks_dashboard" class="text-blue-700 hover:underline">Production Dashboard</a> >
      <span class="font-bold">Direct Labour Strength</span>
    </div>
  </div>

  <!-- KPI Cards -->
  <div class="px-6 grid grid-cols-5 gap-4 mb-6">
    <div class="card"><div class="text-sm text-gray-600">Total Strength</div><div class="text-3xl font-bold" id="strengthVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Present</div><div class="text-3xl font-bold text-blue-600" id="presentVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Allocated</div><div class="text-3xl font-bold text-green-600" id="allocatedVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Unallocated</div><div class="text-3xl font-bold text-orange-600" id="unallocatedVal">0</div></div>
    <div class="card"><div class="text-sm text-gray-600">Absenteeism</div><div class="text-3xl font-bold text-red-600" id="absenteeismVal">0%</div></div>
  </div>

  <!-- Charts -->
  <div class="px-6 grid grid-cols-2 gap-6 mb-6">
    <div class="card">
      <div class="flex justify-between items-center mb-3">
        <div class="font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
          Labour On Roll vs Unallocated
        </div>
        <div id="barCrumb" class="text-xs font-medium text-gray-600">
          <span class="crumb" data-level="0">All Units</span>
        </div>
      </div>
      <div class="chart-container"><canvas id="barChart"></canvas></div>
    </div>
    <div class="card inactive-card">
      <div class="font-semibold mb-3 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Recruitment Trend
      </div>
      <div class="chart-container"><canvas id="lineChart"></canvas></div>
    </div>
  </div>

  <!-- Summary Table -->
  <div class="px-6 mb-10">
    <div class="card">
      <div class="font-semibold mb-4 bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent">
        Summary Table
      </div>
      <div class="overflow-x-auto">
        <table class="w-full text-gray-700" id="summaryTable">
          <thead class="text-xs text-gray-500 uppercase tracking-wider border-b">
            <tr>
              <th class="text-left py-3 pl-6">Name</th>
              <th class="text-center py-3">On Roll</th>
              <th class="text-center py-3">Present</th>
              <th class="text-center py-3">Allocated</th>
              <th class="text-center py-3">Unallocated</th>
              <th class="text-center py-3">Absent</th>
              <th class="text-center py-3 text-green-600 font-bold">Recruitment</th>
              <th class="text-center py-3 text-red-600 font-bold">Attrition</th>
            </tr>
          </thead>
          <tbody class="divide-y text-sm"></tbody>
        </table>
      </div>
    </div>
  </div>

  <script>
    let rawData = null;
    let drillPath = [];
    let barChart = null;

    function processData(resp) {
      if (!resp?.units?.length) return null;

      const units = {};
      const total = { onroll:0, present:0, allocated:0, unallocated:0, absent:0, recruitment:0, attrition:0 };

      resp.units.forEach(u => {
        const unitName = u.Unit || "Unknown";
        const t = u.Totals || {};

        units[unitName] = {
          totals: {
            onroll: +t.Total_Onroll || 0,
            present: +t.Total_Present || 0,
            allocated: +t.Total_Allocated || 0,
            unallocated: +t.Total_Unallocated || 0,
            absent: +t.Total_Absboren || 0,
            recruitment: +t.Total_Recruitment || 0,
            attrition: +t.Total_Attrition || 0
          },
          depts: {},
          lines: {}
        };

        // Global total
        total.onroll += units[unitName].totals.onroll;
        total.present += units[unitName].totals.present;
        total.allocated += units[unitName].totals.allocated;
        total.unallocated += units[unitName].totals.unallocated;
        total.absent += units[unitName].totals.absent || (units[unitName].totals.onroll - units[unitName].totals.present);
        total.recruitment += units[unitName].totals.recruitment;
        total.attrition += units[unitName].totals.attrition;

        // Department & Line aggregation
        (u.data || []).forEach(item => {
          const dept = (item.Department || "Others").trim();
          const line = item.Line_No || "No Line";

          if (!units[unitName].depts[dept]) {
            units[unitName].depts[dept] = {
              onroll: 0, present: 0, allocated: 0, unallocated: 0, absent: 0,
              recruitment: 0, attrition: 0
            };
          }

          const onroll = +item.Onroll || 0;
          const present = +item.Present || 0;
          const allocated = +item.Emp_Cnt || 0;
          const unallocated = +item.Diff || 0;  // This is Present but Unallocated
          const recruitment = +item.Recruitment || 0;
          const attrition = +item.Attrition || 0;

          units[unitName].depts[dept].onroll += onroll;
          units[unitName].depts[dept].present += present;
          units[unitName].depts[dept].allocated += allocated;
          units[unitName].depts[dept].unallocated += unallocated;
          units[unitName].depts[dept].absent += (onroll - present);
          units[unitName].depts[dept].recruitment += recruitment;
          units[unitName].depts[dept].attrition += attrition;

          // Line level (for future use)
          const lineKey = `${dept}|${line}`;
          if (!units[unitName].lines[lineKey]) {
            units[unitName].lines[lineKey] = { name: line, allocated: 0, unallocated: 0 };
          }
          units[unitName].lines[lineKey].allocated += allocated;
          units[unitName].lines[lineKey].unallocated += unallocated;
        });
      });

      return { units, total };
    }

    function fetchData(from, to) {
      $.get('/socks_directlabour_details', {
        fromdate: from,
        enddate: to,
        empid: "{{ Session::get('empl_id') }}"
      }, res => {
        if (res?.units) {
          rawData = processData(res);
          drillPath = [];
          updateAll();
        }
      });
    }

    function updateAll() {
      updateKPIs();
      updateBarChart();
      updateTable();
      updateCrumb();
    }

    function updateKPIs() {
      let kpi = rawData?.total || { onroll:0, present:0, allocated:0, unallocated:0, absent:0 };
      if (drillPath.length === 1) {
        kpi = rawData.units[drillPath[0]]?.totals || kpi;
      }

      $('#strengthVal').text(kpi.onroll.toLocaleString());
      $('#presentVal').text(kpi.present.toLocaleString());
      $('#allocatedVal').text(kpi.allocated.toLocaleString());
      $('#unallocatedVal').text(kpi.unallocated.toLocaleString());
      $('#absenteeismVal').text(kpi.onroll > 0 ? ((kpi.absent / kpi.onroll) * 100).toFixed(1) + '%' : '0%');
    }

    function updateBarChart() {
      if (!barChart || !rawData) return;

      let labels = [], allocated = [], unallocated = [];

      if (drillPath.length === 0) {
        labels = Object.keys(rawData.units);
        labels.forEach(unit => {
          const d = rawData.units[unit].totals;
          allocated.push(d.allocated);
          unallocated.push(d.unallocated);
        });
      } else if (drillPath.length === 1) {
        const unit = drillPath[0];
        const depts = rawData.units[unit].depts;
        labels = Object.keys(depts);
        labels.forEach(dept => {
          const d = depts[dept];
          allocated.push(d.allocated);
          unallocated.push(d.unallocated);
        });
      }

      barChart.data.labels = labels;
      barChart.data.datasets[0].data = allocated;
      barChart.data.datasets[1].data = unallocated;
      barChart.update();
    }

    function updateTable() {
      const tbody = $('#summaryTable tbody').empty();
      if (!rawData) return;

      if (drillPath.length === 0) {
        // All Units Level
        Object.entries(rawData.units).forEach(([unit, d]) => {
          const t = d.totals;
          tbody.append(`
            <tr class="table-row" data-unit="${unit}">
              <td class="py-3 pl-6 font-medium">${unit}</td>
              <td class="text-center">${t.onroll}</td>
              <td class="text-center text-blue-600 font-semibold">${t.present}</td>
              <td class="text-center text-green-600 font-bold">${t.allocated}</td>
              <td class="text-center text-orange-600 font-bold">${t.unallocated}</td>
              <td class="text-center text-red-600">${t.absent || (t.onroll - t.present)}</td>
              <td class="text-center text-green-600 font-bold">${t.recruitment}</td>
              <td class="text-center text-red-600 font-bold">${t.attrition}</td>
            </tr>
          `);
        });
      } else if (drillPath.length === 1) {
        // Department Level
        const unit = drillPath[0];
        Object.entries(rawData.units[unit].depts).forEach(([dept, d]) => {
          const absent = d.onroll - d.present;
          tbody.append(`
            <tr class="table-row">
              <td class="py-3 pl-6 font-medium">${dept}</td>
              <td class="text-center">${d.onroll}</td>
              <td class="text-center text-blue-600 font-semibold">${d.present}</td>
              <td class="text-center text-green-600 font-bold">${d.allocated}</td>
              <td class="text-center text-orange-600 font-bold">${d.unallocated}</td>
              <td class="text-center text-red-600">${absent}</td>
              <td class="text-center text-green-600 font-bold">${d.recruitment}</td>
              <td class="text-center text-red-600 font-bold">${d.attrition}</td>
            </tr>
          `);
        });
      }
    }

    function updateCrumb() {
      let html = '<span class="crumb" data-level="0">All Units</span>';
      if (drillPath.length === 1) {
        html += ` → <span class="crumb" data-level="0">${drillPath[0]}</span>`;
      }
      $('#barCrumb').html(html);

      $('#barCrumb').off('click', '.crumb').on('click', '.crumb', function() {
        const level = parseInt($(this).data('level'));
        drillPath = drillPath.slice(0, level);
        updateAll();
      });
    }

    $(document).ready(function() {
      // Bar Chart Init
      barChart = new Chart($('#barChart')[0].getContext('2d'), {
        type: 'bar',
        data: {
          labels: [],
          datasets: [
            { label: 'Allocated', backgroundColor: 'navy', data: [] },
            { label: 'Unallocated (Present)', backgroundColor: 'grey', data: [] }
          ]
        },
        options: {
          responsive: true,
          maintainAspectRatio: false,
          plugins: { legend: { display: true, position: 'bottom' } },
          scales: {
            x: { stacked: true },
            y: { stacked: true, beginAtZero: true }
          },
          onClick: (e, el) => {
            if (!el.length || drillPath.length >= 1) return;
            const idx = el[0].index;
            const unit = barChart.data.labels[idx];
            if (rawData.units[unit]?.depts && Object.keys(rawData.units[unit].depts).length > 0) {
              drillPath = [unit];
              updateAll();
            }
          }
        }
      });

      // Dummy Line Chart
      new Chart($('#lineChart')[0].getContext('2d'), {
        type: 'line',
        data: {
          labels: ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'],
          datasets: [{ label: 'Recruitment', data: [15,22,18,30,25,35,32,40,38,45,42,50], borderColor: '#2563eb', backgroundColor: 'rgba(37,99,235,0.1)', fill: true, tension: 0.4 }]
        },
        options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
      });

      // Date Range Picker
      const urlParams = new URLSearchParams(window.location.search);
      const fromParam = urlParams.get('fromdate');
      const endParam = urlParams.get('enddate');
      const initialStart = (fromParam && moment(fromParam).isValid()) ? moment(fromParam) : moment();
      const initialEnd = (endParam && moment(endParam).isValid()) ? moment(endParam) : moment();

      $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: initialStart,
        endDate: initialEnd,
        maxDate: moment(),
        ranges: {
          'Today': [moment(), moment()],
          'Last Week': [moment().subtract(6, 'days'), moment()],
          'This Month': [moment().startOf('month'), moment().endOf('month')]
        },
        locale: { format: 'DD-MM-YY' }
      }, function(start, end) {
        const startFmt = start.format('YYYY-MM-DD');
        const endFmt = end.format('YYYY-MM-DD');
        fetchData(startFmt, endFmt);
      });

      // Initial Load
      fetchData(initialStart.format('YYYY-MM-DD'), initialEnd.format('YYYY-MM-DD'));
    });
  </script>
</body>
</html>