<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Historical Data</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" />
  <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
  <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
  <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>
<script>
    tailwind.config = {
        theme: {
            extend: {
                colors: {
                    'background': 'rgb(245, 246, 247)',
                    'foreground': 'rgb(31, 41, 55)',
                    'card': 'rgb(255, 255, 255)',
                    'card-foreground': 'rgb(31, 41, 55)',
                    'primary': 'rgb(15, 61, 126)',
                    'primary-foreground': 'rgb(248, 250, 252)',
                    'primary-100': 'rgb(219, 234, 254)',
                    'primary-50': 'rgb(239, 246, 255)',
                    'primary-25': 'rgb(248, 250, 252)',
                    'secondary': 'rgb(241, 245, 249)',
                    'muted': 'rgb(241, 245, 249)',
                    'muted-foreground': 'rgb(100, 116, 139)',
                    'border': 'rgb(226, 232, 240)',
                    'input': 'rgb(226, 232, 240)',
                    'ring': 'rgb(15, 61, 126)',
                },
                fontFamily: {
                    'body': ['Inter', 'system-ui', 'sans-serif'],
                }
            }
        }
    }
</script>
<style>
    .date-picker input {
        background: rgba(248, 250, 252, 0.1);
        color: white;
    }
    .date-picker input::placeholder {
        color: rgba(248, 250, 252, 0.7);
    }
    .styled-select {
        background: rgba(248, 250, 252, 0.1);
        color: white;
    }
    .styled-select option {
        color: #1f2937;
        background: white;
    }
    .metrics-dropdown .menu.level2{display:none}
    .metrics-dropdown .menu.level1 .has-sub:hover>.menu.level2{display:block}
    .metrics-trigger{display:flex;align-items:center;gap:.25rem}
    .spinner {
        display: inline-block;
        width: 1.5rem;
        height: 1.5rem;
        border: 3px solid rgba(255, 255, 255, 0.3);
        border-top-color: #fff;
        border-radius: 50%;
        animation: spin 1s ease-in-out infinite;
    }
    @keyframes spin {
        to { transform: rotate(360deg); }
    }
    submenu-item:disabled {
        opacity: 0.5;
        cursor: not-allowed;
        pointer-events: none;
    }
    .highlight {
        background-color: rgb(219, 234, 254) !important;
        font-weight: 600;
    }

	/* Right-align all content inside table body */
	.tablewise tbody td {
	  text-align: right !important;
	}

	/* Left-align first column (Unit/Buyer) in table body */
	.tablewise tbody td:first-child {
	  text-align: left !important;
	}

	/* Right-align all content inside table footer */
	.tablewise tfoot td {
	  text-align: right !important;
	}

	/* Left-align first column in table footer */
	.tablewise tfoot td:first-child {
	  text-align: left !important;
	}

	/* Optional: keep header left-aligned */
	.tablewise thead {
	  text-align: left;
	}

</style>
</head>
<body class="bg-gray-100 font-sans">
    <nav class="bg-gradient-to-r from-blue-900 via-blue-800 to-blue-900 text-white px-4 py-1.5 shadow-md flex items-center justify-between sticky top-0 z-50 backdrop-blur-md border-b border-blue-700/40">
	  <div class="flex items-center space-x-2">
		<a href="/dashboard"
		  class="inline-flex items-center justify-center rounded-lg border border-white/20 px-2 py-1.5 bg-white/10 hover:bg-white/20 transition-all duration-200">
		  <i class="fa-solid fa-house text-white text-base"></i>
		</a>
		<h1 class="text-base font-semibold leading-tight tracking-wide dashboard_title">
		  Historical Data {{$portal_type}}
		</h1>
	  </div>

	  <div class="flex items-center space-x-2">
		<div id="loadSpinner" class="spinner hidden"></div>

		<!-- Dropdown -->
		<div class="metrics-dropdown relative">
		  <button id="metricsBtn"
			class="metrics-trigger px-3 py-1.5 rounded-md bg-white/10 text-sm border border-white/20 hover:bg-white/20 flex items-center transition-all duration-200"
			data-val="Production:Knitting" data-type="1">
			Knitting <span class="ml-1 text-blue-200 text-xs">▾</span>
		  </button>

		  <!-- Level 1 menu -->
		  <div
			class="menu level1 hidden absolute right-0 mt-1 bg-white text-gray-800 rounded-md shadow-xl border border-gray-200 min-w-[160px] z-[60]">
			<div class="menu-title px-3 py-1.5 border-b text-sm font-semibold bg-gray-50">All Metrics</div>

			<!-- Production -->
			<div class="menu-item has-sub flex items-center justify-between px-3 py-1.5 hover:bg-gray-100 cursor-default text-sm" data-key="Production">
			  <span>Production</span><span class="text-gray-400">›</span>
			  <div class="menu level2 hidden absolute left-full top-8 bg-white rounded-md shadow-xl border border-gray-200 min-w-[150px] z-[70]">
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Knitting" data-type="1">Knitting</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Linking" data-type="1">Linking</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Washing" data-type="1">Washing</button>
                <button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Boarding" data-type="1">Boarding</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Pairing" data-type="1">Pairing</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:Packing" data-type="1">Packing</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100" data-val="Production:OEE" data-type="1">OEE</button>
			  </div>
			</div>

			<!-- Line Stop -->
			<div class="menu-item has-sub flex items-center justify-between px-3 py-1.5 hover:bg-gray-100 cursor-default text-sm" data-key="Line Stop">
			  <span>Line Stop Mins</span><span class="text-gray-400">›</span>
			  <div class="menu level2 hidden absolute left-full top-10 bg-white rounded-md shadow-xl border border-gray-200 min-w-[150px] z-[70]">
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 line_stop_item" data-val="Line Stop:Minutes" data-type="1">Minutes</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 line_stop_item" data-val="Line Stop:Loss Amount" data-type="2">Loss Amount</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 line_stop_item" disabled>MTBS</button>
			  </div>
			</div>

			<!-- Cutting -->
			<div class="menu-item has-sub flex items-center justify-between px-3 py-1.5 hover:bg-gray-100 cursor-default text-sm" data-key="Cutting">
			  <span>Cutting</span><span class="text-gray-400">›</span>
			  <div class="menu level2 hidden absolute left-full top-12 bg-white rounded-md shadow-xl border border-gray-200 min-w-[150px] z-[70]">
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 cutting_item" data-val="Cutting:Total Cut" data-type="1">Total Cut</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 cutting_item" data-val="Cutting:Fabric Utilization" disabled>Fabric Utilization</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 cutting_item" data-val="Cutting:FTR %" disabled>FTR %</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 cutting_item" data-val="Cutting:Marker Efficiency" disabled>Marker Efficiency</button>
			  </div>
			</div>

			<!-- WIP -->
			<div class="menu-item has-sub flex items-center justify-between px-3 py-1.5 hover:bg-gray-100 cursor-default text-sm" data-key="WIP">
			  <span>WIP</span><span class="text-gray-400">›</span>
			  <div class="menu level2 hidden absolute left-full top-14 bg-white rounded-md shadow-xl border border-gray-200 min-w-[150px] z-[70]">
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 wip_item" data-val="WIP:WIP"  disabled>WIP</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 wip_item" data-val="WIP:WIP Value" disabled>WIP Value</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 wip_item" data-val="WIP:Avg. WIP Age" disabled>Avg. WIP Age</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 wip_item" data-val="WIP:Critical WIP" disabled>Critical WIP</button>
			  </div>
			</div>

			<!-- Labour -->
			<div class="menu-item has-sub flex items-center justify-between px-3 py-1.5 hover:bg-gray-100 cursor-default text-sm" data-key="Labour">
			  <span>Labour</span><span class="text-gray-400">›</span>
			  <div class="menu level2 hidden absolute left-full top-18 bg-white rounded-md shadow-xl border border-gray-200 min-w-[150px] z-[70]">
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 labour_item" data-val="Labour:Strength" disabled>Strength</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 labour_item" data-val="Labour:Absenteeism" disabled>Absenteeism</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 labour_item" data-val="Labour:Attrition" disabled>Attrition</button>
				<button class="submenu-item block w-full text-left px-3 py-1.5 hover:bg-gray-100 labour_item" data-val="Labour:Recruitment" disabled>Recruitment</button>
			  </div>
			</div>
		  </div>
		</div>

		<!-- Date Picker -->
		<div class="date-picker">
		  <input type="text" placeholder="Select date range"
			class="h-8 px-3 rounded-md bg-white/10 text-white border border-white/20 hover:bg-white/20 text-sm cursor-pointer transition-all duration-200 placeholder-blue-200/80 focus:outline-none focus:ring-1 focus:ring-amber-300/50"
			id="dateRangePicker">
		</div>

		<!-- Notification -->
		<button
		  class="inline-flex items-center justify-center rounded-md bg-white/10 text-white border border-white/20 hover:bg-white/20 hover:scale-105 h-8 w-8 transition-all duration-200"
		  id="notificationsBtn">
		  <i class="fa-solid fa-bell text-base text-amber-300"></i>
		</button>
	  </div>
	</nav>

  <div class="px-6 py-3 text-sm text-gray-600 main_breadcrumb">
    <a id="productionBreadcrumb" href="/production_socks_dashboard?fromdate={{$fromdate}}&enddate={{$enddate}}" class="text-blue-700 hover:underline">Production Dashboard</a> 
    <span> &gt; </span>
    <span class="text-gray-500 main_breadcrumb_text portal_type_link"> </span>
    <span> &gt; </span>
    <span class="text-gray-800 font-medium">Historical Data</span>
  </div>
    <div class="px-6 grid grid-cols-1 lg:grid-cols-2 gap-6">
      <!-- Unit Table (Always Visible) -->
      <div class="bg-white px-6 py-2 rounded-lg shadow">
          <div class="flex items-center gap-2">
              <h2 class="text-lg font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Unit-wise Performance</h2>
              <p class="text-sm text-gray-500">Month-on-month output by production unit.</p>
          </div>    
        <table class="w-full border-collapse text-sm tablewise" id="unitTable">
          <thead>
            <tr class="bg-gray-100">
              <th class="px-2 py-2 text-left">Unit</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="Apr">Apr</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="May">May</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="Jun">Jun</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="Jul">Jul</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="Aug">Aug</th>
              <th class="px-2 py-2 month-header cursor-pointer" data-type="unit" data-month="Sep">Sep</th>
            </tr>
          </thead>
          <tbody>
            <tr><td colspan="7" class="px-4 py-2 text-center">No data available</td></tr>
          </tbody>
        </table>
      </div>
      
      <!-- Right Column - Toggle between OEE Chart and Buyer Table -->
      <div>
        <!-- OEE Chart (Hidden by default, shown when OEE is selected) -->
        <div class="bg-white px-6 py-4 rounded-lg shadow mb-6 hidden" id="oeeChartContainer">
            <div class="flex items-center gap-2 mb-4">
                <h2 class="text-lg font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">OEE Trend</h2>
                <p class="text-sm text-gray-500">Monthly OEE performance by unit</p>
            </div>
            <div class="relative" style="height: 300px;">
                <canvas id="oeeTrendChart"></canvas>
            </div>
        </div>
        
        <!-- Buyer Table (Visible by default, hidden when OEE is selected) -->
        <div class="bg-white px-6 py-2 rounded-lg shadow" id="buyerTableContainer">
            <div class="flex items-center gap-2">
                <h2 class="text-lg font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Buyer-wise Performance</h2>
                <span id="buyerFilterInfo" class="text-sm font-semibold text-blue-500"></span>
                <p class="text-sm text-gray-500">Month-on-month output by buyer.</p>
            </div>    
            <table class="w-full border-collapse text-sm tablewise" id="buyerTable">
              <thead>
                <tr class="bg-gray-100">
                  <th class="px-2 py-2 text-left">Buyer</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="Apr">Apr</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="May">May</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="Jun">Jun</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="Jul">Jul</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="Aug">Aug</th>
                  <th class="px-2 py-2 month-header cursor-pointer" data-type="buyer" data-month="Sep">Sep</th>
                </tr>
              </thead>
              <tbody>
                <tr><td colspan="7" class="px-4 py-2 text-center">No data available</td></tr>
              </tbody>
            </table>
        </div>
      </div>
  </div>
  <div class="px-6 py-4">
    <div class="bg-white px-6 py-2 rounded-lg shadow">
        <div class="flex items-center gap-2">
            <h2 id="drilldownTitle" class="text-lg font-semibold bg-gradient-to-r from-blue-500 to-indigo-500 bg-clip-text text-transparent animate-gradient">Daily Performance table</h2>
            <p class="text-sm text-gray-500 ">Showing daily output values. Click a cell to navigate to that day's overview.</p>
        </div>    
      <div class="overflow-x-auto">
        <table id="drilldownTable" class="w-full border-collapse text-sm ">
          <tr><td colspan="100" class="px-4 py-2 text-center">No data available</td></tr>
        </table>
      </div>
    </div>
  </div>
  <div class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden" id="notificationOverlay">
    <div class="fixed right-0 top-0 h-full w-80 bg-white shadow-lg transform translate-x-full transition-transform duration-300" id="notificationPanel">
      <div class="p-6 border-b border-border">
        <div class="flex items-center justify-between">
          <h2 class="text-lg font-semibold">Production Alerts</h2>
          <button class="text-muted-foreground hover:text-foreground" id="closeNotifications">
            <i class="fa-solid fa-xmark h-5 w-5"></i>
          </button>
        </div>
      </div>
      <div class="p-6">
        <p class="text-muted-foreground">No new alerts at this time.</p>
      </div>
    </div>
  </div>
<script>
    let apiData = {};
    let selectedMetric = "Production:Knitting";
    let selectedType = "1";
    // Active Buyer table filter context (set when unit/month filter is applied on Buyer table)
    let currentBuyerFilterUnit = null; // e.g., 'G-I'
    let currentBuyerFilterMonth = null; // e.g., 'Jul'
    const portal_type = "{{$portal_type}}" || "Production";
    let initialLoad = true; // do not auto-restore selection on first load
    // Persist selection across visits
    const STORAGE_KEY = 'history_dash_selection_v2';
    function saveSelection(payload = {}) {
      try {
        const sel = {
          portal: portal_type || '',
          metric: selectedMetric || '',
          mode: payload.mode || '', // 'cell' | 'month'
          type: payload.type || '', // 'unit' | 'buyer'
          name: payload.name || null,
          month: payload.month || null,
          unitContext: payload.unitContext || null,
          savedAt: Date.now(),
        };
        localStorage.setItem(STORAGE_KEY, JSON.stringify(sel));
      } catch (e) { /* no-op */ }
    }
    function getStoredSelection() {
      try {
        const raw = localStorage.getItem(STORAGE_KEY);
        if (!raw) return null;
        const obj = JSON.parse(raw);
        if (!obj || obj.portal !== (portal_type || '')) return null;
        return obj;
      } catch { return null; }
    }
    const apiRoutes = {
      'Production:Knitting': '/knitproduction',
      'Production:Linking': '/linkedproduction',
      'Production:Washing': '/washproduction',
      'Production:Boarding': '/boardproduction',
      'Production:Pairing': '/parproduction',
      'Production:Packing': '/packproduction',
      'Production:OEE': '/oeehistoricaldata',
      'WIP:WIP': '/wip_historicaldata',
      'WIP:WIP Value': '/wip_value_historicaldata',
      'WIP:Avg. WIP Age': '/avg_wip_age_historicaldata',
      'WIP:Critical WIP': '/critical_wip_historicaldata',
      'Line Stop:Minutes': '/production_minis_historicaldata',
      'Line Stop:Loss Amount': '/production_minis_historicaldata',
      'Line Stop:MTBS': '/mtbs_historicaldata',
      'Cutting:Total Cut': '/production_cutting_historicaldata',
      'Cutting:Fabric Utilization': '/production_cutting_historicaldata',
      'Cutting:FTR %': '/production_cutting_historicaldata',
      'Cutting:Marker Efficiency': '/production_cutting_historicaldata',
      'Labour:Strength': '/strength_historicaldata',
      'Labour:Absenteeism': '/absenteeism_historicaldata',
      'Labour:Attrition': '/attrition_historicaldata',
      'Labour:Recruitment': '/recruitment_historicaldata'
    };
    function fetchData(type = "", fromDate = "", toDate = "", portal_type = "", metric = "Production:Knitting") {
        console.log(portal_type);
      const apiUrl = apiRoutes[metric];
      if (!apiUrl) {
        $("#loadSpinner").hide();
        console.warn(`No API route available for metric: ${metric}`);
        return;
      }
      const unitTable = $('#unitTable');
      unitTable.find('thead').html('<tr class="bg-gray-100"><th class="px-2 py-2 text-left">Unit</th></tr>');
      unitTable.find('tbody').html('<tr><td colspan="1" class="px-4 py-2 text-center">No data available</td></tr>');
      unitTable.find('tfoot').remove();
      
      const buyerTable = $('#buyerTable');
      buyerTable.find('thead').html('<tr class="bg-gray-100"><th class="px-2 py-2 text-left">Buyer</th></tr>');
      buyerTable.find('tbody').html('<tr><td colspan="1" class="px-4 py-2 text-center">No data available</td></tr>');
      buyerTable.find('tfoot').remove();
      $('#buyerFilterInfo').text('');
      // Clear active Buyer filter context on fresh data load
      currentBuyerFilterUnit = null;
      currentBuyerFilterMonth = null;
      $("#drilldownTable").html('<tr><td colspan="100" class="px-4 py-2 text-center">No data available</td></tr>');
      $("#drilldownTitle").text('Daily Performance table');
      $('.drilldown, .month-header').removeClass('highlight');
      $("#loadSpinner").show();
      $.ajax({
        url: apiUrl,
        method: 'GET',
        data: { type: type, fromdate: fromDate, enddate: toDate },
        success: function(response) {
          $("#loadSpinner").hide();
          if (response && response.success && response.data) {
            apiData = response.data;
            // If OEE data, ensure we have month_OEE calculated for each unit
            if (metric === 'Production:OEE') {
                apiData.forEach(monthData => {
                    if (monthData.units) {
                        monthData.units.forEach(unit => {
                            // Calculate average OEE for the month if not present
                            if (!unit.month_OEE && unit.shifts && unit.shifts.length > 0) {
                                const totalOEE = unit.shifts.reduce((sum, shift) => sum + (parseFloat(shift.OEE) || 0), 0);
                                unit.month_OEE = totalOEE / unit.shifts.length;
                            }
                        });
                    }
                });
            }
            updateTables(apiData, fromDate, toDate, metric);
            // Re-apply stored selection only after first load
            const stored = initialLoad ? null : getStoredSelection();
            if (stored && stored.month) {
              $('.drilldown, .month-header').removeClass('highlight');
              if (stored.type === 'unit') {
                if (stored.mode === 'cell') {
                  // Recreate buyer filter table for this unit/month
                  updateBuyerTableFilter(stored.month, stored.name);
                  const $cell = $(`.drilldown[data-type="unit"][data-name="${stored.name}"][data-month="${stored.month}"]`).first();
                  if ($cell.length) {
                    $cell.addClass('highlight');
                    updateDrilldownTable('unit', stored.name, stored.month, false, null);
                  }
            // mark first load as complete after successful render
            initialLoad = false;
                } else if (stored.mode === 'month') {
                  const $hdr = $(`.month-header[data-type="unit"][data-month="${stored.month}"]`).first();
                  if ($hdr.length) {
                    $hdr.addClass('highlight');
                    // Also refresh buyer table for the selected month (no unit filter)
                    updateBuyerTableFilter(stored.month, null);
                    updateDrilldownTable('unit', null, stored.month, true, null);
                  }
                }
              } else if (stored.type === 'buyer') {
                if (stored.mode === 'cell') {
                  // If last click was a buyer cell under a unit filter, restore that buyer table filter first
                  if (stored.unitContext) updateBuyerTableFilter(stored.month, stored.unitContext);
                  const $cell = $(`.drilldown[data-type="buyer"][data-name="${stored.name}"][data-month="${stored.month}"]`).first();
                  if ($cell.length) {
                    $cell.addClass('highlight');
                    updateDrilldownTable('buyer', stored.name, stored.month, false, stored.unitContext || null);
                  }
                } else if (stored.mode === 'month') {
                  if (stored.unitContext) updateBuyerTableFilter(stored.month, stored.unitContext);
                  const $hdr = $(`.month-header[data-type="buyer"][data-month="${stored.month}"]`).first();
                  if ($hdr.length) {
                    $hdr.addClass('highlight');
                    updateDrilldownTable('buyer', null, stored.month, true, stored.unitContext || null);
                  }
                }
              }
            }
          } else {
            console.warn("Invalid API response, resetting data");
            apiData = {};
          }
        },
        error: function() {
          $("#loadSpinner").hide();
          console.error("Error fetching data from", apiUrl);
          apiData = {};
        }
      });
    }

let oeeChart = null;

function renderOEEChart(months, unitsData, data) {
    const ctx = document.getElementById('oeeTrendChart').getContext('2d');
    
    // Destroy existing chart if it exists
    if (oeeChart) {
        oeeChart.destroy();
    }
    
    // Prepare datasets
    const datasets = [];
    const colors = [
        'rgba(75, 192, 192, 1)',
        'rgba(54, 162, 235, 1)',
        'rgba(255, 99, 132, 1)',
        'rgba(255, 159, 64, 1)',
        'rgba(153, 102, 255, 1)',
        'rgba(255, 205, 86, 1)'
    ];
    
    console.log('Rendering OEE Chart with data:', {months, unitsData, data});
    
    // If unitsData is an array of objects with unit and data properties
    if (Array.isArray(unitsData) && unitsData.length > 0 && unitsData[0].data) {
        unitsData.forEach((unitData, index) => {
            if (unitData && unitData.unit && Array.isArray(unitData.data)) {
                const color = colors[index % colors.length];
                datasets.push({
                    label: unitData.unit,
                    data: unitData.data,
                    borderColor: color,
                    backgroundColor: color.replace('1)', '0.2)'),
                    borderWidth: 2,
                    pointBackgroundColor: color,
                    pointRadius: 4,
                    pointHoverRadius: 6,
                    tension: 0.1,
                    fill: false
                });
            }
        });
    }
    
    oeeChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: months,
            datasets: datasets
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `${context.dataset.label}: ${context.raw ? context.raw.toLocaleString() : 'N/A'}`;
                        }
                    }
                },
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true,
                        pointStyle: 'circle'
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'OEE Quantity',
                        padding: {top: 10, bottom: 10}
                    },
                    ticks: {
                        callback: function(value) {
                            return value;
                        }
                    },
                    // No max limit for quantities
                },
                x: {
                    grid: {
                        display: false
                    }
                }
            },
            interaction: {
                intersect: false,
                mode: 'index'
            }
        }
    });
}

function updateTables(data, fromDate, toDate, metric) {
    const start = fromDate ? moment(fromDate, 'YYYY-MM-DD') : moment().subtract(5, 'months').startOf('month');
    const end = toDate ? moment(toDate, 'YYYY-MM-DD') : moment();
    const months = [];
    for (let m = moment(start); m.isSameOrBefore(end); m.add(1, 'month')) {
        months.push(m.format('MMM'));
    }
    
    // Show/hide appropriate containers based on metric
    const isOEE = metric === 'Production:OEE';
    if (isOEE) {
        $('#oeeChartContainer').removeClass('hidden');
        $('#buyerTableContainer').addClass('hidden');
        
        // Process OEE data for chart and table
        const units = [];
        data.forEach(monthData => {
            if (monthData.units) {
                monthData.units.forEach(unit => {
                    if (!units.includes(unit.Unit)) {
                        units.push(unit.Unit);
                    }
                });
            }
        });

        // Prepare data for chart using month_OEE from API response
        const unitsData = [];
        
        // Add individual unit data
        units.forEach(unit => {
            const unitData = {
                unit: unit,
                data: months.map(month => {
                    const monthData = data.find(d => d.Mon_Name.startsWith(month));
                    if (!monthData || !monthData.units) return null;
                    const unitMonthData = monthData.units.find(u => u.Unit === unit);
                    return unitMonthData ? (parseFloat(unitMonthData.month_OEE) || null) : null;
                })
            };
            // Only add unit if it has data
            if (unitData.data.some(val => val !== null)) {
                unitsData.push(unitData);
            }
        });

        // Render the chart
        renderOEEChart(months, unitsData, data);
        
        // Process unit table for OEE
        let unitTableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">Unit</th>`;
        months.forEach(month => {
            unitTableHTML += `<th class="px-2 py-2 month-header cursor-pointer text-right" data-type="unit" data-month="${month}">${month}</th>`;
        });
        unitTableHTML += `<th class="px-2 py-2 text-right">Avg OEE</th></tr></thead><tbody>`;
        
        if (units.length === 0) {
            unitTableHTML += `<tr><td colspan="${months.length + 2}" class="px-4 py-2 text-center">No data available</td></tr>`;
        } else {
            units.forEach(unit => {
                unitTableHTML += `<tr class="border-t"><td class="px-2 py-2">${unit}</td>`;
                let totalOEE = 0;
                let monthCount = 0;
                
                months.forEach(month => {
                    const monthData = data.find(d => d.Mon_Name.startsWith(month));
                    let monthOEE = null;
                    
                    if (monthData && monthData.units) {
                        const unitData = monthData.units.find(u => u.Unit === unit);
                        if (unitData && unitData.shifts && unitData.shifts.length > 0) {
                            const total = unitData.shifts.reduce((sum, shift) => sum + (parseFloat(shift.OEE) || 0), 0);
                            monthOEE = Math.round(total / unitData.shifts.length);
                            totalOEE += monthOEE;
                            monthCount++;
                        }
                    }
                    
                    unitTableHTML += `<td class="px-2 py-2 drilldown cursor-pointer text-right" 
                        data-type="unit" data-name="${unit}" data-month="${month}">
                        ${monthOEE !== null ? Math.floor(monthOEE) : '-'}
                    </td>`;
                });
                
                const avgOEE = monthCount > 0 ? Math.floor(totalOEE / monthCount) : '-';
                unitTableHTML += `<td class="px-2 py-2 text-right font-semibold">${avgOEE}</td>`;
                unitTableHTML += `</tr>`;
            });
        }
        
        unitTableHTML += `</tbody>`;
        $('#unitTable').html(unitTableHTML);
        
        // Reattach event listeners for the new elements
        setTimeout(() => {
            $('.month-header').off('click').on('click', function() {
                const month = $(this).data('month');
                const type = $(this).data('type');
                updateDrilldownTable(type, null, month, true);
            });

            $('.drilldown').off('click').on('click', function() {
                const type = $(this).data('type');
                const name = $(this).data('name');
                const month = $(this).data('month');
                updateDrilldownTable(type, name, month);
            });
        }, 0);
        
        return;
    } else {
        // Original code for non-OEE metrics
        $('#oeeChartContainer').addClass('hidden');
        $('#buyerTableContainer').removeClass('hidden');
    }

    // Unit table
    let unitMonthlyTotals = {};
    let unitTotals = {};
    months.forEach(month => unitMonthlyTotals[month] = 0);
    const units = [...new Set(data.map(item => item.units.map(u => u.Unit)).flat())];
    units.forEach(unit => unitTotals[unit] = 0);

    data.forEach(monthData => {
        const month = monthData.Mon_Name.slice(0, 3);
        if (months.includes(month)) {
            monthData.units.forEach(unit => {
                const total = unit.buyers.reduce((sum, b) => sum + (b.buyer_total_Knit_Qty || 0), 0);
                unitMonthlyTotals[month] += total;
                unitTotals[unit.Unit] += total;
            });
        }
    });

    let unitTableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">Unit</th>`;
    months.forEach(month => {
        unitTableHTML += `<th class="px-2 py-2 month-header cursor-pointer text-right" data-type="unit" data-month="${month}">${month}</th>`;
    });
    unitTableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
    if (units.length === 0) {
        unitTableHTML += `<tr><td colspan="${months.length + 2}" class="px-4 py-2 text-center">No data available</td></tr>`;
    } else {
        units.forEach(unit => {
            unitTableHTML += `<tr class="border-t"><td class="px-2 py-2">${unit}</td>`;
            months.forEach(month => {
                const monthData = data.find(d => d.Mon_Name.startsWith(month));
                const unitData = monthData?.units.find(u => u.Unit === unit);
                const total = unitData ? unitData.buyers.reduce((sum, b) => sum + (b.buyer_total_Knit_Qty || 0), 0) : 0;
                unitTableHTML += `<td class="px-2 py-2 drilldown cursor-pointer text-left" data-type="unit" data-name="${unit}" data-month="${month}">${total.toLocaleString()}</td>`;
            });
            unitTableHTML += `<td class="px-2 py-2 text-left font-semibold">${unitTotals[unit].toLocaleString()}</td>`;
            unitTableHTML += `</tr>`;
        });
    }
    unitTableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
    months.forEach(month => {
        unitTableHTML += `<td class="px-2 py-2 text-left">${unitMonthlyTotals[month].toLocaleString()}</td>`;
    });
    unitTableHTML += `<td class="px-2 py-2 text-left">${Object.values(unitMonthlyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
    unitTableHTML += `</tr></tfoot>`;
    $('table').eq(0).html(unitTableHTML);

    // Buyer table
    let buyerMonthlyTotals = {};
    let buyerTotals = {};
    months.forEach(month => buyerMonthlyTotals[month] = 0);
    const buyers = [...new Set(data.map(item => item.units.map(u => u.buyers.map(b => b.Buyer)).flat()).flat())];
    buyers.forEach(buyer => buyerTotals[buyer] = 0);

    data.forEach(monthData => {
        const month = monthData.Mon_Name.slice(0, 3);
        if (months.includes(month)) {
            monthData.units.forEach(unit => {
                unit.buyers.forEach(buyer => {
                    buyerMonthlyTotals[month] += (buyer.buyer_total_Knit_Qty || 0);
                    buyerTotals[buyer.Buyer] += (buyer.buyer_total_Knit_Qty || 0);
                });
            });
        }
    });

    let buyerTableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">Buyer</th>`;
    months.forEach(month => {
        buyerTableHTML += `<th class="px-2 py-2 month-header  text-right cursor-pointer" data-type="buyer" data-month="${month}">${month}</th>`;
    });
    buyerTableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
    if (buyers.length === 0) {
        buyerTableHTML += `<tr><td colspan="${months.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
    } else {
        buyers.forEach(buyer => {
            let buyername = buyer;
            if (buyername) {
                buyername = `${buyername.slice(0, 6)}`;
            }
            buyerTableHTML += `<tr class="border-t"><td class="px-2 py-2" title="${buyer}">${buyername}</td>`;
            months.forEach(month => {
                const monthData = data.find(d => d.Mon_Name.startsWith(month));
                let total = 0;
                if (monthData) {
                    monthData.units.forEach(unit => {
                        const buyerData = unit.buyers.find(b => b.Buyer === buyer);
                        if (buyerData) total += (buyerData.buyer_total_Knit_Qty || 0);
                    });
                }
                buyerTableHTML += `<td class="px-2 py-2 drilldown cursor-pointer text-left" data-type="buyer" data-name="${buyer}" data-month="${month}">${total.toLocaleString()}</td>`;
            });
            buyerTableHTML += `<td class="px-2 py-2 text-left font-semibold">${buyerTotals[buyer].toLocaleString()}</td>`;
            buyerTableHTML += `</tr>`;
        });
    }
    buyerTableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
    months.forEach(month => {
        buyerTableHTML += `<td class="px-2 py-2 text-left">${buyerMonthlyTotals[month].toLocaleString()}</td>`;
    });
    buyerTableHTML += `<td class="px-2 py-2 text-left">${Object.values(buyerMonthlyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
    buyerTableHTML += `</tr></tfoot>`;
    $('table').eq(1).html(buyerTableHTML);

    // No initial drilldown selection; keep empty until user clicks
}

function updateBuyerTableFilter(month, unitName = null) {
    const monthData = apiData.find(d => d.Mon_Name.startsWith(month));
    // Get current 6 months from Unit table header to keep consistent columns
    const months = $('table').eq(0).find('thead th.month-header').map(function(){return $(this).data('month');}).get();
    // Aggregate buyer totals for the selected month (optionally by unit)
    let buyerSelectedMonthTotals = {};
    if (monthData) {
        if (unitName) {
            const unitData = monthData.units.find(u => u.Unit === unitName);
            if (unitData) {
                unitData.buyers.forEach(b => {
                    const key = b.Buyer;
                    const val = (b.buyer_total_Knit_Qty || 0);
                    buyerSelectedMonthTotals[key] = (buyerSelectedMonthTotals[key] || 0) + val;
                });
            }
        } else {
            monthData.units.forEach(u => {
                u.buyers.forEach(b => {
                    const key = b.Buyer;
                    const val = (b.buyer_total_Knit_Qty || 0);
                    buyerSelectedMonthTotals[key] = (buyerSelectedMonthTotals[key] || 0) + val;
                });
            });
        }
    }

    const buyers = Object.keys(buyerSelectedMonthTotals);
    // Header: keep all months
    let buyerTableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">Buyer</th>`;
    months.forEach(m => {
        buyerTableHTML += `<th class="px-2 py-2 month-header  text-right cursor-pointer" data-type="buyer" data-month="${m}">${m}</th>`;
    });
    buyerTableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;

    if (buyers.length === 0) {
        buyerTableHTML += `<tr><td colspan="${months.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
    } else {
        buyers.forEach(buyer => {
            let buyername = buyer;
            if (buyername) {
                buyername = `${buyername.slice(0, 6)}`;
            }
            buyerTableHTML += `<tr class="border-t"><td class="px-2 py-2" title="${buyer}">${buyername}</td>`;
            months.forEach(m => {
                if (m === month) {
                    const val = buyerSelectedMonthTotals[buyer] || 0;
                    buyerTableHTML += `<td class="px-2 py-2 drilldown cursor-pointer text-left" data-type="buyer" data-name="${buyer}" data-month="${m}">${val.toLocaleString()}</td>`;
                } else {
                    buyerTableHTML += `<td class="px-2 py-2 text-left">-</td>`;
                }
            });
            const total = buyerSelectedMonthTotals[buyer] || 0;
            buyerTableHTML += `<td class="px-2 py-2 text-left font-semibold">${total.toLocaleString()}</td>`;
            buyerTableHTML += `</tr>`;
        });
    }

    // Footer: '-' for non-selected months; total only for selected month and overall
    const grandTotal = Object.values(buyerSelectedMonthTotals).reduce((sum, v) => sum + v, 0);
    buyerTableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
    months.forEach(m => {
        if (m === month) {
            buyerTableHTML += `<td class="px-2 py-2 text-left">${grandTotal.toLocaleString()}</td>`;
        } else {
            buyerTableHTML += `<td class="px-2 py-2 text-left">-</td>`;
        }
    });
    buyerTableHTML += `<td class="px-2 py-2 text-left">${grandTotal.toLocaleString()}</td>`;
    buyerTableHTML += `</tr></tfoot>`;
    $('table').eq(1).html(buyerTableHTML);

    // Set info label next to Buyer title
    if (unitName) {
        $('#buyerFilterInfo').text(`| Unit: ${unitName} | Month: ${month} |`);
        currentBuyerFilterUnit = unitName;
        currentBuyerFilterMonth = month;
    } else {
        $('#buyerFilterInfo').text(`| Month: ${month} |`);
        currentBuyerFilterUnit = null;
        currentBuyerFilterMonth = month;
    }
}

function updateDrilldownTable(type, name, month, isMonthHeader = false, unitContext = null) {
    const monthData = apiData.find(d => d.Mon_Name.startsWith(month));
    if (!monthData) {
        $("#drilldownTable").html('<tr><td colspan="100" class="px-4 py-2 text-center">No data available</td></tr>');
        $("#drilldownTitle").text('Daily Performance table');
        return;
    }

    // Handle OEE data structure
    if (selectedMetric === 'Production:OEE') {
        // Get all unique days in the month data
        const daysSet = new Set();
        const shiftsByUnitAndDay = {};
        
        // Process all shifts and group by unit and day
        monthData.units?.forEach(unit => {
            if (!shiftsByUnitAndDay[unit.Unit]) {
                shiftsByUnitAndDay[unit.Unit] = {};
            }
            
            unit.shifts?.forEach(shift => {
                if (shift.EntryDate) {
                    const date = moment(shift.EntryDate);
                    const day = date.date();
                    daysSet.add(day);
                    
                    if (!shiftsByUnitAndDay[unit.Unit][day]) {
                        shiftsByUnitAndDay[unit.Unit][day] = [];
                    }
                    shiftsByUnitAndDay[unit.Unit][day].push({
                        shift: shift.Shift || 'N/A',
                        date: date.format('YYYY-MM-DD'),
                        oee: parseFloat(shift.OEE) || 0
                    });
                }
            });
        });
        
        const days = Array.from(daysSet).sort((a, b) => a - b);
        
        if (isMonthHeader) {
            // Show day-wise OEE for all units
            let tableHTML = `
                <thead>
                    <tr class="bg-gray-100">
                        <th class="px-2 py-2 text-left">Unit</th>
                        ${days.map(day => `<th class="px-2 py-2 text-right">${day}</th>`).join('')}
                        <th class="px-2 py-2 text-right">Avg</th>
                    </tr>
                </thead>
                <tbody>
            `;
            
            // Add rows for each unit
            Object.entries(shiftsByUnitAndDay).forEach(([unitName, unitData]) => {
                let rowTotal = 0;
                let dayCells = '';
                
                let dayCount = 0;
                days.forEach(day => {
                    const dayShifts = unitData[day] || [];
                    const dayOEE = dayShifts.length > 0 ? 
                        Math.round(dayShifts.reduce((sum, s) => sum + s.oee, 0) / dayShifts.length) : 0;
                    if (dayShifts.length > 0) {
                        rowTotal += dayOEE;
                        dayCount++;
                    }
                    dayCells += `<td class="px-2 py-2 text-right">${dayShifts.length > 0 ? dayOEE : '-'}</td>`;
                });
                
                const avgOEE = dayCount > 0 ? Math.round(rowTotal / dayCount) : '-';
                tableHTML += `
                    <tr class="border-t">
                        <td class="px-2 py-2 font-medium">${unitName}</td>
                        ${dayCells}
                        <td class="px-2 py-2 text-right font-semibold">${typeof avgOEE === 'number' ? Math.round(avgOEE) : avgOEE}</td>
                    </tr>
                `;
            });
            
            // Add averages row
            tableHTML += `
                <tr class="border-t bg-gray-50 font-semibold">
                    <td class="px-2 py-2">Average</td>
                    ${days.map(day => {
                        const dayData = Object.values(shiftsByUnitAndDay).reduce((acc, unitData) => {
                            const dayShifts = unitData[day] || [];
                            if (dayShifts.length > 0) {
                                acc.total += dayShifts.reduce((sum, shift) => sum + shift.oee, 0);
                                acc.count += dayShifts.length;
                            }
                            return acc;
                        }, { total: 0, count: 0 });
                        
                        const avgOEE = dayData.count > 0 ? Math.round(dayData.total / dayData.count) : '-';
                        return `<td class="px-2 py-2 text-right">${avgOEE}</td>`;
                    }).join('')}
                    <td class="px-2 py-2 text-right">
                        ${(() => {
                            const totals = Object.values(shiftsByUnitAndDay).reduce((acc, unitData) => {
                                const unitTotal = Object.values(unitData).reduce((sum, dayShifts) => {
                                    if (dayShifts.length > 0) {
                                        acc.total += dayShifts.reduce((s, shift) => s + shift.oee, 0);
                                        acc.count += dayShifts.length;
                                    }
                                    return acc;
                                }, { total: 0, count: 0 });
                                return {
                                    total: acc.total + unitTotal.total,
                                    count: acc.count + unitTotal.count
                                };
                            }, { total: 0, count: 0 });
                            
                            return totals.count > 0 ? Math.round(totals.total / totals.count) : '-';
                        })()}
                    </td>
                </tr>
            `;
            
            tableHTML += '</tbody>';
            $("#drilldownTable").html(tableHTML);
            $("#drilldownTitle").text(`Daily OEE for ${month} - All Units`);
            return;
            
        } else if (type === 'unit') {
            // Show day-wise shifts for the selected unit
            const unitData = shiftsByUnitAndDay[name] || {};
            const allShifts = [];
            
            // Collect all unique shifts
            Object.values(unitData).forEach(dayShifts => {
                dayShifts.forEach(shift => {
                    if (!allShifts.some(s => s.shift === shift.shift)) {
                        allShifts.push(shift);
                    }
                });
            });
            
            let tableHTML = `
                <thead>
                    <tr class="bg-gray-100">
                        <th class="px-2 py-2 text-left">Shift</th>
                        ${days.map(day => `<th class="px-2 py-2 text-right">${day}</th>`).join('')}
                        <th class="px-2 py-2 text-right">Avg</th>
                    </tr>
                </thead>        
                <tbody>
            `;
            
            // Add rows for each shift
            allShifts.forEach(shift => {
                let shiftTotal = 0;
                let dayCount = 0;
                let dayCells = '';
                
                days.forEach(day => {
                    const dayShifts = unitData[day] || [];
                    const shiftData = dayShifts.filter(s => s.shift === shift.shift);
                    const dayOEE = shiftData.length > 0 ? 
                        parseFloat((shiftData.reduce((sum, s) => sum + s.oee, 0) / shiftData.length).toFixed(2)) : 0;
                    
                    if (shiftData.length > 0) {
                        shiftTotal += dayOEE;
                        dayCount++;
                    }
                    
                    dayCells += `<td class="px-2 py-2 text-right">${shiftData.length > 0 ? Math.round(dayOEE) : '-'}</td>`;
                });
                
                const avgOEE = dayCount > 0 ? Math.round(shiftTotal / dayCount) : '-';
                tableHTML += `
                    <tr class="border-t">
                        <td class="px-2 py-2">${shift.shift}</td>
                        ${dayCells}
                        <td class="px-2 py-2 text-right font-semibold">${typeof avgOEE === 'number' ? Math.round(avgOEE) : avgOEE}</td>
                    </tr>
                `;
            });
            
            // Add averages row
            tableHTML += `
                <tr class="border-t bg-gray-50 font-semibold">
                    <td>Average</td>
                    ${days.map(day => {
                        const dayShifts = unitData[day] || [];
                        if (dayShifts.length === 0) return '<td class="px-2 py-2 text-right">-</td>';
                        
                        const dayTotal = dayShifts.reduce((sum, s) => sum + s.oee, 0);
                        const dayAvg = dayShifts.length > 0 ? Math.round(dayTotal / dayShifts.length) : 0;
                        return `<td class="px-2 py-2 text-right">${dayAvg}</td>`;
                    }).join('')}
                    <td class="px-2 py-2 text-right">
                        ${(() => {
                            const allShifts = Object.values(unitData).flat();
                            if (allShifts.length === 0) return '-';
                            const totalOEE = allShifts.reduce((sum, s) => sum + s.oee, 0);
                            return allShifts.length > 0 ? Math.round(totalOEE / allShifts.length) : 0;
                        })()}
                    </td>
                </tr>
            `;
            
            tableHTML += '</tbody>';
            $("#drilldownTable").html(tableHTML);
            $("#drilldownTitle").text(`Daily OEE for ${name} in ${month}`);
            return;
        }
    }

    let daysSet = new Set();
    monthData.units.forEach(unit => {
        unit.buyers.forEach(buyer => {
            if (buyer.shifts) {
                buyer.shifts.forEach(shift => {
                    if (shift.machines) {
                        shift.machines.forEach(item => {
                            const day = moment(item.EntryDate || item.User_Date).date();
                            daysSet.add(day);
                        });
                    }
                });
            } else if (buyer.lines) {
                buyer.lines.forEach(line => {
                    line.data.forEach(item => {
                        const day = moment(item.User_Date).date();
                        daysSet.add(day);
                    });
                });
            } else if (buyer.data) {
                buyer.data.forEach(item => {
                    const day = moment(item.User_Date).date();
                    daysSet.add(day);
                });
            }
        });
    });
    let days = Array.from(daysSet).sort((a, b) => a - b);

    if (isMonthHeader) {
        // Special handling: if Buyer header clicked while a Unit is filtered, show buyers for that unit only
        if (type === 'buyer' && unitContext) {
            const entityType = 'Buyer';
            $("#drilldownTitle").text(`Daily Performance for All ${entityType}s in ${month} (Unit: ${unitContext})`);
            let unitData = monthData.units.find(u => u.Unit === unitContext);
            let entities = unitData ? [...new Set(unitData.buyers.map(b => b.Buyer))] : [];
            let tableData = {};
            let dailyTotals = {};
            let entityTotals = {};
            days.forEach(day => dailyTotals[day] = 0);
            entities.forEach(entity => {
                tableData[entity] = {};
                entityTotals[entity] = 0;
                days.forEach(day => tableData[entity][day] = 0);
            });

            if (unitData) {
                unitData.buyers.forEach(buyer => {
                    if (buyer.shifts) {
                        buyer.shifts.forEach(shift => {
                            (shift.machines || []).forEach(item => {
                                const day = moment(item.EntryDate || item.User_Date).date();
                                const qty = parseInt(item.Knit_Qty || item.Qty || 0) || 0;
                                const entity = buyer.Buyer;
                                tableData[entity][day] += qty;
                                dailyTotals[day] += qty;
                                entityTotals[entity] += qty;
                            });
                        });
                    } else if (buyer.lines) {
                        buyer.lines.forEach(line => {
                            line.data.forEach(item => {
                                const day = moment(item.User_Date).date();
                                const qty = parseInt(item.Qty) || 0;
                                const entity = buyer.Buyer;
                                tableData[entity][day] += qty;
                                dailyTotals[day] += qty;
                                entityTotals[entity] += qty;
                            });
                        });
                    } else if (buyer.data) {
                        buyer.data.forEach(item => {
                            const day = moment(item.User_Date).date();
                            const qty = parseInt(item.Qty) || 0;
                            const entity = buyer.Buyer;
                            tableData[entity][day] += qty;
                            dailyTotals[day] += qty;
                            entityTotals[entity] += qty;
                        });
                    }
                });
            }

            let tableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">${entityType}</th>`;
            days.forEach(day => { tableHTML += `<th class="px-2 py-2 text-right">${day}</th>`; });
            tableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
            if (entities.length === 0) {
                tableHTML += `<tr><td colspan="${days.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
            } else {
                entities.forEach(entity => {
                    let entityname = entity;
                    if (entityname) {
                        entityname = `${entityname.slice(0, 6)}`;
                        tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left" title="${entity}">${entityname}</td>`;
                    } else {
                        tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left">${entityname}</td>`;
                    }
                    // Infer year from any sample date in this month
                    let inferredYear = null;
                    if (!inferredYear) {
                        const u0 = monthData.units?.[0];
                        let sampleDate = null;
                        if (u0?.buyers?.[0]?.shifts?.[0]?.machines?.[0]) sampleDate = u0.buyers[0].shifts[0].machines[0].EntryDate;
                        else if (u0?.buyers?.[0]?.data?.[0]) sampleDate = u0.buyers[0].data[0].User_Date;
                        else if (u0?.buyers?.[0]?.lines?.[0]?.data?.[0]) sampleDate = u0.buyers[0].lines[0].data[0].User_Date;
                        if (sampleDate) inferredYear = moment(sampleDate).year();
                    }
                    days.forEach(day => {
                        const value = tableData[entity][day] || 0;
                        const clickable = value > 0;
                        const cls = `px-2 py-2 text-right`; // ${clickable ? 'day-cell clickable text-blue-700 hover:underline' : ''}
                        tableHTML += `<td class="${cls}" data-day="${day}" data-month="${month}" data-year="${inferredYear || moment().year()}">${value.toLocaleString()}</td>`;
                    });
                    tableHTML += `<td class="px-2 py-2 text-right font-semibold">${entityTotals[entity].toLocaleString()}</td>`;
                    tableHTML += `</tr>`;
                });
            }
            tableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
            days.forEach(day => { tableHTML += `<td class="px-2 py-2 text-right">${dailyTotals[day].toLocaleString()}</td>`; });
            tableHTML += `<td class="px-2 py-2 text-right">${Object.values(dailyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
            tableHTML += `</tr></tfoot>`;
            $("#drilldownTable").html(tableHTML);
            return;
        }

        const entityType = type === 'unit' ? 'Unit' : 'Buyer';
        $("#drilldownTitle").text(`Daily Performance for All ${entityType}s in ${month}`);
        let entities = type === 'unit' ? [...new Set(monthData.units.map(u => u.Unit))] : 
                      [...new Set(monthData.units.flatMap(u => u.buyers.map(b => b.Buyer)))];
        let tableData = {};
        let dailyTotals = {};
        let entityTotals = {};
        days.forEach(day => dailyTotals[day] = 0);
        entities.forEach(entity => {
            tableData[entity] = {};
            entityTotals[entity] = 0;
            days.forEach(day => tableData[entity][day] = 0);
        });

        monthData.units.forEach(unit => {
            unit.buyers.forEach(buyer => {
                if (buyer.shifts) {
                    buyer.shifts.forEach(shift => {
                        (shift.machines || []).forEach(item => {
                            const day = moment(item.EntryDate || item.User_Date).date();
                            const qty = parseInt(item.Knit_Qty || item.Qty || 0) || 0;
                            const entity = type === 'unit' ? unit.Unit : buyer.Buyer;
                            tableData[entity][day] += qty;
                            dailyTotals[day] += qty;
                            entityTotals[entity] += qty;
                        });
                    });
                } else if (buyer.lines) {
                    buyer.lines.forEach(line => {
                        line.data.forEach(item => {
                            const day = moment(item.User_Date).date();
                            const qty = parseInt(item.Qty) || 0;
                            const entity = type === 'unit' ? unit.Unit : buyer.Buyer;
                            tableData[entity][day] += qty;
                            dailyTotals[day] += qty;
                            entityTotals[entity] += qty;
                        });
                    });
                } else if (buyer.data) {
                    buyer.data.forEach(item => {
                        const day = moment(item.User_Date).date();
                        const qty = parseInt(item.Qty) || 0;
                        const entity = type === 'unit' ? unit.Unit : buyer.Buyer;
                        tableData[entity][day] += qty;
                        dailyTotals[day] += qty;
                        entityTotals[entity] += qty;
                    });
                }
            });
        });

        let tableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">${entityType}</th>`;
        days.forEach(day => {
            tableHTML += `<th class="px-2 py-2 text-right">${day}</th>`;
        });
        tableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
        if (entities.length === 0) {
            tableHTML += `<tr><td colspan="${days.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
        } else {
            entities.forEach(entity => {
                let entityname = entity;
                if (entityname && type === 'buyer') {
                    entityname = `${entityname.slice(0, 6)}`;
                    tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left" title="${entity}">${entityname}</td>`;
                } else {
                    tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left">${entityname}</td>`;
                }
                // Infer year for this month
                let inferredYear = null;
                if (!inferredYear) {
                    const u0 = monthData.units?.[0];
                    let sampleDate = null;
                    if (u0?.buyers?.[0]?.shifts?.[0]?.machines?.[0]) sampleDate = u0.buyers[0].shifts[0].machines[0].EntryDate;
                    else if (u0?.buyers?.[0]?.data?.[0]) sampleDate = u0.buyers[0].data[0].User_Date;
                    else if (u0?.buyers?.[0]?.lines?.[0]?.data?.[0]) sampleDate = u0.buyers[0].lines[0].data[0].User_Date;
                    if (sampleDate) inferredYear = moment(sampleDate).year();
                }
                days.forEach(day => {
                    const value = tableData[entity][day] || 0;
                    const clickable = value > 0;
                    const cls = `px-2 py-2 text-right`; //  ${clickable ? 'day-cell clickable text-blue-700 hover:underline' : ''}
                    tableHTML += `<td class="${cls}" data-day="${day}" data-month="${month}" data-year="${inferredYear || moment().year()}">${value.toLocaleString()}</td>`;
                });
                tableHTML += `<td class="px-2 py-2 text-right font-semibold">${entityTotals[entity].toLocaleString()}</td>`;
                tableHTML += `</tr>`;
            });
        }
        tableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
        days.forEach(day => {
            tableHTML += `<td class="px-2 py-2 text-right">${dailyTotals[day].toLocaleString()}</td>`;
        });
        tableHTML += `<td class="px-2 py-2 text-right">${Object.values(dailyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
        tableHTML += `</tr></tfoot>`;
        $("#drilldownTable").html(tableHTML);
    } else {
        const entityType = type === 'unit' ? 'Unit' : 'Buyer';
        const isCuttingMetric = selectedMetric.startsWith('Cutting');
        const titlePrefix = type === 'unit' ? (isCuttingMetric ? 'Buyer-wise' : 'Shift-wise') : (isCuttingMetric ? 'Order-wise' : 'OCN-wise');
        $("#drilldownTitle").text(`${titlePrefix} Daily Performance for ${entityType} ${name} in ${month}`);

        if (type === 'unit') {
            const isCuttingMetric = selectedMetric.startsWith('Cutting');
            let entitiesSet = new Set();
            let tableData = {};
            let dailyTotals = {};
            let entityTotals = {};
            // machine-wise aggregation per shift
            let machineData = {}; // { shiftNo: { machineId: { [day]: qty } } }
            let machineTotals = {}; // { shiftNo: { machineId: totalQty } }
            days.forEach(day => dailyTotals[day] = 0);
            const unitData = monthData.units.find(u => u.Unit === name);
            
            if (unitData) {
                unitData.buyers.forEach(buyer => {
                    if (isCuttingMetric && buyer.data) {
                        // Material type-wise data for Cutting metrics
                        buyer.data.forEach(item => {
                            const materialType = item.Machine_type || 'Unknown';
                            entitiesSet.add(materialType);
                            const day = moment(item.EntryDate || item.User_Date).date();
                            const qty = parseInt(item.Knit_Qty || item.Qty || 0) || 0;
                            if (!tableData[materialType]) tableData[materialType] = {};
                            if (!tableData[materialType][day]) tableData[materialType][day] = 0;
                            if (!entityTotals[materialType]) entityTotals[materialType] = 0;
                            tableData[materialType][day] += qty;
                            dailyTotals[day] += qty;
                            entityTotals[materialType] += qty;
                        });
                    } else if (buyer.shifts) {
                        // Shift-wise data for non-Cutting metrics
                        buyer.shifts.forEach(shift => {
                            const shiftNo = (shift.Shift || '').toString().trim() || 'Unknown';
                            entitiesSet.add(shiftNo);
                            (shift.machines || []).forEach(item => {
                                const day = moment(item.EntryDate || item.User_Date).date();
                                const qty = parseInt(item.Knit_Qty || item.Qty || 0) || 0;
                                if (!tableData[shiftNo]) tableData[shiftNo] = {};
                                if (!tableData[shiftNo][day]) tableData[shiftNo][day] = 0;
                                if (!entityTotals[shiftNo]) entityTotals[shiftNo] = 0;
                                tableData[shiftNo][day] += qty;
                                dailyTotals[day] += qty;
                                entityTotals[shiftNo] += qty;
                                // machine aggregation
                                const mch = (item.MachineID || '').toString().trim() || 'Unknown';
                                if (!machineData[shiftNo]) machineData[shiftNo] = {};
                                if (!machineData[shiftNo][mch]) machineData[shiftNo][mch] = {};
                                if (!machineData[shiftNo][mch][day]) machineData[shiftNo][mch][day] = 0;
                                machineData[shiftNo][mch][day] += qty;
                                if (!machineTotals[shiftNo]) machineTotals[shiftNo] = {};
                                if (!machineTotals[shiftNo][mch]) machineTotals[shiftNo][mch] = 0;
                                machineTotals[shiftNo][mch] += qty;
                            });
                        });
                    } else if (buyer.data) {
                        // Fallback
                        const buyerName = buyer.Buyer;
                        entitiesSet.add(buyerName);
                        buyer.data.forEach(item => {
                            const day = moment(item.User_Date).date();
                            const qty = parseInt(item.Qty) || 0;
                            if (!tableData[buyerName]) tableData[buyerName] = {};
                            if (!tableData[buyerName][day]) tableData[buyerName][day] = 0;
                            if (!entityTotals[buyerName]) entityTotals[buyerName] = 0;
                            tableData[buyerName][day] += qty;
                            dailyTotals[day] += qty;
                            entityTotals[buyerName] += qty;
                        });
                    }
                });
            }

            let entities = Array.from(entitiesSet).sort((a, b) => a.localeCompare(b, undefined, { numeric: true }));
            let tableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">${isCuttingMetric ? 'Machine Type' : 'Shift'}</th>`;
            days.forEach(day => {
                tableHTML += `<th class="px-2 py-2 text-right">${day}</th>`;
            });
            tableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
            if (entities.length === 0) {
                tableHTML += `<tr><td colspan="${days.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
            } else {
                entities.forEach(entity => {
                    let entityName = entity;
                    if (isCuttingMetric) {
                        entityName = `${entityName.slice(0, 6)}`;
                        tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left" title="${entity}">${entityName}</td>`;
                    } else {
                        tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left">${entityName}</td>`;
                    }
                    days.forEach(day => {
                        const value = tableData[entity]?.[day] || 0;
                        const clickable = value > 0;
                        const cls = `px-2 py-2 text-right `; // ${clickable ? 'day-cell clickable text-blue-700 hover:underline' : ''}
                        tableHTML += `<td class="${cls}" data-day="${day}" data-month="${month}" data-year="${moment().year()}">${value.toLocaleString()}</td>`;
                    });
                    tableHTML += `<td class="px-2 py-2 text-right font-semibold">${entityTotals[entity].toLocaleString()}</td>`;
                    tableHTML += `</tr>`;
                });
            }
            tableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
            days.forEach(day => {
                tableHTML += `<td class="px-2 py-2 text-right">${dailyTotals[day].toLocaleString()}</td>`;
            });
            tableHTML += `<td class="px-2 py-2 text-right">${Object.values(dailyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
            tableHTML += `</tr></tfoot>`;
            $("#drilldownTable").html(tableHTML);
        } 
		  else {
            let ocnsSet = new Set();
            let tableData = {};
            let dailyTotals = {};
            let ocnTotals = {};
            // Per-OCN nested aggregations
            let ocnShiftData = {};   // { ocn: { shiftNo: { [day]: qty } } }
            let ocnShiftTotals = {}; // { ocn: { shiftNo: total } }
            let ocnMachineData = {}; // { ocn: { shiftNo: { machineId: { [day]: qty } } } }
            let ocnMachineTotals = {}; // { ocn: { shiftNo: { machineId: total } } }
            days.forEach(day => dailyTotals[day] = 0);
            monthData.units.forEach(unit => {
                if (unitContext && unit.Unit !== unitContext) return; // restrict to selected unit when provided
                const buyerData = unit.buyers.find(b => b.Buyer === name);
                if (buyerData) {
                    if (buyerData.shifts) {
                        buyerData.shifts.forEach(shift => {
                            const shiftNo = (shift.Shift || '').toString().trim() || 'Unknown';
                            (shift.machines || []).forEach(item => {
                                const ocn = item.Order_No || 'Unknown';
                                ocnsSet.add(ocn);
                                const day = moment(item.EntryDate || item.User_Date).date();
                                const qty = parseInt(item.Knit_Qty || item.Qty || 0) || 0;
                                if (!tableData[ocn]) tableData[ocn] = {};
                                if (!tableData[ocn][day]) tableData[ocn][day] = 0;
                                if (!ocnTotals[ocn]) ocnTotals[ocn] = 0;
                                tableData[ocn][day] += qty;
                                dailyTotals[day] += qty;
                                ocnTotals[ocn] += qty;
                                // per-OCN shift aggregation
                                if (!ocnShiftData[ocn]) ocnShiftData[ocn] = {};
                                if (!ocnShiftData[ocn][shiftNo]) ocnShiftData[ocn][shiftNo] = {};
                                if (!ocnShiftData[ocn][shiftNo][day]) ocnShiftData[ocn][shiftNo][day] = 0;
                                ocnShiftData[ocn][shiftNo][day] += qty;
                                if (!ocnShiftTotals[ocn]) ocnShiftTotals[ocn] = {};
                                if (!ocnShiftTotals[ocn][shiftNo]) ocnShiftTotals[ocn][shiftNo] = 0;
                                ocnShiftTotals[ocn][shiftNo] += qty;
                                // per-OCN machine aggregation within shift
                                if (!ocnMachineData[ocn]) ocnMachineData[ocn] = {};
                                if (!ocnMachineData[ocn][shiftNo]) ocnMachineData[ocn][shiftNo] = {};
                                const mch = (
                                    item.Machine || item.Machine_No || item.MachineNo || item.MachineID || item.Line_No || ''
                                ).toString().trim() || 'Unknown';
                                if (!ocnMachineData[ocn][shiftNo][mch]) ocnMachineData[ocn][shiftNo][mch] = {};
                                if (!ocnMachineData[ocn][shiftNo][mch][day]) ocnMachineData[ocn][shiftNo][mch][day] = 0;
                                ocnMachineData[ocn][shiftNo][mch][day] += qty;
                                if (!ocnMachineTotals[ocn]) ocnMachineTotals[ocn] = {};
                                if (!ocnMachineTotals[ocn][shiftNo]) ocnMachineTotals[ocn][shiftNo] = {};
                                if (!ocnMachineTotals[ocn][shiftNo][mch]) ocnMachineTotals[ocn][shiftNo][mch] = 0;
                                ocnMachineTotals[ocn][shiftNo][mch] += qty;
                            });
                        });
                    } else if (buyerData.lines) {
                        buyerData.lines.forEach(line => {
                            line.data.forEach(item => {
                                const ocn = item.Order_No || 'Unknown';
                                ocnsSet.add(ocn);
                                const day = moment(item.User_Date).date();
                                const qty = parseInt(item.Qty) || 0;
                                if (!tableData[ocn]) tableData[ocn] = {};
                                if (!tableData[ocn][day]) tableData[ocn][day] = 0;
                                if (!ocnTotals[ocn]) ocnTotals[ocn] = 0;
                                tableData[ocn][day] += qty;
                                dailyTotals[day] += qty;
                                ocnTotals[ocn] += qty;
                            });
                        });
                    } else if (buyerData.data) {
                        buyerData.data.forEach(item => {
                            const ocn = item.Order_No || 'Unknown';
                            ocnsSet.add(ocn);
                            const day = moment(item.User_Date).date();
                            const qty = parseInt(item.Qty) || 0;
                            if (!tableData[ocn]) tableData[ocn] = {};
                            if (!tableData[ocn][day]) tableData[ocn][day] = 0;
                            if (!ocnTotals[ocn]) ocnTotals[ocn] = 0;
                            tableData[ocn][day] += qty;
                            dailyTotals[day] += qty;
                            ocnTotals[ocn] += qty;
                        });
                    }
                }
            });

            let ocns = Array.from(ocnsSet).sort((a, b) => a.localeCompare(b, undefined, { numeric: true }));
            let tableHTML = `<thead><tr class="bg-gray-100"><th class="px-2 py-2 text-left">${isCuttingMetric ? 'Order' : 'OCN'}</th>`;
            days.forEach(day => {
                tableHTML += `<th class="px-2 py-2 text-right">${day}</th>`;
            });
            tableHTML += `<th class="px-2 py-2 text-right">Total</th></tr></thead><tbody>`;
            if (ocns.length === 0) {
                tableHTML += `<tr><td colspan="${days.length + 2}" class="px-2 py-2 text-center">No data available</td></tr>`;
            } else {
                ocns.forEach(ocn => {
                    tableHTML += `<tr class="border-t"><td class="px-2 py-2 text-left">${ocn.slice(-5)}</td>`;
                    // Infer year for this month
                    let inferredYear = null;
                    if (!inferredYear) {
                        const u0 = monthData.units?.[0];
                        let sampleDate = null;
                        if (u0?.buyers?.[0]?.data?.[0]) sampleDate = u0.buyers[0].data[0].User_Date;
                        else if (u0?.buyers?.[0]?.lines?.[0]?.data?.[0]) sampleDate = u0.buyers[0].lines[0].data[0].User_Date;
                        if (sampleDate) inferredYear = moment(sampleDate).year();
                    }
                    days.forEach(day => {
                        const value = tableData[ocn]?.[day] || 0;
                        const clickable = value > 0;
                        const cls = `px-2 py-2 text-right `; // ${clickable ? 'day-cell clickable text-blue-700 hover:underline' : ''}
                        tableHTML += `<td class="${cls}" data-day="${day}" data-month="${month}" data-year="${inferredYear || moment().year()}">${value.toLocaleString()}</td>`;
                    });
                    tableHTML += `<td class="px-2 py-2 text-right font-semibold">${ocnTotals[ocn].toLocaleString()}</td>`;
                    tableHTML += `</tr>`;
                });
            }
            tableHTML += `</tbody><tfoot><tr class="bg-gray-200 font-semibold"><td class="px-2 py-2 text-left">Total</td>`;
            days.forEach(day => {
                tableHTML += `<td class="px-2 py-2 text-right">${dailyTotals[day].toLocaleString()}</td>`;
            });
            tableHTML += `<td class="px-2 py-2 text-right">${Object.values(dailyTotals).reduce((sum, val) => sum + val, 0).toLocaleString()}</td>`;
            tableHTML += `</tr></tfoot>`;
            $("#drilldownTable").html(tableHTML);
        }
    }
}

$(document).ready(function() {
   

    const $menuL1 = $(".metrics-dropdown .menu.level1");
    const $btn = $("#metricsBtn");
    $btn.on('click', function(e) {
        e.stopPropagation();
        $menuL1.toggleClass('hidden');
    });
    $(document).on('click', function() {
        $menuL1.addClass('hidden');
    });
    $menuL1.on('click', function(e) {
        e.stopPropagation();
    });
    $('.submenu-item').on('click', function(e) {
        e.stopPropagation();
        const val = $(this).data('val');
        const type = $(this).data('type') || '';
        $btn.contents().filter(function() { return this.nodeType === 3; }).first().replaceWith(val.split(':')[1] + ' ');
        $btn.data('val', val).data('type', type);
        $menuL1.addClass('hidden');
        selectedMetric = val;
        selectedType = type;
        const metricCat = (selectedMetric || '').split(':')[0] || '';
        // Get the query params from current URL for breadcrumb links
        const urlParams = new URLSearchParams(window.location.search);
        const fromdate = urlParams.get('fromdate') || '';
        const enddate  = urlParams.get('enddate') || '';
        if (metricCat === 'Production') {
            // Simplified breadcrumb: Production Dashboard > Historical Data
            $('.main_breadcrumb').html(
              `<a id="productionBreadcrumb" href="/production_socks_dashboard?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-700 hover:underline">Production Dashboard</a>
               <span> &gt; </span>
               <span class="text-gray-800 font-medium">Historical Data</span>`
            );
        } else {
            // Restore original breadcrumb structure so specific handlers can populate portal_type_link
            $('.main_breadcrumb').html(
              `<a id="productionBreadcrumb" href="/production_socks_dashboard?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-700 hover:underline">Production Dashboard</a>
               <span> &gt; </span>
               <span class="text-gray-500 main_breadcrumb_text portal_type_link"> </span>
               <span> &gt; </span>
               <span class="text-gray-800 font-medium">Historical Data</span>`
            );
        }
        const [displayStart, displayEnd] = $('#dateRangePicker').val().split(' - ');
        const startFormatted = displayStart ? moment(displayStart, 'DD-MM-YY').format('YYYY-MM-DD') : '';
        const endFormatted = displayEnd ? moment(displayEnd, 'DD-MM-YY').format('YYYY-MM-DD') : '';
        const [metricCategory, metricName] = selectedMetric.split(':');
        const titleText = `History for Socks Overview : ${displayStart} - ${displayEnd} | ${metricCategory} : ${metricName}`;
        $('.dashboard_title').text(titleText);
        $('.drilldown, .month-header').removeClass('highlight');
        fetchData(selectedType, startFormatted, endFormatted, portal_type, selectedMetric);
    });

    $(document).on('click', '.drilldown', function() {
        const type = $(this).data('type');
        const name = $(this).data('name');
        const month = $(this).data('month');
        $('.drilldown, .month-header').removeClass('highlight');
        $(this).addClass('highlight');
        // If clicking a Buyer cell while a Unit is filtered, pass the unit context to drilldown
        const unitCtx = (type === 'buyer' && currentBuyerFilterUnit) ? currentBuyerFilterUnit : null;
        updateDrilldownTable(type, name, month, false, unitCtx);
        if (type === 'unit') {
            updateBuyerTableFilter(month, name);
            saveSelection({ mode: 'cell', type, name, month });
        } else {
            saveSelection({ mode: 'cell', type, name, month, unitContext: unitCtx });
        }
    });
    // Toggle machine detail rows (unit drilldown -> shift-wise)
    $(document).on('click', '.toggle-shift', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const $btn = $(this);
        const $row = $btn.closest('tr');
        const $details = $row.next('.machine-details');
        if ($details.length) {
            $details.toggleClass('hidden');
            $btn.find('i').toggleClass('fa-chevron-down fa-chevron-up');
        }
    });
    // Toggle OCN detail rows (buyer drilldown -> OCN-wise)
    $(document).on('click', '.toggle-ocn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const $btn = $(this);
        const $row = $btn.closest('tr');
        const $details = $row.next('.ocn-details');
        if ($details.length) {
            const isOpen = $details.is(':visible');
            if (isOpen) {
                $details.hide();
                $btn.find('i').removeClass('fa-chevron-up').addClass('fa-chevron-down');
            } else {
                $details.show();
                $btn.find('i').removeClass('fa-chevron-down').addClass('fa-chevron-up');
                // Collapse all inner machine rows and reset chevrons when opening
                $details.find('.machine-details-inner').each(function(){
                    $(this).hide();
                    $(this).addClass('hidden');
                    $(this).find('.machine-container').hide();
                });
                $details.find('.toggle-shift-inner i').each(function(){
                    $(this).removeClass('fa-chevron-up').addClass('fa-chevron-down');
                });
            }
        }
    });
    // Toggle inner machine list under a shift inside OCN detail table
    $(document).on('click', '.toggle-shift-inner', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const $btn = $(this);
        const $row = $btn.closest('tr');
        const $next = $row.nextAll('.machine-details-inner').first();
        if ($next.length) {
            const $container = $next.find('.machine-container').first();
            const opening = !$next.is(':visible') || ($container.length && !$container.is(':visible'));
            if (opening) {
                $next.removeClass('hidden').css('display','table-row');
                if ($container.length) $container.css('display','block');
            } else {
                if ($container.length) $container.css('display','none');
                $next.addClass('hidden').css('display','none');
            }
            $btn.find('i').toggleClass('fa-chevron-down fa-chevron-up');
        }
    });
    $(document).on('click', '.month-header', function() {
        const type = $(this).data('type');
        const month = $(this).data('month');
        $('.drilldown, .month-header').removeClass('highlight');
        $(this).addClass('highlight');
        // If clicking Buyer header while a Unit is filtered, show Buyer-wise drilldown for that Unit
        const isBuyerHeader = type === 'buyer';
        const unitCtx = (isBuyerHeader && currentBuyerFilterUnit) ? currentBuyerFilterUnit : null;
        updateDrilldownTable(type, null, month, true, unitCtx);
        if (type === 'unit') {
            updateBuyerTableFilter(month, null);
            saveSelection({ mode: 'month', type, month });
        } else {
            saveSelection({ mode: 'month', type, month, unitContext: unitCtx });
        }
    });
    // Helper: resolve base dashboard URL dynamically from breadcrumb or portal type
    function resolveDashboardBase() {
        const href = $('#productionBreadcrumb').attr('href') || $('.portal_type_link a').attr('href') || '';
        if (href) {
            return href.split('?')[0];
        }
        const map = {
            'Production': '/production_socks_dashboard',
            'Line Stop': '/linestop_analysis',
            'Cutting': '/cutting_output',
            'WIP': '/wip',
            'Labour': '/direct_labour'
        };
        return map[portal_type] || '/production_socks_dashboard';
    }
    // Navigate when clicking a day cell in the drilldown table
    $(document).on('click', '#drilldownTable td.day-cell', function() {
        const day = $(this).data('day');
        const mon = $(this).data('month');
        const year = $(this).data('year');
        if (!day || !mon || !year) return;
        const date = moment(`${day}-${mon}-${year}`, 'D-MMM-YYYY').format('YYYY-MM-DD');
        if (!date || date === 'Invalid date') return;
       const base = resolveDashboardBase();
        window.location.href = `${base}?fromdate=${date}&enddate=${date}`;
    });
    $('#dateRangePicker').daterangepicker({
        opens: 'left',
        startDate: moment().subtract(5, 'months').startOf('month'),
        endDate: moment(),
        maxDate: moment(),
        ranges: {
            'Last 6 Months': [moment().subtract(5, 'months').startOf('month'), moment()],
            'This Month': [moment().startOf('month'), moment().endOf('month')]
        },
        locale: {
            format: 'DD-MM-YY',
            separator: ' - '
        },
        isInvalidDate: function(date) {
            return date.isAfter(moment(), 'day');
        }
    }, function(start, end, label) {
        const displayStartFormatted = start.format('DD-MM-YY');
        const displayEndFormatted = end.format('DD-MM-YY');
        const [metricCategory, metricName] = selectedMetric.split(':');
        const titleText = `History for Socks Overview : ${displayStartFormatted} - ${displayEndFormatted} | ${metricCategory} : ${metricName}`;
        $('.dashboard_title').text(titleText);
        $('#dateRangePicker').val(`${displayStartFormatted} - ${displayEndFormatted}`);
        const startFormatted = start.format('YYYY-MM-DD');
        const endFormatted = end.format('YYYY-MM-DD');
        $('.drilldown, .month-header').removeClass('highlight');
        fetchData(selectedType, startFormatted, endFormatted, portal_type, selectedMetric);
    });

    // Define mapping of portal types to their first available metric
    const portalMetricMap = {
        'Production': 'Production:Knitting',
        'Line Stop': 'Line Stop:Minutes',
        'Cutting': 'Cutting:Total Cut',
        'WIP': 'WIP:WIP',
        'Labour': 'Labour:Strength'
    };

    // Get the portal type (default to 'Production' if not set)
    const portalType = portal_type || 'Production';

    // Set the initial metric based on portal type
    selectedMetric = portalMetricMap[portalType] || 'Production:Knitting';
    selectedType = $(`.submenu-item[data-val="${selectedMetric}"]`).data('type') || '1';

    // Update dropdown button
    $btn.contents().filter(function() { return this.nodeType === 3; }).first().replaceWith(selectedMetric.split(':')[1] + ' ');
    $btn.data('val', selectedMetric).data('type', selectedType);

    const initialStart = moment().subtract(5, 'months').startOf('month');
    const initialEnd = moment();
    const initialStartFormatted = initialStart.format('YYYY-MM-DD');
    const initialEndFormatted = initialEnd.format('YYYY-MM-DD');
    const initialDisplayStartFormatted = initialStart.format('DD-MM-YY');
    const initialDisplayEndFormatted = initialEnd.format('DD-MM-YY');
    const [metricCategory, metricName] = selectedMetric.split(':');
    $('.dashboard_title').text(`History for Socks Overview : ${initialDisplayStartFormatted} - ${initialDisplayEndFormatted} | ${metricCategory} : ${metricName}`);
    $('#dateRangePicker').val(`${initialDisplayStartFormatted} - ${initialDisplayEndFormatted}`);
    selectedMetric = $btn.data('val') || 'Production:Knitting';
    selectedType = $btn.data('type') || '1';
    // Set initial breadcrumb based on selected metric category
    (function(){
      const cat = (selectedMetric || '').split(':')[0] || '';
      const urlParams = new URLSearchParams(window.location.search);
      const fromdate = urlParams.get('fromdate') || '';
      const enddate  = urlParams.get('enddate') || '';
      if (cat === 'Production') {
        $('.main_breadcrumb').html(
          `<a id="productionBreadcrumb" href="/production_socks_dashboard?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-700 hover:underline">Production Dashboard</a>
           <span> &gt; </span>
           <span class="text-gray-800 font-medium">Historical Data</span>`
        );
      } else {
        $('.main_breadcrumb').html(
          `<a id="productionBreadcrumb" href="/production_socks_dashboard?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-700 hover:underline">Production Dashboard</a>
           <span> &gt; </span>
           <span class="text-gray-500 main_breadcrumb_text portal_type_link"> </span>
           <span> &gt; </span>
           <span class="text-gray-800 font-medium">Historical Data</span>`
        );
      }
    })();
    fetchData(selectedType, initialStartFormatted, initialEndFormatted, portalType, selectedMetric);

    $('#notificationsBtn').click(function() {
        $('#notificationOverlay').removeClass('hidden');
        setTimeout(() => {
        $('#notificationPanel').removeClass('translate-x-full');
        }, 10);
    });
    $(document).on('click', '#notificationOverlay, #closeNotifications', function(e) {
        if ($(e.target).is('#notificationOverlay') || $(e.target).closest('#closeNotifications').length) {
        $('#notificationPanel').addClass('translate-x-full');
        setTimeout(() => {
            $('#notificationOverlay').addClass('hidden');
        }, 300);
        }
    });

    // Get the query params from current URL
    const urlParams = new URLSearchParams(window.location.search);
    const fromdate = urlParams.get('fromdate') || '';
    const enddate  = urlParams.get('enddate') || '';

    
    // Line Stop Dashboard
    $('.line_stop_item').on('click', function() {
        $('.main_breadcrumb_text').html(
            `<a href="/linestop_analysis?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Line Stop Dashboard
            </a>`
        );
    });

    // WIP Dashboard
    $('.wip_item').on('click', function() {
        $('.main_breadcrumb_text').html(
            `<a href="/wip?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                WIP Dashboard
            </a>`
        );
    });

    // Cutting Dashboard
    $('.cutting_item').on('click', function() {
        $('.main_breadcrumb_text').html(
            `<a href="/cutting_output?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Cutting Dashboard
            </a>`
        );
    });

    // Labour Dashboard
    $('.labour_item').on('click', function() {
        $('.main_breadcrumb_text').html(
            `<a href="/direct_labour?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Labour Dashboard
            </a>`
        );
    });

    if (portal_type === 'Line Stop') {
        $('.portal_type_link').html(
            `<a href="/linestop_analysis?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Line Stop Dashboard
            </a>`
        );
    } else if (portal_type === 'WIP') {
        $('.portal_type_link').html(
            `<a href="/wip?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                WIP Dashboard
            </a>`
        );
    } else if (portal_type === 'Cutting') {
        $('.portal_type_link').html(
            `<a href="/cutting_output?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Cutting Dashboard
            </a>`
        );
    } else if (portal_type === 'Labour') {
        $('.portal_type_link').html(
            `<a href="/directlabour?fromdate=${fromdate}&enddate=${enddate}" class="text-blue-600 hover:underline">
                Labour Dashboard
            </a>`
        );
    } else {
        
    }
    
});

</script>
</body>
</html>