<!DOCTYPE html>
<html lang="en" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Production Socks Overview</title>
    
    <!-- Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    
    <!-- jQuery CDN -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    
    <!-- Chart.js CDN for sparklines -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <!-- Lucide Icons CDN -->
    <script src="https://unpkg.com/lucide@latest/dist/umd/lucide.js"></script>

    <!-- Date picker CDN -->
    <link rel="stylesheet" type="text/css" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" />
    <script type="text/javascript" src="https://cdn.jsdelivr.net/jquery/latest/jquery.min.js"></script>
    <script type="text/javascript" src="https://cdn.jsdelivr.net/momentjs/latest/moment.min.js"></script>
    <script type="text/javascript" src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script>

    
    <!-- Custom Tailwind Configuration -->
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'background': 'rgb(245, 246, 247)',
                        'foreground': 'rgb(31, 41, 55)',
                        'card': 'rgb(255, 255, 255)',
                        'card-foreground': 'rgb(31, 41, 55)',
                        'primary': 'rgb(15, 61, 126)',
                        'primary-foreground': 'rgb(248, 250, 252)',
                        'primary-100': 'rgb(219, 234, 254)',
                        'primary-50': 'rgb(239, 246, 255)',
                        'primary-25': 'rgb(248, 250, 252)',
                        'secondary': 'rgb(241, 245, 249)',
                        'muted': 'rgb(241, 245, 249)',
                        'muted-foreground': 'rgb(100, 116, 139)',
                        'border': 'rgb(226, 232, 240)',
                        'input': 'rgb(226, 232, 240)',
                        'ring': 'rgb(15, 61, 126)',
                    },
                    fontFamily: {
                        'body': ['Inter', 'system-ui', 'sans-serif'],
                    }
                }
            }
        }
    </script>
    
    <style>
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Oxygen', 'Ubuntu', 'Cantarell', 'Fira Sans', 'Droid Sans', 'Helvetica Neue', sans-serif;
        }
        
        /* Custom scrollbar */
        .custom-scrollbar::-webkit-scrollbar {
            width: 6px;
        }
        .custom-scrollbar::-webkit-scrollbar-track {
            background: #f1f5f9;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 3px;
        }
        
        /* Sparkline container */
        .sparkline-container {
            width: 80px;
            height: 40px;
        }
        
        /* Tooltip styles */
        .tooltip {
            position: absolute;
            z-index: 1000;
            background: rgb(15, 23, 42);
            color: white;
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 12px;
            pointer-events: none;
            opacity: 0;
            transition: opacity 0.2s;
        }
        
        .tooltip.show {
            opacity: 1;
        }

        #tooltip {
            position: absolute;
            z-index: 9999;
            background: #fff;
            color: #333;
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
            white-space: nowrap;
            display: none;
        }

        #tooltip.show {
            display: block;
        }
        
        /* Dropdown styles */
        .dropdown-menu {
            display: none;
            position: absolute;
            top: 100%;
            right: 0;
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 6px;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            z-index: 50;
            min-width: 200px;
            max-height: 240px;
            overflow-y: auto;
        }
        
        .dropdown-menu.show {
            display: block;
        }
        
        /* Animation for row highlighting */
        .highlight-row {
            background-color: rgb(219, 234, 254) !important;
            transition: background-color 0.5s ease;
        }
        
        /* Date picker custom styles */
        .date-picker input {
            background: rgba(248, 250, 252, 0.1);
            border: 1px solid rgba(248, 250, 252, 0.2);
            color: white;
        }
        
        .date-picker input::placeholder {
            color: rgba(248, 250, 252, 0.7);
        }
        
        /* Loading skeleton animation */
        @keyframes skeleton-loading {
            0% { background-color: #e2e8f0; }
            50% { background-color: #f1f5f9; }
            100% { background-color: #e2e8f0; }
        }
        
        .skeleton {
            animation: skeleton-loading 2s ease-in-out infinite;
            border-radius: 4px;
        }
        
        /* Status indicators */
        .status-dot-red {
            width: 8px;
            height: 8px;
            background-color: #ef4444;
            border-radius: 50%;
        }
        
        .status-dot-yellow {
            width: 8px;
            height: 8px;
            background-color: #facc15;
            border-radius: 50%;
        }
    </style>
</head>
<body class="h-full bg-background text-foreground antialiased">
    <!-- Tooltip container -->
    <div id="tooltip" class="tooltip"></div>
    
    <div class="flex min-h-screen w-full flex-col">
        <!-- Dashboard Header -->
        <header class="sticky top-0 z-30 flex h-16 items-center justify-between gap-4 border-b border-border bg-primary px-4 shadow-sm sm:px-6">
            <div class="flex items-center space-x-3 my-2">
                <!-- Back button -->
                <a href="/dashboard" 
                    class="inline-flex items-center justify-center rounded-xl border border-white-300 px-3 py-2 text-sm font-medium text-white-700 hover:bg-gray-25 transition">
                    <i data-lucide="house" class="w-5 h-5 text-white"></i>
                </a>

                <!-- Page Title -->
                <h1 class="text-2xl font-semibold tracking-tight text-primary-foreground">
                    Production Socks Overview
                </h1>
            </div>
            <div class="flex items-center gap-4">
                <!-- Historical Data Button -->
                <button class="inline-flex items-center justify-center rounded-md text-sm font-medium bg-primary-foreground/10 text-white border border-primary-foreground/20 hover:bg-primary-foreground/20 hover:text-white h-10 px-4 py-2 transition-colors">
                    <i data-lucide="history" class="mr-2 h-4 w-4"></i>
                    Historical Data
                </button>
                
                <!-- Date Range Picker -->
                <div class="date-picker">
                    <input type="text" placeholder="Select date range" readonly class="h-10 px-3 py-2 rounded-md text-sm cursor-pointer transition-colors" id="dateRangePicker">
                </div>
                
                <!-- Notifications Button -->
                <button class="inline-flex items-center justify-center rounded-md bg-primary-foreground/10 text-white border border-primary-foreground/20 hover:bg-primary-foreground/20 hover:text-white h-10 w-10 transition-colors" id="notificationsBtn">
                    <i data-lucide="bell" class="h-5 w-5"></i>
                </button>
            </div>
        </header>

        <!-- Main Content -->
        <main class="flex flex-1 flex-col gap-4 p-4 md:gap-8 md:p-8">
            <!-- KPI Header -->
            <div class="grid gap-4 md:grid-cols-2 lg:grid-cols-4 xl:grid-cols-5" id="kpiGrid">
                <!-- KPI cards will be dynamically generated -->
            </div>

            <!-- Drilldown Table -->
            <div class="grid gap-4 md:gap-8 grid-cols-1">
                <div class="bg-card border border-border rounded-lg shadow-sm">
                    <!-- Card Header -->
                    <div class="flex flex-row items-start justify-between p-6 border-b border-border">
                        <div>
                            <h3 class="text-2xl font-semibold leading-none tracking-tight">Daily snapshot</h3>
                            <p class="text-sm text-muted-foreground mt-1.5">
                                Aggregated production data. Click on a unit or zone to expand.
                            </p>
                        </div>
                        <div class="flex items-center gap-2">
                            <!-- Filter Button -->
                            <div class="relative">
                                <button class="inline-flex items-center justify-center rounded-md border border-input bg-background hover:bg-muted h-10 px-4 py-2 text-sm font-medium transition-colors" id="filterBtn">
                                    <i data-lucide="filter" class="mr-2 h-4 w-4"></i>
                                    Filters
                                    <span class="ml-2 bg-secondary text-secondary-foreground rounded-full h-5 w-5 text-xs flex items-center justify-center hidden" id="filterBadge">0</span>
                                </button>
                                
                                <!-- Filter Dropdown -->
                                <div class="dropdown-menu custom-scrollbar" id="filterDropdown">
                                    <div class="p-4">
                                        <!-- <div class="space-y-2 mb-4">
                                            <h4 class="font-medium leading-none">Filters</h4>
                                            <p class="text-sm text-muted-foreground">
                                                Refine the production data below.
                                            </p>
                                        </div> -->
                                        
                                        <div class="mb-4 relative">
                                            <!-- <label class="text-sm font-medium mb-2 block">Buyer</label> -->
                                            <div class="custom-dropdown" id="buyerDropdown">
                                                <button type="button" 
                                                class="w-full border border-input rounded-md p-2 text-sm flex justify-between items-center bg-white"
                                                id="buyerDropdownBtn">
                                                <span id="buyerSelected" style="font-size: 10px;">SELECT BUYER</span>
                                                <i data-lucide="chevron-down" class="h-4 w-4"></i>
                                                </button>
                                                <ul class="absolute hidden bg-white border border-input rounded-md mt-1 w-full max-h-40 overflow-y-auto z-10" 
                                                    id="buyerDropdownMenu">
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="Nike">Nike</li>
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="Adidas">Adidas</li>
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="Puma">Puma</li>
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="Under Armour">Under Armour</li>
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="H&M">H&M</li>
                                                <li class="px-3 py-2 hover:bg-muted cursor-pointer" data-value="Zara">Zara</li>
                                                </ul>
                                            </div>
                                        </div>

                                        <div class="mb-4 relative">
                                            <!-- <label class="text-sm font-medium mb-2 block">OCN</label> -->
                                            <div class="custom-dropdown" id="ocnDropdown">
                                                <button type="button" 
                                                class="w-full border border-input rounded-md p-2 text-sm flex justify-between items-center bg-white"
                                                id="ocnDropdownBtn">
                                                <span id="ocnSelected" style="font-size: 11px;">SELECT OCN</span>
                                                <i data-lucide="chevron-down" class="h-4 w-4"></i>
                                                </button>
                                                <ul class="absolute hidden bg-white border border-input rounded-md mt-1 w-full max-h-40 overflow-y-auto z-10" 
                                                    id="ocnDropdownMenu">
                                                <li>Select a buyer first</li>
                                                </ul>
                                            </div>
                                        </div>
                                        
                                        <!-- Reset Button -->
                                        <button class="w-full bg-secondary hover:bg-secondary/80 text-sm py-2 px-4 rounded-md transition-colors" id="resetFiltersBtn">
                                            <i data-lucide="x" class="mr-2 h-4 w-4 inline"></i>
                                            Reset All Filters
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Export Button -->
                            <button class="inline-flex items-center justify-center rounded-md border border-input bg-background hover:bg-muted h-10 w-10 transition-colors" title="Export data">
                                <i data-lucide="upload" class="h-4 w-4"></i>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Table Content -->
                    <div class="p-6">
                        <div class="relative w-full overflow-auto">
                            <table class="w-full caption-bottom text-sm">
                                <thead>
                                    <tr class="border-b border-border">
                                        <th class="h-12 px-4 text-left align-middle font-medium text-muted-foreground w-[350px]">Name</th>
                                        <th class="h-12 px-4 text-left align-middle font-medium text-muted-foreground">OCN</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Running days</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">SAM</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Tailors</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Direct Labour</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Line WIP</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Output</th>
                                        <th class="h-12 px-4 text-center align-middle font-medium text-muted-foreground w-[120px]">Output Trend</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Earned Minutes</th>
                                        <th class="h-12 px-4 text-right align-middle font-medium text-muted-foreground">Efficiency</th>
                                        <th class="h-12 px-4 text-center align-middle font-medium text-muted-foreground w-[120px]">Efficiency Trend</th>
                                    </tr>
                                </thead>
                                <tbody id="drilldownTableBody">
                                    <!-- Table rows will be dynamically generated -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Notification Side Panel (hidden by default) -->
    <div class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden" id="notificationOverlay">
        <div class="fixed right-0 top-0 h-full w-80 bg-white shadow-lg transform translate-x-full transition-transform duration-300" id="notificationPanel">
            <div class="p-6 border-b border-border">
                <div class="flex items-center justify-between">
                    <h2 class="text-lg font-semibold">Production Alerts</h2>
                    <button class="text-muted-foreground hover:text-foreground" id="closeNotifications">
                        <i data-lucide="x" class="h-5 w-5"></i>
                    </button>
                </div>
            </div>
            <div class="p-6">
                <p class="text-muted-foreground">No new alerts at this time.</p>
            </div>
        </div>
    </div>

    <script>
        // Mock Data
        const mockKpiData = {
            output: {
                label: "Output",
                value: 12547,
                yesterday: 11823,
                target: 13000,
                unit: "",
                description: "Total pieces produced today",
                trendData: [
                    { value: 11200 }, { value: 11400 }, { value: 11600 }, 
                    { value: 11823 }, { value: 12100 }, { value: 12300 }, { value: 12547 }
                ]
            },
            efficiency: {
                label: "Efficiency",
                value: 87.3,
                yesterday: 85.1,
                target: 90.0,
                unit: "%",
                description: "Overall production efficiency",
                trendData: [
                    { value: 83.2 }, { value: 84.1 }, { value: 84.8 }, 
                    { value: 85.1 }, { value: 86.2 }, { value: 86.9 }, { value: 87.3 }
                ]
            },
            wip: {
                label: "Line WIP",
                value: 2847,
                yesterday: 3012,
                target: 2500,
                unit: "",
                description: "Work in progress on production lines",
                trendData: [
                    { value: 3200 }, { value: 3150 }, { value: 3080 }, 
                    { value: 3012 }, { value: 2950 }, { value: 2890 }, { value: 2847 }
                ]
            },
            lineStopMins: {
                label: "Line Stop Minutes",
                value: 184,
                yesterday: 203,
                target: 150,
                unit: "",
                description: "Total minutes lost due to line stops",
                trendData: [
                    { value: 245 }, { value: 230 }, { value: 218 }, 
                    { value: 203 }, { value: 195 }, { value: 189 }, { value: 184 }
                ]
            },
            directLabourStrength: {
                label: "Direct Labour",
                value: 1247,
                yesterday: 1238,
                target: 1250,
                unit: "",
                description: "Number of direct labour workers present",
                trendData: [
                    { value: 1235 }, { value: 1240 }, { value: 1242 }, 
                    { value: 1238 }, { value: 1245 }, { value: 1246 }, { value: 1247 }
                ]
            },
            packingOutput: {
                label: "Packing Output",
                value: 11847,
                yesterday: 11234,
                target: 12500,
                unit: "",
                description: "Total pieces packed today",
                trendData: [
                    { value: 10800 }, { value: 11000 }, { value: 11150 }, 
                    { value: 11234 }, { value: 11450 }, { value: 11650 }, { value: 11847 }
                ]
            },
            smStock: {
                label: "SM Stock",
                value: 18429,
                yesterday: 17856,
                target: 20000,
                unit: "",
                description: "Supermarket stock levels",
                trendData: [
                    { value: 17200 }, { value: 17400 }, { value: 17650 }, 
                    { value: 17856 }, { value: 18100 }, { value: 18280 }, { value: 18429 }
                ]
            },
            capacityUtilization: {
                label: "Capacity Utilization",
                value: 91.4,
                yesterday: 89.2,
                target: 95.0,
                unit: "%",
                description: "Current capacity utilization percentage",
                trendData: [
                    { value: 87.5 }, { value: 88.1 }, { value: 88.8 }, 
                    { value: 89.2 }, { value: 90.1 }, { value: 90.8 }, { value: 91.4 }
                ]
            },
            ordersOnTrack: {
                label: "Orders On Track",
                value: 94.2,
                yesterday: 92.8,
                target: 98.0,
                unit: "%",
                description: "Percentage of orders on schedule",
                trendData: [
                    { value: 91.2 }, { value: 91.8 }, { value: 92.1 }, 
                    { value: 92.8 }, { value: 93.4 }, { value: 93.8 }, { value: 94.2 }
                ]
            },
            cuttingOutput: {
                label: "Cutting Output",
                value: 13247,
                yesterday: 12834,
                target: 14000,
                unit: "",
                description: "Total pieces cut today",
                trendData: [
                    { value: 12200 }, { value: 12400 }, { value: 12650 }, 
                    { value: 12834 }, { value: 13000 }, { value: 13120 }, { value: 13247 }
                ]
            }
        };

        const mockDrilldownData = [
            {
                unitId: "unit1",
                name: "Unit A",
                type: "unit",
                sam: 2.45,
                tailorStrength: 286,
                tailorStrengthTarget: 280,
                labourStrength: 342,
                labourStrengthTarget: 340,
                wip: 847,
                output: 3247,
                target: 3200,
                earnedMinutes: 7954,
                efficiency: 89.2,
                outputTrend: [
                    { value: 3100 }, { value: 3150 }, { value: 3180 }, 
                    { value: 3200 }, { value: 3220 }, { value: 3235 }, { value: 3247 }
                ],
                efficiencyTrend: [
                    { value: 87.2 }, { value: 87.8 }, { value: 88.1 }, 
                    { value: 88.5 }, { value: 88.9 }, { value: 89.0 }, { value: 89.2 }
                ],
                totalInputBlocks: 2,
                totalLineStops: 3,
                newLinesCount: 1,
                zones: [
                    {
                        zoneId: "zone1",
                        name: "Zone A1",
                        type: "zone",
                        sam: 2.50,
                        tailorStrength: 143,
                        tailorStrengthTarget: 140,
                        labourStrength: 171,
                        labourStrengthTarget: 170,
                        wip: 423,
                        output: 1623,
                        target: 1600,
                        earnedMinutes: 4058,
                        efficiency: 88.9,
                        outputTrend: [
                            { value: 1580 }, { value: 1590 }, { value: 1600 }, 
                            { value: 1610 }, { value: 1615 }, { value: 1620 }, { value: 1623 }
                        ],
                        efficiencyTrend: [
                            { value: 87.5 }, { value: 88.0 }, { value: 88.2 }, 
                            { value: 88.5 }, { value: 88.7 }, { value: 88.8 }, { value: 88.9 }
                        ],
                        totalInputBlocks: 1,
                        totalLineStops: 2,
                        lines: [
                            {
                                lineId: "line1",
                                name: "Line A1-01",
                                type: "line",
                                ocn: "OCN001",
                                runningDays: 12,
                                sam: 2.48,
                                tailorStrength: 48,
                                tailorStrengthTarget: 46,
                                labourStrength: 57,
                                labourStrengthTarget: 56,
                                wip: 142,
                                output: 541,
                                target: 530,
                                earnedMinutes: 1342,
                                efficiency: 89.5,
                                outputTrend: [
                                    { value: 520 }, { value: 525 }, { value: 530 }, 
                                    { value: 535 }, { value: 538 }, { value: 540 }, { value: 541 }
                                ],
                                efficiencyTrend: [
                                    { value: 88.2 }, { value: 88.5 }, { value: 88.8 }, 
                                    { value: 89.0 }, { value: 89.2 }, { value: 89.3 }, { value: 89.5 }
                                ],
                                isInputBlocked: false,
                                hasLineStopToday: true,
                                remark: {
                                    author: "Supervisor John",
                                    text: "Good progress today, maintaining steady output."
                                }
                            },
                            {
                                lineId: "line2",
                                name: "Line A1-02",
                                type: "line",
                                ocn: "OCN002",
                                runningDays: 8,
                                sam: 2.52,
                                tailorStrength: 47,
                                tailorStrengthTarget: 47,
                                labourStrength: 57,
                                labourStrengthTarget: 57,
                                wip: 281,
                                output: 512,
                                target: 520,
                                earnedMinutes: 1290,
                                efficiency: 88.1,
                                outputTrend: [
                                    { value: 500 }, { value: 505 }, { value: 508 }, 
                                    { value: 510 }, { value: 511 }, { value: 511 }, { value: 512 }
                                ],
                                efficiencyTrend: [
                                    { value: 86.8 }, { value: 87.2 }, { value: 87.5 }, 
                                    { value: 87.8 }, { value: 88.0 }, { value: 88.1 }, { value: 88.1 }
                                ],
                                isInputBlocked: true,
                                hasLineStopToday: false
                            }
                        ]
                    },
                    {
                        zoneId: "zone2",
                        name: "Zone A2",
                        type: "zone",
                        sam: 2.40,
                        tailorStrength: 143,
                        tailorStrengthTarget: 140,
                        labourStrength: 171,
                        labourStrengthTarget: 170,
                        wip: 424,
                        output: 1624,
                        target: 1600,
                        earnedMinutes: 3896,
                        efficiency: 89.5,
                        outputTrend: [
                            { value: 1590 }, { value: 1600 }, { value: 1610 }, 
                            { value: 1615 }, { value: 1620 }, { value: 1622 }, { value: 1624 }
                        ],
                        efficiencyTrend: [
                            { value: 88.2 }, { value: 88.8 }, { value: 89.0 }, 
                            { value: 89.2 }, { value: 89.3 }, { value: 89.4 }, { value: 89.5 }
                        ],
                        totalInputBlocks: 1,
                        totalLineStops: 1,
                        lines: [
                            {
                                lineId: "line3",
                                name: "Line A2-01",
                                type: "line",
                                ocn: "OCN003",
                                runningDays: 15,
                                sam: 2.38,
                                tailorStrength: 48,
                                tailorStrengthTarget: 47,
                                labourStrength: 57,
                                labourStrengthTarget: 57,
                                wip: 212,
                                output: 543,
                                target: 535,
                                earnedMinutes: 1292,
                                efficiency: 90.1,
                                outputTrend: [
                                    { value: 530 }, { value: 535 }, { value: 538 }, 
                                    { value: 540 }, { value: 541 }, { value: 542 }, { value: 543 }
                                ],
                                efficiencyTrend: [
                                    { value: 89.2 }, { value: 89.5 }, { value: 89.7 }, 
                                    { value: 89.8 }, { value: 89.9 }, { value: 90.0 }, { value: 90.1 }
                                ],
                                isInputBlocked: false,
                                hasLineStopToday: false
                            }
                        ]
                    }
                ]
            },
            {
                unitId: "unit2",
                name: "Unit B",
                type: "unit",
                sam: 2.55,
                tailorStrength: 278,
                tailorStrengthTarget: 280,
                labourStrength: 335,
                labourStrengthTarget: 340,
                wip: 752,
                output: 3124,
                target: 3150,
                earnedMinutes: 7966,
                efficiency: 87.8,
                outputTrend: [
                    { value: 3080 }, { value: 3090 }, { value: 3100 }, 
                    { value: 3110 }, { value: 3115 }, { value: 3120 }, { value: 3124 }
                ],
                efficiencyTrend: [
                    { value: 86.5 }, { value: 87.0 }, { value: 87.2 }, 
                    { value: 87.4 }, { value: 87.6 }, { value: 87.7 }, { value: 87.8 }
                ],
                totalInputBlocks: 1,
                totalLineStops: 2,
                newLinesCount: 0,
                zones: [
                    {
                        zoneId: "zone3",
                        name: "Zone B1",
                        type: "zone",
                        sam: 2.58,
                        tailorStrength: 139,
                        tailorStrengthTarget: 140,
                        labourStrength: 168,
                        labourStrengthTarget: 170,
                        wip: 376,
                        output: 1562,
                        target: 1575,
                        earnedMinutes: 4030,
                        efficiency: 87.2,
                        outputTrend: [
                            { value: 1550 }, { value: 1555 }, { value: 1558 }, 
                            { value: 1560 }, { value: 1561 }, { value: 1561 }, { value: 1562 }
                        ],
                        efficiencyTrend: [
                            { value: 86.8 }, { value: 87.0 }, { value: 87.1 }, 
                            { value: 87.1 }, { value: 87.2 }, { value: 87.2 }, { value: 87.2 }
                        ],
                        totalInputBlocks: 0,
                        totalLineStops: 1,
                        lines: [
                            {
                                lineId: "line4",
                                name: "Line B1-01",
                                type: "line",
                                ocn: "OCN004",
                                runningDays: 9,
                                sam: 2.56,
                                tailorStrength: 46,
                                tailorStrengthTarget: 47,
                                labourStrength: 56,
                                labourStrengthTarget: 57,
                                wip: 188,
                                output: 521,
                                target: 525,
                                earnedMinutes: 1334,
                                efficiency: 86.9,
                                outputTrend: [
                                    { value: 515 }, { value: 518 }, { value: 519 }, 
                                    { value: 520 }, { value: 520 }, { value: 521 }, { value: 521 }
                                ],
                                efficiencyTrend: [
                                    { value: 86.2 }, { value: 86.5 }, { value: 86.6 }, 
                                    { value: 86.7 }, { value: 86.8 }, { value: 86.8 }, { value: 86.9 }
                                ],
                                isInputBlocked: false,
                                hasLineStopToday: true
                            }
                        ]
                    }
                ]
            }
        ];

        // Chart utilities
        function createSparkline(containerId, data, color = '#3b82f6') {
            const canvas = document.getElementById(containerId);
            if (!canvas) return;
            
            const ctx = canvas.getContext('2d');
            
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.map((_, i) => i),
                    datasets: [{
                        data: data.map(d => d.value),
                        borderColor: color,
                        backgroundColor: color + '20',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        pointHoverRadius: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: { display: false },
                        y: { display: false }
                    },
                    plugins: {
                        legend: { display: false },
                        tooltip: { enabled: false }
                    },
                    interaction: {
                        intersect: false
                    },
                    elements: {
                        point: { radius: 0 }
                    }
                }
            });
        }

        // Initialize KPI Cards
        function initializeKpiCards() {
            const kpiGrid = $('#kpiGrid');
            kpiGrid.empty();
            
            Object.entries(mockKpiData).forEach(([key, kpi]) => {
                const diff = kpi.value - kpi.yesterday;
                const diffPercentage = kpi.yesterday === 0 ? 0 : (diff / kpi.yesterday) * 100;
                const isNeutral = key === 'directLabourStrength';
                const isGood = ['output', 'efficiency', 'packingOutput', 'smStock', 'capacityUtilization', 'ordersOnTrack', 'cuttingOutput'].includes(key) ? diff >= 0 : diff <= 0;
                
                const comparisonColor = isNeutral ? 'text-gray-500' : (isGood ? 'text-green-500' : 'text-red-500');
                const arrowIcon = isNeutral ? 'minus' : (isGood ? 'arrow-up' : 'arrow-down');
                
                const formatValue = (value) => {
                    if (key === 'lineStopMins') {
                        return Math.round(value).toLocaleString();
                    }
                    if (kpi.unit === '%') {
                        return `${value.toFixed(1)}${kpi.unit}`;
                    }
                    return value.toLocaleString();
                };
                
                const trend = kpi.trendData;
                const isIncreasing = trend.length > 1 && trend[trend.length - 1].value > trend[0].value;
                const chartColor = isIncreasing ? '#22c55e' : (trend[trend.length - 1].value < trend[0].value ? '#ef4444' : '#3b82f6');
                
                const cardHtml = `
                    <div class="bg-card border border-border rounded-lg shadow-sm flex-1 ${['lineStopMins', 'wip', 'smStock', 'directLabourStrength', 'packingOutput', 'ordersOnTrack', 'cuttingOutput'].includes(key) ? 'cursor-pointer hover:bg-muted transition-colors' : ''}">
                        <div class="p-4">
                            <div class="flex flex-col md:flex-row md:items-stretch h-full">
                                <div class="flex-1 flex flex-col justify-between">
                                    <div>
                                        <div class="flex items-center gap-1.5">
                                            <p class="text-sm font-medium text-muted-foreground">${kpi.label}</p>
                                            <i data-lucide="info" class="h-3 w-3 text-muted-foreground cursor-help" title="${kpi.description}"></i>
                                        </div>
                                        <p class="text-2xl font-bold">${formatValue(kpi.value)}</p>
                                    </div>
                                    <div>
                                        <div class="flex items-center text-xs text-muted-foreground">
                                            <i data-lucide="${arrowIcon}" class="h-4 w-4 ${comparisonColor}"></i>
                                            <span class="ml-1 ${comparisonColor}">${Math.abs(diffPercentage).toFixed(1)}%</span>
                                            <span class="ml-1">vs yesterday</span>
                                        </div>
                                        <p class="text-xs text-muted-foreground mt-1">
                                            Target: ${formatValue(kpi.target)}
                                        </p>
                                    </div>
                                </div>
                                <div class="w-full h-12 md:w-24 md:h-auto mt-2 md:mt-0 md:ml-4">
                                    <canvas id="chart-${key}" class="sparkline-container w-full h-full"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                
                kpiGrid.append(cardHtml);
                
                // Create sparkline chart after DOM is updated
                setTimeout(() => createSparkline(`chart-${key}`, kpi.trendData, chartColor), 100);
            });
            
            // Initialize Lucide icons
            lucide.createIcons();
        }

        // Drilldown table functionality
        let expandedRows = {};
        let filteredData = [...mockDrilldownData];
        let selectedBuyers = [];
        let selectedOcns = [];
        let buyerOcnData = [];
        let selectedBuyer = null;
        let selectedOcn = null;

        function buildTableRows() {
            const rows = [];
            
            function addRows(items, level = 0, parentUnit = null, parentZone = null) {
                items.forEach(item => {
                    rows.push({ item, level, parentUnit, parentZone });
                    const itemId = item.unitId || item.zoneId || item.lineId;
                    
                    if (expandedRows[itemId]) {
                        if (item.zones) {
                            addRows(item.zones, level + 1, item);
                        } else if (item.lines) {
                            addRows(item.lines, level + 1, parentUnit, item);
                        }
                    }
                });
            }
            
            addRows(filteredData);
            return rows;
        }

        function renderTable() {
            const tbody = $('#drilldownTableBody');
            tbody.empty();
            
            const rows = buildTableRows();
            
            if (rows.length === 0) {
                tbody.append(`
                    <tr>
                        <td colspan="12" class="h-24 text-center text-muted-foreground">
                            No data available for the current selection.
                        </td>
                    </tr>
                `);
                return;
            }
            
            rows.forEach(({ item, level, parentUnit, parentZone }) => {
                const rowId = item.unitId || item.zoneId || item.lineId;
                const isExpandable = item.zones || item.lines;
                const isExpanded = expandedRows[rowId];
                const isLine = !isExpandable;
                
                // Calculate differences and colors
                const tailorDiff = item.tailorStrength - item.tailorStrengthTarget;
                const isTailorGood = tailorDiff >= 0;
                const labourDiff = item.labourStrength - item.labourStrengthTarget;
                const isLabourGood = labourDiff >= 0;
                const outputDiff = item.output - item.target;
                const isOutputGood = outputDiff >= 0;
                const earnedMinutesDiff = item.earnedMinutes - (item.target * (item.sam || 2.5));
                const isEarnedMinutesGood = earnedMinutesDiff >= 0;
                
                const rowClass = level === 0 ? 'bg-primary-50' : (level === 1 ? 'bg-muted/50' : '');
                
                const rowHtml = `
                    <tr class="border-b border-border ${rowClass}" data-row-id="${rowId}">
                        <td class="h-12 px-4 align-middle" style="padding-left: ${level * 20 + 16}px">
                            <div class="flex items-center gap-2">
                                ${isExpandable ? 
                                    `<button class="inline-flex items-center justify-center h-6 w-6 rounded hover:bg-muted transition-colors expand-btn" data-row-id="${rowId}">
                                        <i data-lucide="${isExpanded ? 'chevron-down' : 'chevron-right'}" class="h-4 w-4"></i>
                                    </button>` :
                                    `<div class="w-6 h-6 flex items-center justify-center">
                                        ${item.isInputBlocked ? '<div class="status-dot-red cursor-pointer" title="Input Blocked - High WIP"></div>' : ''}
                                        ${item.hasLineStopToday && !item.isInputBlocked ? '<div class="status-dot-yellow cursor-pointer" title="Line Stop Today"></div>' : ''}
                                    </div>`
                                }
                                <span class="font-medium">${item.name}</span>
                                ${item.remark ? 
                                    `<i data-lucide="message-square-text" class="h-4 w-4 text-muted-foreground cursor-pointer hover:text-primary transition-colors" title="${item.remark.author} says: ${item.remark.text}"></i>` : ''
                                }
                                ${item.newLinesCount > 0 ? 
                                    `<span class="bg-blue-100 text-blue-800 text-xs font-semibold px-2 py-0.5 rounded-full">${item.newLinesCount} New</span>` : ''
                                }
                                ${item.totalInputBlocks > 0 ? 
                                    `<div class="flex items-center gap-1.5 bg-red-100 text-red-700 px-2 py-0.5 rounded-full text-xs font-semibold cursor-pointer" title="Lines blocked - High WIP">
                                        <div class="status-dot-red"></div>
                                        ${item.totalInputBlocks} WIP
                                    </div>` : ''
                                }
                                ${item.totalLineStops > 0 ? 
                                    `<div class="flex items-center gap-1.5 bg-yellow-100 text-yellow-700 px-2 py-0.5 rounded-full text-xs font-semibold cursor-pointer" title="Total Line Stops">
                                        <div class="status-dot-yellow"></div>
                                        ${item.totalLineStops} ${item.totalLineStops === 1 ? 'Stop' : 'Stops'}
                                    </div>` : ''
                                }
                            </div>
                        </td>
                        <td class="h-12 px-4 align-middle text-left">${isLine ? item.ocn || '' : ''}</td>
                        <td class="h-12 px-4 align-middle text-right tabular-nums">${isLine ? item.runningDays || '' : ''}</td>
                        <td class="h-12 px-4 align-middle text-right tabular-nums">${item.sam ? item.sam.toFixed(2) : ''}</td>
                        <td class="h-12 px-4 align-middle text-right">
                            <div class="flex items-center justify-end gap-1.5 tabular-nums">
                                <span class="font-semibold">${item.tailorStrength.toLocaleString()}</span>
                                <div class="flex items-center text-xs ${isTailorGood ? 'text-green-600' : 'text-red-600'}">
                                    <i data-lucide="${isTailorGood ? 'arrow-up' : 'arrow-down'}" class="h-3 w-3"></i>
                                    <span>${Math.abs(tailorDiff)}</span>
                                </div>
                            </div>
                        </td>
                        <td class="h-12 px-4 align-middle text-right">
                            <div class="flex items-center justify-end gap-1.5 tabular-nums">
                                <span class="font-semibold">${item.labourStrength.toLocaleString()}</span>
                                <div class="flex items-center text-xs ${isLabourGood ? 'text-green-600' : 'text-red-600'}">
                                    <i data-lucide="${isLabourGood ? 'arrow-up' : 'arrow-down'}" class="h-3 w-3"></i>
                                    <span>${Math.abs(labourDiff)}</span>
                                </div>
                            </div>
                        </td>
                        <td class="h-12 px-4 align-middle text-right tabular-nums">${item.wip.toLocaleString()}</td>
                        <td class="h-12 px-4 align-middle text-right">
                            <div class="flex items-center justify-end gap-1.5 tabular-nums">
                                <span class="font-semibold">${item.output.toLocaleString()}</span>
                                <div class="flex items-center text-xs ${isOutputGood ? 'text-green-600' : 'text-red-600'}">
                                    <i data-lucide="${isOutputGood ? 'arrow-up' : 'arrow-down'}" class="h-3 w-3"></i>
                                    <span>${Math.abs(outputDiff)}</span>
                                </div>
                            </div>
                        </td>
                        <td class="h-12 px-4 align-middle text-center">
                            <canvas id="output-trend-${rowId}" class="sparkline-container mx-auto"></canvas>
                        </td>
                        <td class="h-12 px-4 align-middle text-right">
                            <div class="flex items-center justify-end gap-1.5 tabular-nums">
                                <span class="font-semibold">${item.earnedMinutes.toLocaleString()}</span>
                                <div class="flex items-center text-xs ${isEarnedMinutesGood ? 'text-green-600' : 'text-red-600'}">
                                    <i data-lucide="${isEarnedMinutesGood ? 'arrow-up' : 'arrow-down'}" class="h-3 w-3"></i>
                                    <span>${Math.abs(Math.round(earnedMinutesDiff))}</span>
                                </div>
                            </div>
                        </td>
                        <td class="h-12 px-4 align-middle text-right tabular-nums">
                            <span class="font-semibold">${item.efficiency.toFixed(1)}%</span>
                        </td>
                        <td class="h-12 px-4 align-middle text-center">
                            <canvas id="efficiency-trend-${rowId}" class="sparkline-container mx-auto"></canvas>
                        </td>
                    </tr>
                `;
                
                tbody.append(rowHtml);
                
                // Create sparkline charts
                setTimeout(() => {
                    const outputTrend = item.outputTrend || [];
                    const efficiencyTrend = item.efficiencyTrend || [];
                    
                    if (outputTrend.length > 0) {
                        createSparkline(`output-trend-${rowId}`, outputTrend, '#3b82f6');
                    }
                    if (efficiencyTrend.length > 0) {
                        createSparkline(`efficiency-trend-${rowId}`, efficiencyTrend, '#10b981');
                    }
                }, 100);
            });
            
            // Reinitialize Lucide icons
            lucide.createIcons();
        }

        function applyFilters() {
            filteredData = mockDrilldownData.filter(unit => {
                // For simplicity, we'll filter by a mock buyer assignment
                // In real implementation, this would check actual data relationships
                return selectedBuyers.length === 0 || selectedBuyers.includes('Nike') || selectedBuyers.includes('Adidas');
            });
            
            updateFilterBadge();
            renderTable();
        }

        function updateFilterBadge() {
            const filterBadge = $('#filterBadge');
            const activeFilters = selectedBuyers.length + selectedOcns.length;
            
            if (activeFilters > 0) {
                filterBadge.text(activeFilters).removeClass('hidden');
            } else {
                filterBadge.addClass('hidden');
            }
        }

        function fetchBuyers() {
            // Fetch data from API with AJAX GET
            return $.ajax({
                url: '/buyer_ocn_details',  
                type: 'GET',
                dataType: 'json',
                success: function (response) {
                    buyerOcnData = response.data;

                    // Get unique buyers
                    const uniqueBuyers = [...new Set(buyerOcnData.map(item => item.Buyer))];

                    // Populate Buyer dropdown
                    const buyerMenu = $('#buyerDropdownMenu');
                    buyerMenu.empty();
                    uniqueBuyers.forEach(buyer => {
                        buyerMenu.append(
                            `<li class="px-3 py-2 hover:bg-muted cursor-pointer" style="font-size: 10px;" data-value="${buyer}">${buyer}</li>`
                        );
                    });
                },
                error: function (xhr, status, error) {
                    console.error("Error fetching buyer_ocn_details:", error);
                }
            });
        }

        // Event handlers
        function initializeEventHandlers() {
            // Expand/collapse table rows
            $(document).on('click', '.expand-btn', function() {
                const rowId = $(this).data('row-id');
                expandedRows[rowId] = !expandedRows[rowId];
                renderTable();
            });

            // Filter dropdown
            $('#filterBtn').click(function(e) {
                e.stopPropagation();
                $('#filterDropdown').toggleClass('show');
            });

            // Close filter dropdown when clicking outside
            $(document).click(function(e) {
                if (!$(e.target).closest('#filterBtn, #filterDropdown').length) {
                    $('#filterDropdown').removeClass('show');
                }
            });

            // Buyer dropdown toggle
            $('#buyerDropdownBtn').click(function (e) {
                e.stopPropagation();
                $('#buyerDropdownMenu').toggleClass('hidden');
                $('#ocnDropdownMenu').addClass('hidden'); // close other dropdown
            });

            // Buyer option select
            $(document).on('click', '#buyerDropdownMenu li', function () {
                selectedBuyer = $(this).data('value');
                $('#buyerSelected').text(selectedBuyer);
                $('#buyerDropdownMenu').addClass('hidden');

                // Filter OCNs for this buyer
                const ocns = buyerOcnData
                    .filter(item => item.Buyer === selectedBuyer)
                    .map(item => item.Ocn_No);

                // Populate OCN dropdown
                const ocnMenu = $('#ocnDropdownMenu');
                ocnMenu.empty();
                ocns.forEach(ocn => {
                    ocnMenu.append(
                        `<li class="px-3 py-2 hover:bg-muted cursor-pointer" style="font-size: 11px;" data-value="${ocn}">${ocn}</li>`
                    );
                });

                // Reset selected OCN display
                selectedOcn = null;
                $('#ocnSelected').text("Select OCN");

                // Show OCN dropdown (was hidden initially)
                $('#ocnDropdown').removeClass('hidden');

                applyFilters();
            });

            // OCN dropdown toggle
            $('#ocnDropdownBtn').click(function (e) {
                e.stopPropagation();
                $('#ocnDropdownMenu').toggleClass('hidden');
                $('#buyerDropdownMenu').addClass('hidden');
            });

            // OCN option select
            $(document).on('click', '#ocnDropdownMenu li', function () {
                selectedOcn = $(this).data('value');
                $('#ocnSelected').text(selectedOcn);
                $('#ocnDropdownMenu').addClass('hidden');
                applyFilters();
            });

            // Close dropdowns when clicking outside
            $(document).click(function () {
                $('#buyerDropdownMenu, #ocnDropdownMenu').addClass('hidden');
            });

            // Reset filters
            $('#resetFiltersBtn').click(function () {
                selectedBuyer = null;
                selectedOcn = null;
                $('#buyerSelected').text("Select Buyer");
                $('#ocnSelected').text("Select OCN");
                $('#ocnDropdown').addClass('hidden'); // hide ocn filter again
                applyFilters();
            });

            // Notifications panel
            $('#notificationsBtn').click(function() {
                $('#notificationOverlay').removeClass('hidden');
                setTimeout(() => {
                    $('#notificationPanel').removeClass('translate-x-full');
                }, 10);
            });

            $('#closeNotifications, #notificationOverlay').click(function(e) {
                if (e.target === this) {
                    $('#notificationPanel').addClass('translate-x-full');
                    setTimeout(() => {
                        $('#notificationOverlay').addClass('hidden');
                    }, 300);
                }
            });

            // Tooltip functionality
            $(document).on('mouseenter', '[title]', function () {
                const tooltip = $('#tooltip');
                const title = $(this).attr('title');

                if (title) {
                    // Empty tooltip first, then set new text
                    tooltip.text('').text(title).addClass('show');

                    // Position tooltip above the element
                    const offset = $(this).offset();
                    const tooltipHeight = tooltip.outerHeight();
                    const tooltipWidth = tooltip.outerWidth();
                    const elementWidth = $(this).outerWidth();

                    tooltip.css({
                        left: offset.left + (elementWidth / 2) - (tooltipWidth / 2) + 'px',
                        top: offset.top - tooltipHeight - 8 + 'px' // 8px gap above the element
                    });
                }
            });

            $(document).on('mouseleave', '[title]', function () {
                $('#tooltip').removeClass('show').text('');
            });

        }

        $(document).ready(function () {
            initializeKpiCards();
            renderTable();
            initializeEventHandlers();
            lucide.createIcons();
            fetchBuyers();

            // Initialize date range picker
            $('#dateRangePicker').daterangepicker({
                opens: 'left',
                startDate: moment().subtract(6, 'days'),
                endDate: moment(),
                ranges: {
                'Today': [moment(), moment()],
                //'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
                //'Last 7 Days': [moment().subtract(6, 'days'), moment()],
                'This Week': [moment().startOf('week'), moment().endOf('week')],
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                //'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
                }
            }, function(start, end, label) {
                $('#dateRangePicker').val(start.format('MMM D, YYYY') + ' - ' + end.format('MMM D, YYYY'));
                console.log("Selected range:", start.format(), end.format(), label);
                // 🔥 You can trigger dashboard reload here
            });

            // Default display
            $('#dateRangePicker').val('Last 7 days');
        });

    </script>
</body>
</html>
