import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from './ui/table';
import { Button } from './ui/button';
import { Progress } from './ui/progress';
import { TrendingDown, Download, Eye, Award, DollarSign, Target, Users } from 'lucide-react';

interface CostSavingsBreakupProps {
  data: any[];
  selectedManager?: string;
}

export function CostSavingsBreakup({ data, selectedManager }: CostSavingsBreakupProps) {
  // Calculate cost savings for each purchase order
  const calculateCostSavings = (item: any) => {
    if (!item.budgetRate || !item.poRate || item.poRate >= item.budgetRate) {
      return {
        savingsPerUnit: 0,
        totalSavings: 0,
        savingsPercentage: 0
      };
    }
    
    const savingsPerUnit = item.budgetRate - item.poRate;
    const totalSavings = savingsPerUnit * item.poQty;
    const savingsPercentage = ((savingsPerUnit / item.budgetRate) * 100);
    
    return {
      savingsPerUnit,
      totalSavings,
      savingsPercentage
    };
  };

  // Filter data for cost savings (only where purchase rate < budget rate)
  const costSavingsData = data
    .map(item => ({
      ...item,
      ...calculateCostSavings(item)
    }))
    .filter(item => item.totalSavings > 0)
    .sort((a, b) => b.totalSavings - a.totalSavings);

  // Filter by selected manager if specified
  const filteredData = selectedManager 
    ? costSavingsData.filter(item => item.purchaseManager === selectedManager)
    : costSavingsData;

  // Calculate manager-wise cost savings
  const managerSavings = data.reduce((acc, item) => {
    const savings = calculateCostSavings(item);
    if (savings.totalSavings > 0) {
      if (!acc[item.purchaseManager]) {
        acc[item.purchaseManager] = {
          manager: item.purchaseManager,
          totalSavings: 0,
          savingsOrders: 0,
          totalOrders: 0,
          avgSavingsPercentage: 0,
          maxSavings: 0,
          totalBudgetValue: 0,
          totalActualValue: 0
        };
      }
      
      acc[item.purchaseManager].totalSavings += savings.totalSavings;
      acc[item.purchaseManager].savingsOrders += 1;
      acc[item.purchaseManager].maxSavings = Math.max(acc[item.purchaseManager].maxSavings, savings.totalSavings);
      acc[item.purchaseManager].totalBudgetValue += (item.budgetRate * item.poQty);
      acc[item.purchaseManager].totalActualValue += (item.poRate * item.poQty);
    }
    
    acc[item.purchaseManager] = acc[item.purchaseManager] || {
      manager: item.purchaseManager,
      totalSavings: 0,
      savingsOrders: 0,
      totalOrders: 0,
      avgSavingsPercentage: 0,
      maxSavings: 0,
      totalBudgetValue: 0,
      totalActualValue: 0
    };
    
    acc[item.purchaseManager].totalOrders += 1;
    
    return acc;
  }, {});

  // Calculate average savings percentage for each manager
  Object.values(managerSavings).forEach((manager: any) => {
    if (manager.totalBudgetValue > 0) {
      manager.avgSavingsPercentage = ((manager.totalBudgetValue - manager.totalActualValue) / manager.totalBudgetValue) * 100;
    }
    manager.savingsEfficiency = manager.totalOrders > 0 ? (manager.savingsOrders / manager.totalOrders) * 100 : 0;
  });

  const managerSavingsArray = Object.values(managerSavings).sort((a: any, b: any) => b.totalSavings - a.totalSavings);

  const totalSavings = filteredData.reduce((sum, item) => sum + item.totalSavings, 0);
  const avgSavingsPerOrder = filteredData.length > 0 ? totalSavings / filteredData.length : 0;
  const maxSavings = Math.max(...filteredData.map(item => item.totalSavings), 0);
  const savingsOrders = filteredData.length;

  if (filteredData.length === 0) {
    return (
      <Card>
        <CardContent className="p-6 text-center">
          <TrendingDown className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
          <h3 className="font-medium text-muted-foreground">No Cost Savings Data</h3>
          <p className="text-sm text-muted-foreground">
            {selectedManager 
              ? `No cost savings found for ${selectedManager}` 
              : 'No orders found where purchase rate is less than budget rate'
            }
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <DollarSign className="h-4 w-4 text-green-500" />
              <div>
                <p className="text-sm text-muted-foreground">Total Cost Savings</p>
                <p className="text-lg font-bold text-green-600">₹{totalSavings.toLocaleString()}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Target className="h-4 w-4 text-blue-500" />
              <div>
                <p className="text-sm text-muted-foreground">Orders with Savings</p>
                <p className="text-lg font-bold">{savingsOrders}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <TrendingDown className="h-4 w-4 text-purple-500" />
              <div>
                <p className="text-sm text-muted-foreground">Avg Savings/Order</p>
                <p className="text-lg font-bold">₹{avgSavingsPerOrder.toLocaleString()}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Award className="h-4 w-4 text-yellow-500" />
              <div>
                <p className="text-sm text-muted-foreground">Highest Savings</p>
                <p className="text-lg font-bold">₹{maxSavings.toLocaleString()}</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Manager Performance Comparison */}
      {!selectedManager && managerSavingsArray.length > 1 && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5" />
              Manager Cost Savings Comparison
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {managerSavingsArray.map((manager: any, index) => (
                <div key={manager.manager} className="space-y-2">
                  <div className="flex justify-between items-center">
                    <div className="flex items-center gap-2">
                      <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-bold ${
                        index === 0 ? 'bg-green-500 text-white' : 'bg-gray-100 text-gray-700'
                      }`}>
                        #{index + 1}
                      </div>
                      <div>
                        <span className="font-medium">{manager.manager}</span>
                        <p className="text-xs text-muted-foreground">
                          {manager.savingsOrders} orders • {manager.savingsEfficiency.toFixed(1)}% efficiency
                        </p>
                      </div>
                    </div>
                    <div className="text-right">
                      <span className="text-lg font-bold text-green-600">
                        ₹{manager.totalSavings.toLocaleString()}
                      </span>
                      <p className="text-xs text-muted-foreground">
                        {manager.avgSavingsPercentage.toFixed(1)}% avg savings
                      </p>
                    </div>
                  </div>
                  <Progress 
                    value={(manager.totalSavings / managerSavingsArray[0].totalSavings) * 100} 
                    className="h-2" 
                  />
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Detailed Cost Savings Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <TrendingDown className="h-5 w-5 text-green-500" />
              {selectedManager ? `${selectedManager} - ` : ''}Cost Savings Details
            </div>
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>PO No</TableHead>
                  <TableHead>Date</TableHead>
                  <TableHead>Supplier</TableHead>
                  <TableHead>Item Group</TableHead>
                  <TableHead>Manager</TableHead>
                  <TableHead className="text-right">Budget Rate</TableHead>
                  <TableHead className="text-right">Purchase Rate</TableHead>
                  <TableHead className="text-right">Qty</TableHead>
                  <TableHead className="text-right">Savings/Unit</TableHead>
                  <TableHead className="text-right">Total Savings</TableHead>
                  <TableHead className="text-right">Savings %</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredData.map((item, index) => (
                  <TableRow key={`${item.id}-${index}`}>
                    <TableCell className="font-medium">{item.id}</TableCell>
                    <TableCell>{item.poDate}</TableCell>
                    <TableCell className="max-w-[150px] truncate" title={item.supplier}>
                      {item.supplier}
                    </TableCell>
                    <TableCell>{item.itemGroup}</TableCell>
                    <TableCell>
                      <Badge variant="outline">
                        {item.purchaseManager}
                      </Badge>
                    </TableCell>
                    <TableCell className="text-right">₹{item.budgetRate.toFixed(2)}</TableCell>
                    <TableCell className="text-right">₹{item.poRate.toFixed(2)}</TableCell>
                    <TableCell className="text-right">{item.poQty.toLocaleString()} {item.uom}</TableCell>
                    <TableCell className="text-right text-green-600 font-medium">
                      ₹{item.savingsPerUnit.toFixed(2)}
                    </TableCell>
                    <TableCell className="text-right">
                      <span className="font-bold text-green-600">
                        ₹{item.totalSavings.toLocaleString()}
                      </span>
                    </TableCell>
                    <TableCell className="text-right">
                      <Badge variant="default" className="bg-green-100 text-green-800">
                        {item.savingsPercentage.toFixed(1)}%
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <Button variant="ghost" size="sm">
                        <Eye className="h-3 w-3" />
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Summary Footer */}
          <div className="mt-4 p-4 bg-green-50 rounded-lg">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
              <div>
                <span className="text-muted-foreground">Total Cost Savings: </span>
                <span className="font-bold text-green-700">₹{totalSavings.toLocaleString()}</span>
              </div>
              <div>
                <span className="text-muted-foreground">Average Savings per Order: </span>
                <span className="font-bold text-green-700">₹{avgSavingsPerOrder.toLocaleString()}</span>
              </div>
              <div>
                <span className="text-muted-foreground">Orders with Savings: </span>
                <span className="font-bold text-green-700">{savingsOrders} orders</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Top Savings Achievements */}
      <Card className="border-green-200 bg-green-50">
        <CardHeader>
          <CardTitle className="flex items-center gap-2 text-green-800">
            <Award className="h-5 w-5" />
            Top Cost Savings Achievements
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-3">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            {filteredData.slice(0, 3).map((item, index) => (
              <div key={item.id} className="bg-white p-4 rounded-lg border">
                <div className="flex items-center justify-between mb-2">
                  <Badge variant="outline" className="bg-green-100 text-green-800">
                    #{index + 1} Best
                  </Badge>
                  <span className="text-lg font-bold text-green-600">
                    ₹{item.totalSavings.toLocaleString()}
                  </span>
                </div>
                <div className="space-y-1 text-sm">
                  <p><strong>PO:</strong> {item.id}</p>
                  <p><strong>Supplier:</strong> {item.supplier}</p>
                  <p><strong>Manager:</strong> {item.purchaseManager}</p>
                  <p><strong>Savings:</strong> {item.savingsPercentage.toFixed(1)}% ({item.savingsPerUnit.toFixed(2)}/unit)</p>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}