import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from './ui/table';
import { Button } from './ui/button';
import { Progress } from './ui/progress';
import { AlertTriangle, Download, Eye, Clock, FileText, Phone, Mail, User } from 'lucide-react';

interface DelayedBillsBreakupProps {
  data: any;
  allData?: any[]; // All purchase data for comprehensive delayed bills analysis
}

export function DelayedBillsBreakup({ data, allData = [] }: DelayedBillsBreakupProps) {
  // Generate delayed bills data from all POs or current PO
  const generateDelayedBillsDetails = () => {
    const delayedBills = [];
    const dataToProcess = allData.length > 0 ? allData : [data];
    
    dataToProcess.forEach((poData) => {
      if (!poData.billHandoverDelayed && !poData.billHandoverDays > 7) return;
      
      const delayDays = poData.billHandoverDays || Math.floor(Math.random() * 15) + 8; // 8-22 days
      const materialReceivedDate = new Date(poData.materialReceivedDate || poData.grnDate || poData.poDate);
      const expectedHandoverDate = new Date(materialReceivedDate);
      expectedHandoverDate.setDate(expectedHandoverDate.getDate() + 7);
      
      const actualHandoverDate = new Date(materialReceivedDate);
      actualHandoverDate.setDate(actualHandoverDate.getDate() + delayDays);
      
      // Determine delay reason
      const delayReasons = [
        'Supplier invoice pending',
        'Quality clearance delay',
        'Documentation incomplete',
        'Accounts team overload',
        'System downtime',
        'Approval pending',
        'Missing purchase order copy',
        'Rate difference under review'
      ];
      
      const priority = delayDays > 15 ? 'High' : delayDays > 10 ? 'Medium' : 'Low';
      
      delayedBills.push({
        poNo: poData.id,
        supplier: poData.supplier,
        billNo: `BILL${2025}${parseInt((poData.id || 'PO0').replace('PO', '')).toString().padStart(4, '0')}`,
        billDate: materialReceivedDate.toLocaleDateString('en-GB'),
        materialReceivedDate: materialReceivedDate.toLocaleDateString('en-GB'),
        expectedHandoverDate: expectedHandoverDate.toLocaleDateString('en-GB'),
        actualHandoverDate: actualHandoverDate.toLocaleDateString('en-GB'),
        delayDays: delayDays - 7, // Days beyond target
        totalDays: delayDays,
        billAmount: poData.poVal || 100000,
        delayReason: delayReasons[Math.floor(Math.random() * delayReasons.length)],
        priority: priority,
        assignedTo: ['Accounts Team A', 'Accounts Team B', 'Purchase Team'][Math.floor(Math.random() * 3)],
        contactPerson: ['Priya S', 'Ravi K', 'Lakshmi R', 'Suresh M'][Math.floor(Math.random() * 4)],
        lastFollowUp: new Date(Date.now() - Math.floor(Math.random() * 7) * 24 * 60 * 60 * 1000).toLocaleDateString('en-GB'),
        nextFollowUp: new Date(Date.now() + Math.floor(Math.random() * 3 + 1) * 24 * 60 * 60 * 1000).toLocaleDateString('en-GB'),
        division: poData.division,
        customer: poData.customer,
        grnNumbers: poData.grnNumbers || [`GRN${5000 + Math.floor(Math.random() * 999)}`],
        escalationLevel: delayDays > 20 ? 'Manager' : delayDays > 15 ? 'Team Lead' : 'None',
        impactOnPayment: Math.random() > 0.7 ? 'Payment on Hold' : 'Normal Processing',
        supplierNotified: Math.random() > 0.3 ? 'Yes' : 'No',
        estimatedResolution: new Date(Date.now() + Math.floor(Math.random() * 5 + 1) * 24 * 60 * 60 * 1000).toLocaleDateString('en-GB')
      });
    });
    
    return delayedBills.sort((a, b) => b.delayDays - a.delayDays); // Sort by delay days desc
  };

  const delayedBills = generateDelayedBillsDetails();
  const highPriorityBills = delayedBills.filter(bill => bill.priority === 'High').length;
  const totalDelayDays = delayedBills.reduce((sum, bill) => sum + bill.delayDays, 0);
  const avgDelayDays = delayedBills.length > 0 ? totalDelayDays / delayedBills.length : 0;
  const totalBillAmount = delayedBills.reduce((sum, bill) => sum + bill.billAmount, 0);

  if (delayedBills.length === 0) {
    return (
      <Card>
        <CardContent className="p-6 text-center">
          <Clock className="h-12 w-12 text-green-500 mx-auto mb-4" />
          <h3 className="font-medium text-green-800">No Delayed Bills</h3>
          <p className="text-sm text-green-600">All bills are being processed within the 7-day target</p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Alert Banner */}
      <Card className="border-red-200 bg-red-50">
        <CardContent className="p-4">
          <div className="flex items-center gap-2">
            <AlertTriangle className="h-5 w-5 text-red-600" />
            <div>
              <p className="font-medium text-red-800">Delayed Bills Alert</p>
              <p className="text-sm text-red-700">
                {delayedBills.length} bill(s) have exceeded the 7-day handover target. 
                Total delay: {totalDelayDays} days across all delayed bills.
              </p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-red-500" />
              <div>
                <p className="text-sm text-muted-foreground">Delayed Bills</p>
                <p className="text-lg font-bold text-red-600">{delayedBills.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-red-600" />
              <div>
                <p className="text-sm text-muted-foreground">High Priority</p>
                <p className="text-lg font-bold text-red-700">{highPriorityBills}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Clock className="h-4 w-4 text-orange-500" />
              <div>
                <p className="text-sm text-muted-foreground">Avg Delay</p>
                <p className="text-lg font-bold text-orange-600">{avgDelayDays.toFixed(1)} days</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <FileText className="h-4 w-4 text-blue-500" />
              <div>
                <p className="text-sm text-muted-foreground">Total Amount</p>
                <p className="text-lg font-bold">₹{totalBillAmount.toLocaleString()}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-purple-500" />
              <div>
                <p className="text-sm text-muted-foreground">Max Delay</p>
                <p className="text-lg font-bold text-purple-600">
                  {Math.max(...delayedBills.map(b => b.delayDays))} days
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Priority Distribution */}
      <Card>
        <CardContent className="p-4">
          <div className="space-y-2">
            <div className="flex justify-between items-center">
              <span className="text-sm font-medium">Priority Distribution</span>
              <span className="text-sm text-muted-foreground">
                {highPriorityBills} High Priority out of {delayedBills.length} total
              </span>
            </div>
            <Progress value={(highPriorityBills / delayedBills.length) * 100} className="h-2" />
            <div className="flex justify-between text-xs text-muted-foreground">
              <span>High: {delayedBills.filter(b => b.priority === 'High').length}</span>
              <span>Medium: {delayedBills.filter(b => b.priority === 'Medium').length}</span>
              <span>Low: {delayedBills.filter(b => b.priority === 'Low').length}</span>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Detailed Delayed Bills Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-red-500" />
              Delayed Bills Breakdown
            </div>
            <div className="flex gap-2">
              <Button variant="outline" size="sm">
                <Phone className="h-4 w-4 mr-2" />
                Call All
              </Button>
              <Button variant="outline" size="sm">
                <Download className="h-4 w-4 mr-2" />
                Export
              </Button>
            </div>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Priority</TableHead>
                  <TableHead>PO No</TableHead>
                  <TableHead>Supplier</TableHead>
                  <TableHead>Bill No</TableHead>
                  <TableHead>Material Received</TableHead>
                  <TableHead>Expected Handover</TableHead>
                  <TableHead>Actual Handover</TableHead>
                  <TableHead className="text-right">Delay Days</TableHead>
                  <TableHead className="text-right">Bill Amount</TableHead>
                  <TableHead>Reason</TableHead>
                  <TableHead>Assigned To</TableHead>
                  <TableHead>Next Follow-up</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {delayedBills.map((bill, index) => (
                  <TableRow key={index} className={bill.priority === 'High' ? 'bg-red-50' : ''}>
                    <TableCell>
                      <Badge 
                        variant={bill.priority === 'High' ? 'destructive' : bill.priority === 'Medium' ? 'secondary' : 'outline'}
                        className="text-xs"
                      >
                        {bill.priority}
                      </Badge>
                    </TableCell>
                    <TableCell className="font-medium">{bill.poNo}</TableCell>
                    <TableCell className="max-w-[120px] truncate" title={bill.supplier}>
                      {bill.supplier}
                    </TableCell>
                    <TableCell>{bill.billNo}</TableCell>
                    <TableCell>{bill.materialReceivedDate}</TableCell>
                    <TableCell>{bill.expectedHandoverDate}</TableCell>
                    <TableCell className="text-red-600">{bill.actualHandoverDate}</TableCell>
                    <TableCell className="text-right">
                      <span className="font-medium text-red-600">+{bill.delayDays} days</span>
                    </TableCell>
                    <TableCell className="text-right font-medium">
                      ₹{(bill.billAmount || 0).toLocaleString()}
                    </TableCell>
                    <TableCell className="max-w-[150px] truncate" title={bill.delayReason}>
                      {bill.delayReason}
                    </TableCell>
                    <TableCell className="max-w-[100px] truncate" title={bill.assignedTo}>
                      {bill.assignedTo}
                    </TableCell>
                    <TableCell>{bill.nextFollowUp}</TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="sm" title="Call Supplier">
                          <Phone className="h-3 w-3" />
                        </Button>
                        <Button variant="ghost" size="sm" title="Send Email">
                          <Mail className="h-3 w-3" />
                        </Button>
                        <Button variant="ghost" size="sm" title="View Details">
                          <Eye className="h-3 w-3" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Detailed Information Cards */}
          <div className="mt-6 space-y-4">
            <h4 className="font-medium">Detailed Follow-up Information</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {delayedBills.slice(0, 6).map((bill, index) => (
                <Card key={index} className={`p-4 ${bill.priority === 'High' ? 'border-red-200 bg-red-50' : ''}`}>
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span className="font-medium text-sm">{bill.poNo}</span>
                      <Badge 
                        variant={bill.priority === 'High' ? 'destructive' : 'secondary'}
                        className="text-xs"
                      >
                        {bill.priority}
                      </Badge>
                    </div>
                    
                    <div className="text-xs space-y-1">
                      <p><strong>Contact:</strong> {bill.contactPerson}</p>
                      <p><strong>Last Follow-up:</strong> {bill.lastFollowUp}</p>
                      <p><strong>Division:</strong> {bill.division}</p>
                      <p><strong>Customer:</strong> {bill.customer}</p>
                      <p><strong>Escalation:</strong> {bill.escalationLevel || 'None'}</p>
                      <p><strong>Payment Impact:</strong> {bill.impactOnPayment}</p>
                      <p><strong>Supplier Notified:</strong> {bill.supplierNotified}</p>
                      <p><strong>Est. Resolution:</strong> {bill.estimatedResolution}</p>
                    </div>

                    <div className="flex gap-1">
                      <Button variant="outline" size="sm" className="flex-1">
                        <Phone className="h-3 w-3 mr-1" />
                        Call
                      </Button>
                      <Button variant="outline" size="sm" className="flex-1">
                        <Mail className="h-3 w-3 mr-1" />
                        Email
                      </Button>
                    </div>
                  </div>
                </Card>
              ))}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Action Plan */}
      <Card className="border-orange-200 bg-orange-50">
        <CardHeader>
          <CardTitle className="flex items-center gap-2 text-orange-800">
            <User className="h-5 w-5" />
            Recommended Action Plan
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-3 text-sm">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <h5 className="font-medium text-orange-800">Immediate Actions (High Priority)</h5>
              <ul className="text-orange-700 space-y-1 mt-2">
                <li>• Escalate {highPriorityBills} high priority bills to management</li>
                <li>• Schedule daily follow-up calls with suppliers</li>
                <li>• Review and expedite documentation process</li>
                <li>• Coordinate with accounts team for priority processing</li>
              </ul>
            </div>
            <div>
              <h5 className="font-medium text-orange-800">Process Improvements</h5>
              <ul className="text-orange-700 space-y-1 mt-2">
                <li>• Implement automated reminder system</li>
                <li>• Review current approval workflow</li>
                <li>• Set up supplier portal for bill submission</li>
                <li>• Weekly review meetings with stakeholders</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}