import React from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from './ui/dialog';
import { Badge } from './ui/badge';
import { Separator } from './ui/separator';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Package, Calendar, User, DollarSign, TrendingUp, TrendingDown, Clock } from 'lucide-react';
import { AdvancedPODetails } from './AdvancedPODetails';

interface PurchaseOrder {
  id: string;
  poDate: string;
  poType: string;
  supplier: string;
  itemGroup: string;
  dcn: string;
  uom: string;
  poQty: number;
  poRate: number;
  poVal: number;
  cancelQty: number;
  budgetRate: number;
  leadDays: string;
  grnDate: string;
  division: string;
  // Advanced fields
  grnNumbers?: string[];
  grnCount?: number;
  supplierAdvancePaid?: boolean;
  advanceAmount?: number;
  advancePendingBillClose?: boolean;
  customerPaymentType?: string;
  customerCreditDays?: number;
  materialReceivedDate?: string;
  billHandoverDate?: string;
  billHandoverDays?: number;
  billHandoverDelayed?: boolean;
  invoiceNumber?: string;
  paymentOutstanding?: number;
  paymentStatus?: string;
}

interface DrilldownModalProps {
  isOpen: boolean;
  onClose: () => void;
  data: PurchaseOrder | any | null;
  title: string;
}

export function DrilldownModal({ isOpen, onClose, data, title }: DrilldownModalProps) {
  if (!data) return null;

  // Check if this is a Purchase Order Details view
  const isPODetails = title === 'Purchase Order Details' && data.id?.startsWith('PO');

  // If it's a PO details view, use the advanced component
  if (isPODetails) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-6xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Package className="h-5 w-5" />
              Advanced Purchase Order Details: {data.id}
            </DialogTitle>
          </DialogHeader>
          <AdvancedPODetails data={data} />
        </DialogContent>
      </Dialog>
    );
  }

  // For other drilldown views, use the basic modal
  const variance = data.budgetRate ? ((data.poRate - data.budgetRate) / data.budgetRate * 100) : 0;
  const isOverBudget = variance > 0;
  const deliveryStatus = data.grnDate ? 'Delivered' : 'Pending';
  const hasQualityIssue = data.cancelQty > 0;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Package className="h-5 w-5" />
            {title}: {data.name || data.id || 'Details'}
          </DialogTitle>
        </DialogHeader>

        {/* For non-PO data, show basic information */}
        <div className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Details</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 gap-4">
                {Object.entries(data).map(([key, value]) => (
                  <div key={key}>
                    <p className="text-sm text-muted-foreground capitalize">
                      {key.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())}
                    </p>
                    <p className="font-medium">
                      {typeof value === 'number' ? value.toLocaleString() : String(value)}
                    </p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>
      </DialogContent>
    </Dialog>
  );
}