import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from './ui/table';
import { Button } from './ui/button';
import { Progress } from './ui/progress';
import { Receipt, Download, Eye, CheckCircle, AlertTriangle, Clock, TruckIcon } from 'lucide-react';

interface GRNToBillBreakupProps {
  data: any;
}

export function GRNToBillBreakup({ data }: GRNToBillBreakupProps) {
  if (!data) return null;

  // Generate mock GRN to Bill details
  const generateGRNToBillDetails = () => {
    const grnDetails = [];
    const grnCount = data.grnCount || Math.floor(Math.random() * 3) + 1;
    
    for (let i = 0; i < grnCount; i++) {
      const gateInwardDate = new Date(data.materialReceivedDate || data.grnDate || data.poDate);
      gateInwardDate.setDate(gateInwardDate.getDate() + (i * 2)); // Stagger GRNs
      
      const billDate = new Date(gateInwardDate);
      billDate.setDate(billDate.getDate() + Math.floor(Math.random() * 5) + 1); // Bill 1-5 days after GRN
      
      const billHandoverDate = new Date(billDate);
      const handoverDays = Math.floor(Math.random() * 15) + 1; // 1-15 days for handover
      billHandoverDate.setDate(billHandoverDate.getDate() + handoverDays);
      
      const targetDays = 7;
      const isDelayed = handoverDays > targetDays;
      const handoverStatus = Math.random() > 0.2 ? 'Completed' : 'Pending'; // 80% completed
      
      grnDetails.push({
        gateInwardNo: `GIN${2025}${(parseInt(data.id.replace('PO', '')) + i + 1000).toString().padStart(4, '0')}`,
        gateInwardDate: gateInwardDate.toLocaleDateString('en-GB'),
        grnNo: data.grnNumbers?.[i] || `GRN${5000 + i}`,
        supplier: data.supplier,
        billNo: `BILL${2025}${(parseInt(data.id.replace('PO', '')) + i + 2000).toString().padStart(4, '0')}`,
        billDate: billDate.toLocaleDateString('en-GB'),
        billHandoverStatus: handoverStatus,
        billHandoverDate: handoverStatus === 'Completed' ? billHandoverDate.toLocaleDateString('en-GB') : null,
        leadDays: handoverStatus === 'Completed' ? handoverDays : null,
        targetDays: targetDays,
        isDelayed: handoverStatus === 'Completed' && isDelayed,
        billAmount: Math.floor(((data.poVal || 100000) / grnCount) * (0.8 + Math.random() * 0.4)), // Vary amounts
        vehicleNo: `TN${Math.floor(Math.random() * 99)}${String.fromCharCode(65 + Math.floor(Math.random() * 26))}${String.fromCharCode(65 + Math.floor(Math.random() * 26))}${Math.floor(Math.random() * 9999).toString().padStart(4, '0')}`,
        driverName: ['Raman', 'Kumar', 'Selvam', 'Murugan', 'Prakash'][Math.floor(Math.random() * 5)],
        receivedBy: ['Store Keeper', 'QC Team', 'Warehouse Incharge'][Math.floor(Math.random() * 3)],
        qcStatus: ['Approved', 'Approved', 'Approved', 'Hold', 'Rejected'][Math.floor(Math.random() * 5)]
      });
    }
    
    return grnDetails;
  };

  const grnDetails = generateGRNToBillDetails();
  const completedBills = grnDetails.filter(grn => grn.billHandoverStatus === 'Completed').length;
  const delayedBills = grnDetails.filter(grn => grn.isDelayed).length;
  const avgLeadDays = grnDetails
    .filter(grn => grn.leadDays)
    .reduce((sum, grn) => sum + grn.leadDays, 0) / completedBills || 0;

  return (
    <div className="space-y-6">
      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Receipt className="h-4 w-4 text-blue-500" />
              <div>
                <p className="text-sm text-muted-foreground">Total GRNs</p>
                <p className="text-lg font-bold">{grnDetails.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <CheckCircle className="h-4 w-4 text-green-500" />
              <div>
                <p className="text-sm text-muted-foreground">Bills Completed</p>
                <p className="text-lg font-bold">{completedBills}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-red-500" />
              <div>
                <p className="text-sm text-muted-foreground">Delayed Bills</p>
                <p className="text-lg font-bold">{delayedBills}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Clock className="h-4 w-4 text-orange-500" />
              <div>
                <p className="text-sm text-muted-foreground">Avg Lead Days</p>
                <p className="text-lg font-bold">{avgLeadDays.toFixed(1)}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <TruckIcon className="h-4 w-4 text-purple-500" />
              <div>
                <p className="text-sm text-muted-foreground">Target Days</p>
                <p className="text-lg font-bold">≤7 days</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Performance Indicator */}
      <Card>
        <CardContent className="p-4">
          <div className="space-y-2">
            <div className="flex justify-between items-center">
              <span className="text-sm font-medium">Bill Handover Performance</span>
              <span className="text-sm text-muted-foreground">
                {((completedBills - delayedBills) / completedBills * 100).toFixed(1)}% On-Time
              </span>
            </div>
            <Progress value={((completedBills - delayedBills) / completedBills * 100)} className="h-2" />
            <p className="text-xs text-muted-foreground">
              {completedBills - delayedBills} out of {completedBills} bills completed within 7-day target
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Detailed GRN to Bill Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <Receipt className="h-5 w-5" />
              GRN to Bill Tracking Details
            </div>
            <Button variant="outline" size="sm">
              <Download className="h-4 w-4 mr-2" />
              Export
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Gate Inward No</TableHead>
                  <TableHead>Gate Date</TableHead>
                  <TableHead>GRN No</TableHead>
                  <TableHead>Supplier</TableHead>
                  <TableHead>Bill No</TableHead>
                  <TableHead>Bill Date</TableHead>
                  <TableHead>Handover Status</TableHead>
                  <TableHead>Handover Date</TableHead>
                  <TableHead className="text-right">Lead Days</TableHead>
                  <TableHead className="text-right">Target Days</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {grnDetails.map((grn, index) => (
                  <TableRow key={index}>
                    <TableCell className="font-medium">{grn.gateInwardNo}</TableCell>
                    <TableCell>{grn.gateInwardDate}</TableCell>
                    <TableCell>{grn.grnNo}</TableCell>
                    <TableCell className="max-w-[120px] truncate" title={grn.supplier}>
                      {grn.supplier}
                    </TableCell>
                    <TableCell>{grn.billNo}</TableCell>
                    <TableCell>{grn.billDate}</TableCell>
                    <TableCell>
                      <Badge 
                        variant={grn.billHandoverStatus === 'Completed' ? 'default' : 'secondary'}
                        className="text-xs"
                      >
                        {grn.billHandoverStatus}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      {grn.billHandoverDate || (
                        <Badge variant="outline" className="text-orange-600">
                          Pending
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell className="text-right">
                      {grn.leadDays ? (
                        <span className={grn.isDelayed ? 'text-red-600 font-medium' : 'text-green-600'}>
                          {grn.leadDays} days
                        </span>
                      ) : '-'}
                    </TableCell>
                    <TableCell className="text-right">{grn.targetDays} days</TableCell>
                    <TableCell>
                      {grn.billHandoverStatus === 'Completed' ? (
                        <Badge 
                          variant={grn.isDelayed ? 'destructive' : 'default'}
                          className="text-xs"
                        >
                          {grn.isDelayed ? 'Delayed' : 'On-Time'}
                        </Badge>
                      ) : (
                        <Badge variant="secondary" className="text-xs">
                          In Progress
                        </Badge>
                      )}
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Button variant="ghost" size="sm">
                          <Eye className="h-3 w-3" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Additional Details Section */}
          <div className="mt-6 space-y-4">
            <h4 className="font-medium">Additional GRN Details</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {grnDetails.map((grn, index) => (
                <Card key={index} className="p-4">
                  <div className="space-y-2">
                    <div className="flex justify-between items-center">
                      <span className="font-medium text-sm">{grn.grnNo}</span>
                      <Badge variant="outline" className="text-xs">
                        {grn.qcStatus}
                      </Badge>
                    </div>
                    <div className="text-xs text-muted-foreground space-y-1">
                      <p><strong>Vehicle:</strong> {grn.vehicleNo}</p>
                      <p><strong>Driver:</strong> {grn.driverName}</p>
                      <p><strong>Received By:</strong> {grn.receivedBy}</p>
                      <p><strong>Bill Amount:</strong> ₹{grn.billAmount.toLocaleString()}</p>
                    </div>
                  </div>
                </Card>
              ))}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Action Items */}
      {delayedBills > 0 && (
        <Card className="border-red-200 bg-red-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-4 w-4 text-red-600" />
              <div>
                <p className="font-medium text-red-800">Urgent Action Required</p>
                <p className="text-sm text-red-700">
                  {delayedBills} bill(s) have exceeded the 7-day handover target. 
                  Immediate attention required to expedite accounts processing.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}