import React from 'react';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  LineChart,
  Line,
  ComposedChart,
  RadarChart,
  PolarGrid,
  PolarAngleAxis,
  PolarRadiusAxis,
  Radar,
  Cell,
  ReferenceLine
} from 'recharts';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { TrendingUp, TrendingDown, Target, Award, Clock, DollarSign } from 'lucide-react';

const MANAGER_COLORS = {
  'Mr.Britto': '#2563eb',
  'Mr.Udhaya': '#dc2626'
};

const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length) {
    return (
      <div className="bg-white p-4 border rounded-lg shadow-lg">
        <p className="font-medium mb-2">{`${label}`}</p>
        {payload.map((entry: any, index: number) => (
          <div key={index} className="flex items-center gap-2">
            <div 
              className="w-3 h-3 rounded-full" 
              style={{ backgroundColor: entry.color }}
            ></div>
            <span className="text-sm">
              {entry.name}: {typeof entry.value === 'number' ? 
                entry.value.toFixed(1) : entry.value}
            </span>
          </div>
        ))}
      </div>
    );
  }
  return null;
};

interface ManagerChartProps {
  data: any[];
  title: string;
  onClick?: (data: any) => void;
  height?: number;
}

export function ManagerPerformanceComparison({ data, title, onClick, height = 400 }: ManagerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Award className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <ComposedChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="period" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              yAxisId="left"
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              yAxisId="right" 
              orientation="right"
              fontSize={12}
              stroke="#64748b"
            />
            <Tooltip content={<CustomTooltip />} />
            <Bar 
              yAxisId="left"
              dataKey="brittoOrders" 
              name="Mr.Britto Orders"
              fill={MANAGER_COLORS['Mr.Britto']}
              fillOpacity={0.8}
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              yAxisId="left"
              dataKey="udhayaOrders" 
              name="Mr.Udhaya Orders"
              fill={MANAGER_COLORS['Mr.Udhaya']}
              fillOpacity={0.8}
              radius={[4, 4, 0, 0]}
            />
            <Line 
              yAxisId="right"
              type="monotone" 
              dataKey="brittoValue" 
              name="Mr.Britto Value"
              stroke={MANAGER_COLORS['Mr.Britto']}
              strokeWidth={3}
              dot={{ fill: MANAGER_COLORS['Mr.Britto'], strokeWidth: 2, r: 4 }}
            />
            <Line 
              yAxisId="right"
              type="monotone" 
              dataKey="udhayaValue" 
              name="Mr.Udhaya Value"
              stroke={MANAGER_COLORS['Mr.Udhaya']}
              strokeWidth={3}
              dot={{ fill: MANAGER_COLORS['Mr.Udhaya'], strokeWidth: 2, r: 4 }}
            />
          </ComposedChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function TargetVsActualChart({ data, title, onClick, height = 350 }: ManagerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Target className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <BarChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 60 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="manager" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
            />
            <Tooltip content={<CustomTooltip />} />
            <ReferenceLine y={0} stroke="#000" />
            <Bar 
              dataKey="leadTimeVariance" 
              name="Lead Time Variance (Days)"
              fill="#f59e0b"
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              dataKey="qualityVariance" 
              name="Quality Variance (%)"
              fill="#10b981"
              radius={[4, 4, 0, 0]}
            />
            <Bar 
              dataKey="costVariance" 
              name="Cost Variance (%)"
              fill="#dc2626"
              radius={[4, 4, 0, 0]}
            />
          </BarChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ManagerRadarChart({ data, title, onClick, height = 400 }: ManagerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <TrendingUp className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <RadarChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <PolarGrid />
            <PolarAngleAxis dataKey="metric" fontSize={10} />
            <PolarRadiusAxis 
              angle={0} 
              domain={[0, 100]} 
              fontSize={10}
              tick={{ fontSize: 10 }}
            />
            <Radar 
              name="Mr.Britto" 
              dataKey="britto" 
              stroke={MANAGER_COLORS['Mr.Britto']}
              fill={MANAGER_COLORS['Mr.Britto']}
              fillOpacity={0.3}
              strokeWidth={2}
            />
            <Radar 
              name="Mr.Udhaya" 
              dataKey="udhaya" 
              stroke={MANAGER_COLORS['Mr.Udhaya']}
              fill={MANAGER_COLORS['Mr.Udhaya']}
              fillOpacity={0.3}
              strokeWidth={2}
            />
            <Tooltip content={<CustomTooltip />} />
          </RadarChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function WeeklyPerformanceChart({ data, title, onClick, height = 350 }: ManagerChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Clock className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <LineChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="week" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
            />
            <Tooltip content={<CustomTooltip />} />
            <Line 
              type="monotone" 
              dataKey="brittoScore" 
              name="Mr.Britto Performance"
              stroke={MANAGER_COLORS['Mr.Britto']}
              strokeWidth={3}
              dot={{ fill: MANAGER_COLORS['Mr.Britto'], strokeWidth: 2, r: 6 }}
              activeDot={{ r: 8, stroke: MANAGER_COLORS['Mr.Britto'] }}
            />
            <Line 
              type="monotone" 
              dataKey="udhayaScore" 
              name="Mr.Udhaya Performance"
              stroke={MANAGER_COLORS['Mr.Udhaya']}
              strokeWidth={3}
              dot={{ fill: MANAGER_COLORS['Mr.Udhaya'], strokeWidth: 2, r: 6 }}
              activeDot={{ r: 8, stroke: MANAGER_COLORS['Mr.Udhaya'] }}
            />
            <ReferenceLine y={80} stroke="#10b981" strokeDasharray="5 5" label="Target" />
          </LineChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ManagerKPICards({ managerData }: { managerData: any[] }) {
  return (
    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
      {managerData.map((manager) => (
        <Card key={manager.name} className="relative overflow-hidden">
          <div 
            className="absolute top-0 left-0 w-1 h-full"
            style={{ backgroundColor: MANAGER_COLORS[manager.name] }}
          ></div>
          <CardHeader className="pb-3">
            <CardTitle className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div 
                  className="w-10 h-10 rounded-full flex items-center justify-center text-white font-bold"
                  style={{ backgroundColor: MANAGER_COLORS[manager.name] }}
                >
                  {manager.name.split('.')[1].charAt(0)}
                </div>
                <div>
                  <h3 className="font-semibold">{manager.name}</h3>
                  <p className="text-sm text-muted-foreground">Purchase Manager</p>
                </div>
              </div>
              <Badge variant={manager.overallScore >= 85 ? 'default' : manager.overallScore >= 70 ? 'secondary' : 'destructive'}>
                {manager.overallScore}% Overall
              </Badge>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div className="text-center p-3 bg-blue-50 rounded-lg">
                <div className="text-xl font-bold text-blue-600">{manager.totalOrders}</div>
                <p className="text-xs text-blue-700">Total Orders</p>
                <div className="flex items-center justify-center mt-1">
                  {manager.ordersGrowth >= 0 ? (
                    <TrendingUp className="h-3 w-3 text-green-500 mr-1" />
                  ) : (
                    <TrendingDown className="h-3 w-3 text-red-500 mr-1" />
                  )}
                  <span className={`text-xs ${manager.ordersGrowth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {manager.ordersGrowth >= 0 ? '+' : ''}{manager.ordersGrowth}%
                  </span>
                </div>
              </div>
              
              <div className="text-center p-3 bg-green-50 rounded-lg">
                <div className="text-xl font-bold text-green-600">₹{manager.totalValue.toLocaleString()}</div>
                <p className="text-xs text-green-700">Total Value</p>
                <div className="flex items-center justify-center mt-1">
                  {manager.valueGrowth >= 0 ? (
                    <TrendingUp className="h-3 w-3 text-green-500 mr-1" />
                  ) : (
                    <TrendingDown className="h-3 w-3 text-red-500 mr-1" />
                  )}
                  <span className={`text-xs ${manager.valueGrowth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {manager.valueGrowth >= 0 ? '+' : ''}{manager.valueGrowth}%
                  </span>
                </div>
              </div>
            </div>

            <div className="space-y-2">
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Lead Time Performance</span>
                <div className="flex items-center gap-1">
                  <Badge variant={manager.leadTimeScore >= 90 ? 'default' : manager.leadTimeScore >= 70 ? 'secondary' : 'destructive'} className="text-xs">
                    {manager.leadTimeScore}%
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    (Target: {manager.avgTargetLeadTime}d, Actual: {manager.avgActualLeadTime}d)
                  </span>
                </div>
              </div>
              
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Quality Performance</span>
                <div className="flex items-center gap-1">
                  <Badge variant={manager.qualityScore >= 90 ? 'default' : manager.qualityScore >= 70 ? 'secondary' : 'destructive'} className="text-xs">
                    {manager.qualityScore}%
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    (Target: {manager.avgTargetQuality}%, Actual: {manager.avgActualQuality}%)
                  </span>
                </div>
              </div>
              
              <div className="flex justify-between items-center">
                <span className="text-sm text-muted-foreground">Cost Performance</span>
                <div className="flex items-center gap-1">
                  <Badge variant={manager.costScore >= 90 ? 'default' : manager.costScore >= 70 ? 'secondary' : 'destructive'} className="text-xs">
                    {manager.costScore}%
                  </Badge>
                  <span className="text-xs text-muted-foreground">
                    (Budget Variance: {manager.avgBudgetVariance >= 0 ? '+' : ''}{manager.avgBudgetVariance}%)
                  </span>
                </div>
              </div>
            </div>

            <div className="pt-2 border-t">
              <div className="flex justify-between items-center">
                <span className="text-sm font-medium">Monthly Target Achievement</span>
                <Badge variant={manager.targetAchievement >= 100 ? 'default' : manager.targetAchievement >= 80 ? 'secondary' : 'destructive'}>
                  {manager.targetAchievement}%
                </Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}