import React from 'react';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  LineChart,
  Line,
  PieChart,
  Pie,
  Cell,
  ComposedChart,
  Area,
  AreaChart,
  Legend,
  ScatterChart,
  Scatter,
  ReferenceLine
} from 'recharts';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { TrendingUp, TrendingDown, Info } from 'lucide-react';

// Custom tooltip component
const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length) {
    return (
      <div className="bg-white p-4 border rounded-lg shadow-lg">
        <p className="font-medium mb-2">{`${label}`}</p>
        {payload.map((entry: any, index: number) => (
          <div key={index} className="flex items-center gap-2">
            <div 
              className="w-3 h-3 rounded-full" 
              style={{ backgroundColor: entry.color }}
            ></div>
            <span className="text-sm">
              {entry.name}: {typeof entry.value === 'number' ? 
                entry.value.toLocaleString() : entry.value}
            </span>
          </div>
        ))}
      </div>
    );
  }
  return null;
};

// Professional color palette
const PROFESSIONAL_COLORS = {
  primary: '#2563eb',
  secondary: '#10b981',
  accent: '#f59e0b',
  danger: '#dc2626',
  warning: '#f59e0b',
  info: '#3b82f6',
  success: '#059669',
  muted: '#6b7280'
};

const CHART_COLORS = [
  '#2563eb', '#10b981', '#f59e0b', '#dc2626', 
  '#8b5cf6', '#06b6d4', '#84cc16', '#f97316'
];

interface ChartProps {
  data: any[];
  title: string;
  onClick?: (data: any) => void;
  height?: number;
}

export function ProfessionalBarChart({ data, title, onClick, height = 300 }: ChartProps) {
  const handleBarClick = (data: any) => {
    if (onClick) onClick(data);
  };

  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <Info className="h-4 w-4 text-muted-foreground" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <BarChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 60 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="name" 
              angle={-45}
              textAnchor="end"
              height={80}
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <Tooltip content={<CustomTooltip />} />
            <Legend />
            <Bar 
              dataKey="value" 
              fill={PROFESSIONAL_COLORS.primary}
              radius={[4, 4, 0, 0]}
              onClick={handleBarClick}
            />
          </BarChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ProfessionalLineChart({ data, title, onClick, height = 300 }: ChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <div className="flex items-center gap-2">
            <TrendingUp className="h-4 w-4 text-green-500" />
            <span className="text-sm text-green-600">+12.5%</span>
          </div>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <LineChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="month" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <Tooltip content={<CustomTooltip />} />
            <Legend />
            <Line 
              type="monotone" 
              dataKey="value" 
              stroke={PROFESSIONAL_COLORS.primary}
              strokeWidth={3}
              dot={{ fill: PROFESSIONAL_COLORS.primary, strokeWidth: 2, r: 6 }}
              activeDot={{ r: 8, stroke: PROFESSIONAL_COLORS.primary }}
            />
            <Line 
              type="monotone" 
              dataKey="orders" 
              stroke={PROFESSIONAL_COLORS.secondary}
              strokeWidth={3}
              dot={{ fill: PROFESSIONAL_COLORS.secondary, strokeWidth: 2, r: 6 }}
            />
          </LineChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ProfessionalPieChart({ data, title, onClick, height = 300 }: ChartProps) {
  const handlePieClick = (data: any) => {
    if (onClick) onClick(data);
  };

  const renderCustomLabel = ({ cx, cy, midAngle, innerRadius, outerRadius, percent, name }: any) => {
    const RADIAN = Math.PI / 180;
    const radius = innerRadius + (outerRadius - innerRadius) * 0.5;
    const x = cx + radius * Math.cos(-midAngle * RADIAN);
    const y = cy + radius * Math.sin(-midAngle * RADIAN);

    return (
      <text 
        x={x} 
        y={y} 
        fill="white" 
        textAnchor={x > cx ? 'start' : 'end'} 
        dominantBaseline="central"
        fontSize={12}
        fontWeight={600}
      >
        {`${(percent * 100).toFixed(0)}%`}
      </text>
    );
  };

  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle>{title}</CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <PieChart>
            <Pie
              data={data}
              cx="50%"
              cy="50%"
              labelLine={false}
              label={renderCustomLabel}
              outerRadius={100}
              fill="#8884d8"
              dataKey="value"
              onClick={handlePieClick}
            >
              {data.map((entry: any, index: number) => (
                <Cell key={`cell-${index}`} fill={CHART_COLORS[index % CHART_COLORS.length]} />
              ))}
            </Pie>
            <Tooltip content={<CustomTooltip />} />
            <Legend 
              verticalAlign="bottom" 
              height={36}
              formatter={(value) => <span style={{ fontSize: 12 }}>{value}</span>}
            />
          </PieChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ProfessionalComposedChart({ data, title, onClick, height = 400 }: ChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <div className="flex items-center gap-4">
            <div className="flex items-center gap-2">
              <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
              <span className="text-sm">Purchase Value</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-3 h-3 bg-green-500 rounded-full"></div>
              <span className="text-sm">Orders</span>
            </div>
          </div>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <ComposedChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="month" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              yAxisId="left"
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <YAxis 
              yAxisId="right" 
              orientation="right"
              fontSize={12}
              stroke="#64748b"
            />
            <Tooltip content={<CustomTooltip />} />
            <Bar 
              yAxisId="left"
              dataKey="value" 
              fill={PROFESSIONAL_COLORS.primary}
              fillOpacity={0.8}
              radius={[4, 4, 0, 0]}
            />
            <Line 
              yAxisId="right"
              type="monotone" 
              dataKey="orders" 
              stroke={PROFESSIONAL_COLORS.secondary}
              strokeWidth={3}
              dot={{ fill: PROFESSIONAL_COLORS.secondary, strokeWidth: 2, r: 4 }}
            />
          </ComposedChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function ProfessionalAreaChart({ data, title, onClick, height = 300 }: ChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle className="flex items-center justify-between">
          {title}
          <TrendingUp className="h-4 w-4 text-green-500" />
        </CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <AreaChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 20 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="month" 
              fontSize={12}
              stroke="#64748b"
            />
            <YAxis 
              fontSize={12}
              stroke="#64748b"
              tickFormatter={(value) => `₹${value.toLocaleString()}`}
            />
            <Tooltip content={<CustomTooltip />} />
            <Area 
              type="monotone" 
              dataKey="value" 
              stackId="1" 
              stroke={PROFESSIONAL_COLORS.primary}
              fill={PROFESSIONAL_COLORS.primary}
              fillOpacity={0.6}
            />
          </AreaChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}

export function SupplierPerformanceChart({ data, title, onClick, height = 400 }: ChartProps) {
  return (
    <Card className="cursor-pointer hover:shadow-lg transition-shadow">
      <CardHeader className="pb-2">
        <CardTitle>{title}</CardTitle>
      </CardHeader>
      <CardContent>
        <ResponsiveContainer width="100%" height={height}>
          <ScatterChart data={data} margin={{ top: 20, right: 30, left: 20, bottom: 80 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f1f5f9" />
            <XAxis 
              dataKey="avgLeadTime" 
              type="number"
              domain={[0, 'dataMax + 5']}
              fontSize={12}
              stroke="#64748b"
              label={{ value: 'Lead Time (Days)', position: 'insideBottom', offset: -5 }}
            />
            <YAxis 
              dataKey="qualityScore"
              type="number"
              domain={[70, 100]}
              fontSize={12}
              stroke="#64748b"
              label={{ value: 'Quality Score (%)', angle: -90, position: 'insideLeft' }}
            />
            <Tooltip 
              content={({ active, payload }) => {
                if (active && payload && payload.length) {
                  const data = payload[0].payload;
                  return (
                    <div className="bg-white p-4 border rounded-lg shadow-lg">
                      <p className="font-medium mb-2">{data.name}</p>
                      <p className="text-sm">Lead Time: {data.avgLeadTime} days</p>
                      <p className="text-sm">Quality Score: {data.qualityScore}%</p>
                      <p className="text-sm">Orders: {data.orders}</p>
                      <p className="text-sm">Value: ₹{data.totalValue.toLocaleString()}</p>
                    </div>
                  );
                }
                return null;
              }}
            />
            <Scatter 
              dataKey="qualityScore" 
              fill={PROFESSIONAL_COLORS.primary}
              onClick={onClick}
            />
            <ReferenceLine x={15} stroke="#ef4444" strokeDasharray="5 5" />
            <ReferenceLine y={90} stroke="#ef4444" strokeDasharray="5 5" />
          </ScatterChart>
        </ResponsiveContainer>
      </CardContent>
    </Card>
  );
}