// Charts management using Chart.js
class ChartManager {
    constructor() {
        this.charts = {};
    }

    // Create division breakdown pie chart
    createDivisionChart(canvasId, data) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return;

        // Destroy existing chart if it exists
        if (this.charts[canvasId]) {
            this.charts[canvasId].destroy();
        }

        this.charts[canvasId] = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: data.map(d => d.name),
                datasets: [{
                    data: data.map(d => d.value),
                    backgroundColor: [
                        '#0088FE', '#00C49F', '#FFBB28', 
                        '#FF8042', '#8884D8', '#82CA9D'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom'
                    },
                    title: {
                        display: true,
                        text: 'Division Breakdown'
                    }
                }
            }
        });
    }

    // Create supplier performance bar chart
    createSupplierChart(canvasId, data) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return;

        // Destroy existing chart if it exists
        if (this.charts[canvasId]) {
            this.charts[canvasId].destroy();
        }

        // Sort by total value and take top 10
        const topSuppliers = data
            .sort((a, b) => b.totalValue - a.totalValue)
            .slice(0, 10);

        this.charts[canvasId] = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: topSuppliers.map(s => s.name),
                datasets: [{
                    label: 'Total Value (₹)',
                    data: topSuppliers.map(s => s.totalValue),
                    backgroundColor: '#0088FE'
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Top Suppliers by Value'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });
    }

    // Create monthly trend line chart
    createTrendChart(canvasId, data) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return;

        // Destroy existing chart if it exists
        if (this.charts[canvasId]) {
            this.charts[canvasId].destroy();
        }

        // Generate monthly data (simplified)
        const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'];
        const values = months.map(() => Math.floor(Math.random() * 1000000) + 500000);

        this.charts[canvasId] = new Chart(ctx, {
            type: 'line',
            data: {
                labels: months,
                datasets: [{
                    label: 'Purchase Value (₹)',
                    data: values,
                    borderColor: '#00C49F',
                    backgroundColor: 'rgba(0, 196, 159, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Monthly Purchase Trend'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + (value / 1000) + 'K';
                            }
                        }
                    }
                }
            }
        });
    }

    // Update all charts with new data
    updateCharts() {
        const divisionData = window.purchaseData.getDivisionBreakdown();
        const supplierData = window.purchaseData.getSupplierPerformance();
        
        this.createDivisionChart('divisionChart', divisionData);
        this.createSupplierChart('supplierChart', supplierData);
        this.createTrendChart('trendChart', []);
    }
}

// Global instance
window.chartManager = new ChartManager();