// Dashboard component management
class DashboardManager {
    constructor() {
        this.currentTab = 'overview';
        this.init();
    }

    init() {
        this.renderDashboard();
        this.bindEvents();
        this.updateContent();
    }

    renderDashboard() {
        const dashboardHTML = `
            <div class="p-6 space-y-6">
                <!-- Header -->
                <div class="flex justify-between items-center">
                    <h1 class="text-3xl font-bold">Purchase Dashboard</h1>
                    <div class="flex gap-4">
                        <select id="divisionSelect" class="p-2 border rounded">
                            <option value="All">All Divisions</option>
                            <option value="Garments">Garments</option>
                            <option value="Spinning">Spinning</option>
                            <option value="Socks">Socks</option>
                            <option value="Gloves">Gloves</option>
                            <option value="Woven">Woven</option>
                            <option value="Headwear">Headwear</option>
                        </select>
                        <select id="timeframeSelect" class="p-2 border rounded">
                            <option value="Monthly">Monthly</option>
                            <option value="Quarterly">Quarterly</option>
                            <option value="Yearly">Yearly</option>
                        </select>
                    </div>
                </div>

                <!-- KPI Cards -->
                <div id="kpiCards" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <!-- KPI cards will be inserted here -->
                </div>

                <!-- Tabs -->
                <div class="tabs">
                    <div class="tabs-list">
                        <button class="tabs-trigger active" data-tab="overview">Overview</button>
                        <button class="tabs-trigger" data-tab="suppliers">Suppliers</button>
                        <button class="tabs-trigger" data-tab="managers">Managers</button>
                        <button class="tabs-trigger" data-tab="customers">Customers</button>
                        <button class="tabs-trigger" data-tab="advanced">Advanced</button>
                        <button class="tabs-trigger" data-tab="reports">Reports</button>
                    </div>

                    <!-- Tab Contents -->
                    <div id="overviewTab" class="tabs-content">
                        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                            <div class="card p-6">
                                <h3 class="mb-4">Division Breakdown</h3>
                                <canvas id="divisionChart" width="400" height="200"></canvas>
                            </div>
                            <div class="card p-6">
                                <h3 class="mb-4">Top Suppliers</h3>
                                <canvas id="supplierChart" width="400" height="200"></canvas>
                            </div>
                            <div class="card p-6">
                                <h3 class="mb-4">Monthly Trend</h3>
                                <canvas id="trendChart" width="400" height="200"></canvas>
                            </div>
                            <div class="card p-6">
                                <h3 class="mb-4">Purchase Summary</h3>
                                <div id="purchaseSummary">
                                    <!-- Summary content -->
                                </div>
                            </div>
                        </div>
                    </div>

                    <div id="suppliersTab" class="tabs-content hidden">
                        <div class="card p-6">
                            <h3 class="mb-4">Supplier Performance</h3>
                            <div class="overflow-x-auto">
                                <table class="table">
                                    <thead>
                                        <tr>
                                            <th>Supplier</th>
                                            <th>Orders</th>
                                            <th>Total Value</th>
                                            <th>Avg Order Value</th>
                                        </tr>
                                    </thead>
                                    <tbody id="supplierTableBody">
                                        <!-- Supplier data will be inserted here -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <div id="managersTab" class="tabs-content hidden">
                        <div class="card p-6">
                            <h3 class="mb-4">Manager Performance</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div class="card p-4">
                                    <h4>Mr. Britto</h4>
                                    <p>Performance metrics and charts</p>
                                </div>
                                <div class="card p-4">
                                    <h4>Mr. Udhaya</h4>
                                    <p>Performance metrics and charts</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div id="customersTab" class="tabs-content hidden">
                        <div class="card p-6">
                            <h3 class="mb-4">Customer Analysis</h3>
                            <p>Customer performance data and analytics</p>
                        </div>
                    </div>

                    <div id="advancedTab" class="tabs-content hidden">
                        <div class="space-y-6">
                            <!-- Advanced breakdown tabs -->
                            <div class="tabs">
                                <div class="tabs-list">
                                    <button class="tabs-trigger active" data-subtab="advance">Advance Details</button>
                                    <button class="tabs-trigger" data-subtab="grn">GRN to Bill</button>
                                    <button class="tabs-trigger" data-subtab="delayed">Delayed Bills</button>
                                    <button class="tabs-trigger" data-subtab="savings">Cost Savings</button>
                                </div>

                                <div id="advanceSubtab" class="tabs-content">
                                    <div class="card p-6">
                                        <h3 class="mb-4">Advance Payment Details</h3>
                                        <div id="advanceContent">
                                            <!-- Advance details content -->
                                        </div>
                                    </div>
                                </div>

                                <div id="grnSubtab" class="tabs-content hidden">
                                    <div class="card p-6">
                                        <h3 class="mb-4">GRN to Bill Tracking</h3>
                                        <p>GRN tracking content</p>
                                    </div>
                                </div>

                                <div id="delayedSubtab" class="tabs-content hidden">
                                    <div class="card p-6">
                                        <h3 class="mb-4">Delayed Bills</h3>
                                        <p>Delayed bills content</p>
                                    </div>
                                </div>

                                <div id="savingsSubtab" class="tabs-content hidden">
                                    <div class="card p-6">
                                        <h3 class="mb-4">Cost Savings Analysis</h3>
                                        <p>Cost savings content</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div id="reportsTab" class="tabs-content hidden">
                        <div class="card p-6">
                            <h3 class="mb-4">Reports & Analytics</h3>
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                <button class="btn btn-outline p-4">
                                    <i data-lucide="download"></i>
                                    Export Overview
                                </button>
                                <button class="btn btn-outline p-4">
                                    <i data-lucide="file-text"></i>
                                    Detailed Report
                                </button>
                                <button class="btn btn-outline p-4">
                                    <i data-lucide="bar-chart-3"></i>
                                    Analytics Export
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.getElementById('purchase-dashboard').innerHTML = dashboardHTML;
    }

    bindEvents() {
        // Division filter
        document.getElementById('divisionSelect').addEventListener('change', (e) => {
            window.purchaseData.setDivision(e.target.value);
            this.updateContent();
        });

        // Tab switching
        document.querySelectorAll('.tabs-trigger').forEach(trigger => {
            trigger.addEventListener('click', (e) => {
                const tab = e.target.getAttribute('data-tab');
                const subtab = e.target.getAttribute('data-subtab');
                
                if (tab) {
                    this.switchTab(tab);
                } else if (subtab) {
                    this.switchSubtab(subtab);
                }
            });
        });
    }

    switchTab(tabName) {
        // Remove active class from all triggers
        document.querySelectorAll('.tabs-trigger').forEach(trigger => {
            if (trigger.getAttribute('data-tab')) {
                trigger.classList.remove('active');
            }
        });

        // Hide all tab contents
        document.querySelectorAll('.tabs-content').forEach(content => {
            if (content.id.endsWith('Tab')) {
                content.classList.add('hidden');
            }
        });

        // Show selected tab
        document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
        document.getElementById(`${tabName}Tab`).classList.remove('hidden');

        this.currentTab = tabName;
    }

    switchSubtab(subtabName) {
        // Remove active class from all subtab triggers
        document.querySelectorAll('[data-subtab]').forEach(trigger => {
            trigger.classList.remove('active');
        });

        // Hide all subtab contents
        document.querySelectorAll('[id$="Subtab"]').forEach(content => {
            content.classList.add('hidden');
        });

        // Show selected subtab
        document.querySelector(`[data-subtab="${subtabName}"]`).classList.add('active');
        document.getElementById(`${subtabName}Subtab`).classList.remove('hidden');
    }

    updateContent() {
        this.updateKPICards();
        this.updateSupplierTable();
        this.updateAdvanceDetails();
        window.chartManager.updateCharts();
        
        // Initialize Lucide icons
        if (window.lucide) {
            window.lucide.createIcons();
        }
    }

    updateKPICards() {
        const kpis = window.purchaseData.calculateKPIs();
        
        const kpiHTML = `
            <div class="card p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600">Total Orders</p>
                        <p class="text-2xl font-bold">${kpis.totalOrders.toLocaleString()}</p>
                    </div>
                    <i data-lucide="package" class="h-8 w-8 text-blue-500"></i>
                </div>
            </div>
            <div class="card p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600">Total Value</p>
                        <p class="text-2xl font-bold">₹${kpis.totalValue.toLocaleString()}</p>
                    </div>
                    <i data-lucide="dollar-sign" class="h-8 w-8 text-green-500"></i>
                </div>
            </div>
            <div class="card p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600">Advance Orders</p>
                        <p class="text-2xl font-bold">${kpis.advancePaymentOrders}</p>
                    </div>
                    <i data-lucide="credit-card" class="h-8 w-8 text-purple-500"></i>
                </div>
            </div>
            <div class="card p-6">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-600">Avg Order Value</p>
                        <p class="text-2xl font-bold">₹${kpis.avgOrderValue.toLocaleString()}</p>
                    </div>
                    <i data-lucide="trending-up" class="h-8 w-8 text-orange-500"></i>
                </div>
            </div>
        `;

        document.getElementById('kpiCards').innerHTML = kpiHTML;
    }

    updateSupplierTable() {
        const suppliers = window.purchaseData.getSupplierPerformance();
        
        const tableHTML = suppliers.map(supplier => `
            <tr>
                <td>${supplier.name}</td>
                <td>${supplier.orders}</td>
                <td>₹${supplier.totalValue.toLocaleString()}</td>
                <td>₹${(supplier.totalValue / supplier.orders).toLocaleString()}</td>
            </tr>
        `).join('');

        document.getElementById('supplierTableBody').innerHTML = tableHTML;
    }

    updateAdvanceDetails() {
        const advanceOrders = window.purchaseData.filteredData.filter(item => item.supplierAdvancePaid);
        const totalAdvanceAmount = advanceOrders.reduce((sum, item) => sum + item.advanceAmount, 0);
        
        const advanceHTML = `
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                <div class="card p-4">
                    <p class="text-sm text-gray-600">Total Advance Orders</p>
                    <p class="text-xl font-bold">${advanceOrders.length}</p>
                </div>
                <div class="card p-4">
                    <p class="text-sm text-gray-600">Total Advance Amount</p>
                    <p class="text-xl font-bold">₹${totalAdvanceAmount.toLocaleString()}</p>
                </div>
                <div class="card p-4">
                    <p class="text-sm text-gray-600">Avg Advance %</p>
                    <p class="text-xl font-bold">25.5%</p>
                </div>
            </div>
            <div class="overflow-x-auto">
                <table class="table">
                    <thead>
                        <tr>
                            <th>PO No</th>
                            <th>Supplier</th>
                            <th>PO Value</th>
                            <th>Advance Amount</th>
                            <th>Advance %</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${advanceOrders.slice(0, 10).map(order => `
                            <tr>
                                <td>${order.id}</td>
                                <td>${order.supplier}</td>
                                <td>₹${order.poVal.toLocaleString()}</td>
                                <td>₹${order.advanceAmount.toLocaleString()}</td>
                                <td>${((order.advanceAmount / order.poVal) * 100).toFixed(1)}%</td>
                                <td><span class="badge badge-default">Active</span></td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;

        document.getElementById('advanceContent').innerHTML = advanceHTML;
    }
}

// Global instance
window.dashboardManager = new DashboardManager();